#
# Copyright (C) 2016-2019, 2022 The slide-switch authors
# https://github.com/jefferyto/openwrt-slide-switch
#
# slide-switch is free software, licensed under the GNU General Public License v2.
#

package=slide-switch
version=1.0.0

data_dir=/usr/share/slide-switch
run_dir=/var/run/slide-switch
lock_dir=/var/lock/slide-switch
conf_dir=/etc/slide-switch
initd_dir=/etc/init.d

board_name=/tmp/sysinfo/board_name
gpio_table=/sys/kernel/debug/gpio
device_tree_keys=/proc/device-tree/keys
hotplug_call=/sbin/hotplug-call
rc_button=/etc/rc.button

custom_switch_data_file=$conf_dir/switch-data.json
switch_data_file=$data_dir/switch-data.json
switches_cache=$run_dir/switches
state_dir=$run_dir/state
update_initd_script=$initd_dir/zzz-slide-switch-update

. /usr/share/libubox/jshn.sh

me=${0##*/}

ex_usage=64
ex_dataerr=65
ex_noinput=66
ex_unavailable=69
ex_oserr=71
ex_osfile=72
ex_noperm=77

log_stderr=
log_syslog=
verbosity=
is_root_result=
locks=


log_level_to_num() {
	local _var="$1"
	local _level="$2"
	local _num

	case $_level in
		emerg|panic)  _num=0 ;;
		alert)        _num=1 ;;
		crit)         _num=2 ;;
		err|error)    _num=3 ;;
		warning|warn) _num=4 ;;
		notice)       _num=5 ;;
		info)         _num=6 ;;
		debug)        _num=7 ;;

		*[!0-9]*)     return 1 ;;

		*)            _num=$_level ;;
	esac

	_set_var "$_var" "$_num"
}

log() {
	local level="$1"
	local message="$2"
	local level_num

	log_level_to_num level_num "$level" &&
	[ "$level_num" -le "$verbosity" ] || return 0

	[ -n "$log_stderr" ] && echo "[$level] $message" >&2

	[ -n "$log_syslog" ] && logger -t "${me}[$$]" -p "user.$level" "$message"

	return 0
}

error() {
	log err "$@"
}

warning() {
	log warning "$@"
}

notice() {
	log notice "$@"
}

info() {
	log info "$@"
}

debug() {
	log debug "$@"
}


is_root() {
	local uid

	[ -n "$is_root_result" ] || {
		uid=$(id -u) && [ "$uid" -eq 0 ]
		is_root_result=$?
	}

	return $is_root_result
}

require_root() {
	is_root || {
		error "require_root: user must be root"
		return $ex_noperm
	}
}

get_board() {
	local _var="$1"
	local _board

	_board=$(cat "$board_name") && _set_var "$_var" "$_board"
}

get_epoch() {
	local _var="$1"
	local _epoch

	_epoch=$(date +%s) || {
		error "get_epoch: failed to get date"
		return $ex_oserr
	}

	_set_var "$_var" "$_epoch"
}

list_contains() {
	local list=" $1 "
	local item=" $2 "

	[ -z "${list##*$item*}" ]
}

is_safe_string() {
	local str="$1"

	[ -n "${str##*[!0-9A-Za-z_-]*}" ]
}


check_prerequisites() {
	local file

	for file in "$board_name" "$gpio_table"; do
		[ -f "$file" ] && [ -r "$file" ] || {
			error "check_prerequisites: cannot read $file"
			return $ex_osfile
		}
	done

	[ -f "$switch_data_file" ] && [ -r "$switch_data_file" ] || {
		error "check_prerequisites: cannot read $switch_data_file"
		return $ex_noinput
	}
}


get_gpio_value() {
	local __var="$1"
	local __label="$2"
	local __allow_no_matches="$3"
	local __pattern
	local __match
	local __result

	__pattern=$(printf '|%-20.20s) in  \(hi\|lo\)' "$__label") || {
		error "get_gpio_value: failed to get search pattern for label \"$__label\""
		return $ex_oserr
	}

	__match="$(grep -m 1 -o -e "$__pattern" "$gpio_table" 2>/dev/null)"
	__result=$?

	[ "$__result" -eq 0 ] || {
		[ "$__result" -eq 1 ] && {
			[ -z "$__allow_no_matches" ] && error "get_gpio_value: cannot find label \"$__label\" in $gpio_table"
			return $ex_dataerr
		}

		error "get_gpio_value: failed to search $gpio_table for label \"$__label\""
		return $ex_oserr
	}

	_set_var "$__var" "${__match##*) in  }"
}

get_position_code() {
	local _var="$1"
	local _switch="$2"
	local _position_name="$3"

	_set_var "$_var" "${_switch}-$_position_name"
}


is_keys_available() {
	[ -d "$device_tree_keys" ]
}

is_key() {
	local key="$1"

	[ -d "$device_tree_keys/$key" ]
}

get_key_label_path() {
	local __var="$1"
	local __key="$2"

	_set_var "$__var" "$device_tree_keys/$__key/label"
}

get_key_code_path() {
	local __var="$1"
	local __key="$2"

	_set_var "$__var" "$device_tree_keys/$__key/linux,code"
}

has_key_info() {
	local key="$1"
	local label_path
	local code_path

	get_key_label_path label_path "$key"
	get_key_code_path code_path "$key"

	[ -f "$label_path" ] && [ -r "$label_path" ] &&
	[ -f "$code_path" ] && [ -r "$code_path" ]
}

get_key_label() {
	local _var="$1"
	local _key="$2"
	local _label_path
	local _label

	get_key_label_path _label_path "$_key"

	_label=$(cat "$_label_path") || {
		error "get_key_label: failed to read $_label_path"
		return $ex_oserr
	}

	_set_var "$_var" "$_label"
}

get_key_code() {
	local _var="$1"
	local _key="$2"
	local _code_path
	local _hex
	local _code

	get_key_code_path _code_path "$_key"

	_hex=$(hexdump -n 4 -v -e '4/1 "%02x"' "$_code_path") || {
		error "get_key_code: failed to read $_code_path"
		return $ex_oserr
	}

	case $_hex in
		00000100) _code=BTN_0 ;;
		00000101) _code=BTN_1 ;;
		00000102) _code=BTN_2 ;;
		00000103) _code=BTN_3 ;;
		00000104) _code=BTN_4 ;;
		00000105) _code=BTN_5 ;;
		00000106) _code=BTN_6 ;;
		00000107) _code=BTN_7 ;;
		00000108) _code=BTN_8 ;;
		00000109) _code=BTN_9 ;;
	esac

	[ -n "$_code" ] || {
		error "get_key_code: unknown value \"$_hex\" in $_code_path"
		return $ex_osfile
	}

	_set_var "$_var" "$_code"
}


is_valid_switch_data_file() {
	local data_file="$1"

	jsonfilter -q -i "$data_file" -t "@" > /dev/null

	[ "$?" -ne 126 ]
}

get_switch_data_file_board_value() {
	local __option="$1"
	local __var="$2"
	local __data_file="$3"
	local __board="$4"
	local __value

	__value=$(jsonfilter -i "$__data_file" "$__option" "@[\"$__board\"]") || __value=

	_set_var "$__var" "$__value"
}

get_switch_data_file_board_data_type() {
	get_switch_data_file_board_value -t "$@"
}

get_switch_data_file_board_data() {
	get_switch_data_file_board_value -e "$@"
}


populate_switches_details() {
	local switches
	local switch
	local keys
	local key
	local label
	local code

	json_set_namespace switches
	json_select

	json_get_keys switches

	for switch in $switches; do
		json_select
		json_select "$switch"

		json_is_a labels '' && json_is_a codes '' || {
			debug "populate_switches_details: switch \"$switch\" already has labels and/or codes properties"
			continue
		}

		json_is_a keys array || {
			debug "populate_switches_details: switch \"$switch\" does not have a keys array"
			continue
		}

		json_get_values keys keys

		[ -n "$keys" ] || {
			debug "populate_switches_details: no keys defined for switch \"$switch\""
			continue
		}

		json_add_array labels
		json_close_array

		json_add_array codes
		json_close_array

		for key in $keys; do
			json_select
			json_select "$switch"

			is_key "$key" || {
				error "populate_switches_details: key \"$key\" of switch \"$switch\" is not a valid gpio key"
				return $ex_dataerr
			}

			has_key_info "$key" || {
				error "populate_switches_details: key \"$key\" of switch \"$switch\" is missing label and/or linux,code"
				return $ex_dataerr
			}

			get_key_label label "$key" &&
			get_key_code code "$key" || return

			debug "populate_switches_details: found label \"$label\" and code \"$code\" for key \"$key\" of switch \"$switch\""

			json_select labels

			json_add_string '' "$label"

			json_select ..
			json_select codes

			json_add_string '' "$code"
		done
	done
}


check_json_type() {
	local key="$1"
	local type="$2"
	local key_name="${3:-$key}"
	local article="a"

	case $type in
		[aeiou]*) article=an ;;
	esac

	json_is_a "$key" "$type" || {
		error "check_json_type: \"$key_name\" is not $article $type"
		return $ex_dataerr
	}
}

check_nonempty_string() {
	local str="$1"
	local str_name="$2"

	[ -n "$str" ] || {
		error "check_nonempty_string: $str_name cannot be an empty string"
		return $ex_dataerr
	}
}

check_safe_string() {
	local str="$1"
	local str_name="$2"

	is_safe_string "$str" || {
		error "check_safe_string: $str_name \"$str\" must contain letters, digits, underscores and/or hyphens only"
		return $ex_dataerr
	}
}

check_input_gpio_label() {
	local label="$1"
	local value

	get_gpio_value value "$label" allow_no_matches && [ -n "$value" ] || {
		error "check_input_gpio_label: \"$label\" is not the label of a valid input gpio"
		return $ex_dataerr
	}
}

check_position_string() {
	local position="$1"
	local str="$position"
	local prev

	while [ "x$str" != "x$prev" ]; do
		prev=$str

		str=${str%_hi}
		str=${str%_lo}
	done

	[ "x$str" = xhi ] || [ "x$str" = xlo ] || {
		warning "check_position_string: position \"$position\" is not in the format \"(hi|lo)(_(hi|lo))*\""
		return $ex_dataerr
	}
}

check_position_code() {
	local switch="$1"
	local position_name="$2"
	local codes="$3"
	local code

	get_position_code code "$switch" "$position_name"

	! list_contains "$codes" "$code" || {
		error "check_position_code: switch \"$switch\" cannot have both a position name \"$position_name\" and a code \"$code\""
		return $ex_dataerr
	}
}

validate_switches_initial() {
	local switches
	local switch

	json_set_namespace switches
	json_select

	json_get_keys switches

	[ -n "$switches" ] || warning "validate_switches_initial: no switches defined"

	for switch in $switches; do
		debug "validate_switches_initial: checking switch \"$switch\""

		check_nonempty_string "$switch" "switch name" &&
		check_safe_string "$switch" "switch name" &&
		check_json_type "$switch" object || return
	done
}

validate_switches_details() {
	local switches
	local switch
	local labels_indices
	local label
	local codes
	local positions
	local position
	local position_name
	local i

	json_set_namespace switches
	json_select

	json_get_keys switches

	for switch in $switches; do
		json_select
		json_select "$switch"

		debug "validate_switches_details: checking data for switch \"$switch\""

		check_json_type labels array "$switch.labels" &&
		check_json_type codes array "$switch.codes" &&
		check_json_type positions object "$switch.positions" || return

		json_get_keys labels_indices labels
		json_get_values codes codes
		json_get_keys positions positions

		[ -n "$labels_indices" ] || warning "validate_switches_details: no labels defined for switch \"$switch\""
		[ -n "$codes" ] || warning "validate_switches_details: no codes defined for switch \"$switch\""
		[ -n "$positions" ] || warning "validate_switches_details: no positions defined for switch \"$switch\""

		json_select labels

		for i in $labels_indices; do
			json_get_var label "$i"
			check_input_gpio_label "$label" || return
		done

		json_select ..
		json_select positions

		for position in $positions; do
			check_position_string "$position" || continue

			check_json_type "$position" string "$switch.positions.$position" || return

			json_get_var position_name "$position"

			check_nonempty_string "$position_name" "position name" &&
			check_safe_string "$position_name" "position name" &&
			check_position_code "$switch" "$position_name" "$codes" || return
		done
	done
}


load_switch_data_from_cache() {
	[ -f "$switches_cache" ] && [ -r "$switches_cache" ] || {
		debug "load_switch_data_from_cache: cannot read $switches_cache"
		return 1
	}

	info "load_switch_data_from_cache: loading switch data from $switches_cache"

	json_set_namespace switches
	json_load_file "$switches_cache"
}

save_switch_data_to_cache() {
	is_root || {
		debug "save_switch_data_to_cache: user must be root"
		return
	}

	debug "save_switch_data_to_cache: caching switch data to $switches_cache"

	mkdir -p "${switches_cache%/*}"

	json_set_namespace switches
	json_select
	json_dump > "$switches_cache"
}

load_switch_data_from_file() {
	local _var="$1"
	local _data_file="$2"
	local _board="$3"
	local _no_alt_board_name="$4"
	local _type
	local _alt_board
	local _data

	debug "load_switch_data_from_file: attempting to load switch data for board \"$_board\" from $_data_file"

	is_valid_switch_data_file "$_data_file" || {
		error "load_switch_data_from_file: failed to parse $_data_file"
		return $ex_dataerr
	}

	get_switch_data_file_board_data_type _type "$_data_file" "$_board"

	[ -z "$_no_alt_board_name" ] && [ "x$_type" = xstring ] && {
		debug "load_switch_data_from_file: board \"$_board\" shares switch data with another model, loading alternate board name"

		get_switch_data_file_board_data _alt_board "$_data_file" "$_board"

		[ -n "$_alt_board" ] || {
			error "load_switch_data_from_file: missing alternate board name for board \"$_board\" in $_data_file"
			return $ex_dataerr
		}

		info "load_switch_data_from_file: using alternate board name \"$_alt_board\" for board \"$_board\""

		_board=$_alt_board
		get_switch_data_file_board_data_type _type "$_data_file" "$_board"
	}

	[ -n "$_type" ] || {
		debug "load_switch_data_from_file: board \"$_board\" not found in $_data_file"
		_set_var "$_var" "$_board"
		return $ex_unavailable
	}

	[ "x$_type" = xobject ] || {
		error "load_switch_data_from_file: \"$_board\" is not an object in $_data_file"
		return $ex_dataerr
	}

	info "load_switch_data_from_file: loading switch data for board \"$_board\" from $_data_file"

	get_switch_data_file_board_data _data "$_data_file" "$_board"

	json_set_namespace switches
	json_load "$_data"
}

load_switch_data() {
	local no_load_from_cache="$1"
	local no_save_to_cache="$2"
	local board
	local final_board
	local result
	local no_alt_board_name

	[ -z "$no_load_from_cache" ] && load_switch_data_from_cache && return

	get_board board && [ -n "$board" ] || {
		error "load_switch_data: missing board name"
		return $ex_dataerr
	}

	[ -f "$custom_switch_data_file" ] && [ -r "$custom_switch_data_file" ] && {
		info "load_switch_data: attempting to load switch data for board \"$board\" from $custom_switch_data_file"

		load_switch_data_from_file final_board "$custom_switch_data_file" "$board"
		result=$?

		[ "$result" -ne 0 ] && {
			[ "$result" -ne "$ex_unavailable" ] && return $result

			notice "load_switch_data: board \"$final_board\" not found in $custom_switch_data_file"

			[ "x$final_board" != "x$board" ] && {
				board=$final_board
				no_alt_board_name=1
			}

			result=
		}
	}

	[ -z "$result" ] && {
		debug "load_switch_data: attempting to load switch data for board \"$board\" from $switch_data_file"

		load_switch_data_from_file final_board "$switch_data_file" "$board" "$no_alt_board_name"
		result=$?

		[ "$result" -ne 0 ] && {
			[ "$result" -eq "$ex_unavailable" ] && warning "load_switch_data: board \"$final_board\" not found in $switch_data_file"

			return $result
		}
	}

	validate_switches_initial || return

	is_keys_available && {
		populate_switches_details || return
	}

	validate_switches_details || return

	[ -z "$no_save_to_cache" ] && save_switch_data_to_cache

	return 0
}


get_switches() {
	local _var="$1"

	json_set_namespace switches
	json_select

	json_get_keys "$_var"
}

get_switch_for_button() {
	local _var="$1"
	local _button="$2"
	local _switches
	local _switch
	local _codes

	json_set_namespace switches
	json_select

	json_get_keys _switches

	for _switch in $_switches; do
		json_select
		json_select "$_switch"

		json_get_values _codes codes

		list_contains "$_codes" "$_button" && {
			debug "get_switch_for_button: found switch \"$_switch\" for button \"$_button\""
			_set_var "$_var" "$_switch"
			return
		}
	done

	debug "get_switch_for_button: no switches found for button \"$_button\""
	return 1
}

get_current_position() {
	local _var="$1"
	local _switch="$2"
	local _labels_indices
	local _label
	local _value
	local _position
	local _i

	json_set_namespace switches
	json_select
	json_select "$_switch"
	json_select labels

	json_get_keys _labels_indices

	for _i in $_labels_indices; do
		json_get_var _label "$_i"
		get_gpio_value _value "$_label" || return
		_position="${_position}_$_value"
	done

	_set_var "$_var" "${_position#_}"
}

get_position_name() {
	local _var="$1"
	local _switch="$2"
	local _position="$3"

	json_set_namespace switches
	json_select
	json_select "$_switch"
	json_select positions

	json_get_var "$_var" "$_position"
}


get_state_file() {
	local _var="$1"
	local _switch="$2"

	_set_var "$_var" "$state_dir/$_switch"
}

init_state() {
	local switch="$1"
	local now="$2"
	local positions
	local position

	debug "init_state: initializing state for switch \"$switch\""

	json_set_namespace switches
	json_select
	json_select "$switch"

	json_get_keys positions positions

	get_current_position position "$switch" || return

	debug "init_state: setting \"$position\" as initial position for switch \"$switch\""

	json_set_namespace state
	json_init

	json_add_string position "$position"

	json_add_object seen
	for position in $positions; do
		json_add_int "$position" "$now"
	done
	json_close_object
}

has_state() {
	local switch="$1"
	local state_file

	get_state_file state_file "$switch"

	[ -f "$state_file" ] && [ -r "$state_file" ]
}

load_state() {
	local switch="$1"
	local state_file

	has_state "$switch" || {
		debug "load_state: state for switch \"$switch\" has not been initialized"
		return 1
	}

	get_state_file state_file "$switch"

	debug "load_state: loading state for switch \"$switch\" from $state_file"

	json_set_namespace state
	json_load_file "$state_file"
}

save_state() {
	local switch="$1"
	local state_file

	get_state_file state_file "$switch"

	debug "save_state: saving state for switch \"$switch\" to $state_file"

	mkdir -p "${state_file%/*}"

	json_set_namespace state
	json_select
	json_dump > "$state_file"
}


get_state_position() {
	local _var="$1"

	json_set_namespace state
	json_select

	json_get_var "$_var" position
}

set_state_position() {
	local position="$1"

	json_set_namespace state
	json_select

	json_add_string position "$position"
}

get_state_seen() {
	local _var="$1"
	local _position="$2"

	json_set_namespace state
	json_select
	json_select seen

	json_get_var "$_var" "$_position"
}

set_state_seen() {
	local position="$1"
	local seen="$2"

	json_set_namespace state
	json_select
	json_select seen

	json_add_int "$position" "$seen"
}


get_lock_dir() {
	local _var="$1"
	local _switch="$2"

	_set_var "$_var" "$lock_dir/$_switch"
}

cleanup_locks() {
	local switch
	local switch_lock_dir

	for switch in $locks; do
		get_lock_dir switch_lock_dir "$switch"

		[ -d "$switch_lock_dir" ] || {
			warning "cleanup_locks: cannot find lock for switch \"$switch\" ($switch_lock_dir)"
			continue
		}

		debug "cleanup_locks: removing lock for switch \"$switch\" ($switch_lock_dir)"
		rmdir --ignore-fail-on-non-empty "$switch_lock_dir"
	done

	locks=
}

get_lock() {
	local switch="$1"
	local switch_lock_dir

	get_lock_dir switch_lock_dir "$switch"

	mkdir -p "${switch_lock_dir%/*}"

	mkdir "$switch_lock_dir" 2>/dev/null || {
		debug "get_lock: failed to get lock for switch \"$switch\" ($switch_lock_dir)"
		return 1
	}

	locks="$locks $switch"

	trap 'cleanup_locks' EXIT
	trap 'cleanup_locks; trap - INT; kill -INT $$' INT
	trap 'exit 129' HUP
	trap 'exit 131' QUIT
	trap 'exit 143' TERM

	debug "get_lock: got lock for $switch ($switch_lock_dir)"
}

release_locks() {
	debug "release_locks: releasing locks"

	cleanup_locks

	trap - EXIT INT HUP QUIT TERM
}


trigger_button_event() {
	local switch="$1"
	local position="$2"
	local action="$3"
	local now="$4"
	local position_name
	local before
	local button
	local seen

	get_position_name position_name "$switch" "$position"

	[ -n "$position_name" ] || {
		warning "trigger_button_event: could not get position name for switch \"$switch\" and position \"$position\""
		return 1
	}

	get_state_seen before "$position"
	set_state_seen "$position" "$now"

	get_position_code button "$switch" "$position_name"
	seen=$((now - before))

	notice "trigger_button_event: triggering button event with BUTTON=\"$button\" ACTION=\"$action\" SEEN=\"$seen\""

	(
		export BUTTON="$button"
		export ACTION="$action"
		export SEEN="$seen"

		debug "trigger_button_event: attempting to call $hotplug_call"
		[ -x "$hotplug_call" ] && "$hotplug_call" button

		debug "trigger_button_event: attempting to call $rc_button/$button"
		[ -x "$rc_button/$button" ] && "$rc_button/$button"
	)

	return 0
}


show_usage() {
	cat <<- EOF
	Usage: $me [OPTION]... COMMAND [ARGUMENT]...
	Print slide switch information, or trigger button presses and releases

	Options:
	  --log=WHERE                  output logging messages to WHERE:
	                                 'stderr', 'syslog', 'all', 'none'
	                                 (default: 'stderr')
	  --verbosity=LEVEL            output logging messages at LEVEL or higher;
	                                 LEVEL is a syslog level keyword or value
	                                 (default: 'notice')
	  --help                       display this help and exit
	  --version                    output version information and exit

	Commands:
	  init                         initialize slide switch state information
	  update [ARGUMENT]...         trigger button presses and releases for the
	                                 listed switch names and/or buttons, or for
	                                 all slide switches if no names given;
	                                 include '-f' or '--force' to trigger button
	                                 presses even when no switch position changes
	                                 have occurred
	  switch                       print list of slide switches for this device
	  position NAME                print current position of slide switch NAME
	  clean                        remove slide switch state information
	  boot STATUS                  whether to trigger button presses on boot;
	                                 STATUS is 'enable' or 'disable'
	                                 (default: 'enable')

	$package home page: <https://github.com/jefferyto/openwrt-slide-switch>
	EOF
}


set_log() {
	local value="$1"

	case $value in
		none)   log_stderr= ; log_syslog=  ;;
		stderr) log_stderr=1; log_syslog=  ;;
		syslog) log_stderr= ; log_syslog=1 ;;
		all)    log_stderr=1; log_syslog=1 ;;

		*)      error "invalid log value \"$value\""; show_usage; return $ex_usage ;;
	esac
}

set_verbosity() {
	local value="$1"

	log_level_to_num verbosity "$value" || {
		error "invalid verbosity value \"$value\""
		show_usage
		return $ex_usage
	}
}


do_init() {
	local switches
	local switch
	local now

	check_prerequisites &&
	require_root &&
	load_switch_data no_load_from_cache || return

	get_switches switches
	get_epoch now || return

	for switch in $switches; do
		get_lock "$switch" || {
			warning "init: cannot get lock for switch \"$switch\""
			continue
		}

		debug "init: initializing switch \"$switch\""

		init_state "$switch" "$now" || return
		save_state "$switch"
	done

	release_locks
}

do_update() {
	local switches
	local switch
	local force
	local has_requested
	local requested
	local locked
	local now
	local current
	local previous
	local arg

	check_prerequisites &&
	require_root &&
	load_switch_data || return

	get_switches switches

	for arg in "$@"; do
		case $arg in
			-f|--force)
				debug "update: given --force option"
				force=1
				continue
				;;
		esac

		is_safe_string "$arg" || {
			error "update: \"$arg\" is not a valid switch name or button"
			show_usage
			return $ex_usage
		}

		has_requested=1
		switch=

		list_contains "$switches" "$arg" && {
			debug "update: \"$arg\" is a switch name"
			switch=$arg
		}

		[ -z "$switch" ] && {
			get_switch_for_button switch "$arg" &&
			[ -n "$switch" ] &&
			debug "update: \"$arg\" is a button for switch \"$switch\""
		}

		[ -n "$switch" ] || {
			debug "update: \"$arg\" is not a switch name or button"
			continue
		}

		! list_contains "$requested" "$switch" && requested="$requested $switch"
	done

	[ -z "$has_requested" ] && {
		notice "update: no switch names or buttons given, updating all switches"
		requested=$switches
	}

	for switch in $requested; do
		has_state "$switch" || {
			debug "update: state for switch \"$switch\" has not been initialized"
			continue
		}

		get_lock "$switch" || {
			debug "update: cannot get lock for switch \"$switch\""
			continue
		}

		locked="$locked $switch"
	done

	[ -n "$locked" ] || {
		debug "update: no switches to update"
		return
	}

	sleep 1

	get_epoch now || return

	for switch in $locked; do
		load_state "$switch" || {
			warning "update: cannot load state for switch \"$switch\""
			continue
		}

		get_state_position previous
		get_current_position current "$switch" || return

		debug "update: previous position for switch \"$switch\" is \"$previous\""
		debug "update: current position for switch \"$switch\" is \"$current\""

		[ -n "$force" ] && [ "x$current" = "x$previous" ] && {
			debug "update: forcing update for switch \"$switch\""
			previous=
		}

		[ "x$current" != "x$previous" ] || {
			debug "update: switch \"$switch\" position unchanged"
			continue
		}

		[ -n "$previous" ] && trigger_button_event "$switch" "$previous" released "$now"
		trigger_button_event "$switch" "$current" pressed "$now"

		set_state_position "$current"
		save_state "$switch"
	done

	release_locks
}

do_switch() {
	local switches
	local switch

	check_prerequisites &&
	load_switch_data '' no_save_to_cache || return

	get_switches switches

	for switch in $switches; do
		echo "$switch"
	done
}

do_position() {
	local switch="$1"
	local switches
	local position
	local position_name

	check_prerequisites &&
	load_switch_data '' no_save_to_cache || return

	get_switches switches

	[ -n "$switch" ] || {
		error "position: missing switch name"
		show_usage
		return $ex_usage
	}

	is_safe_string "$switch" || {
		error "position: \"$switch\" is not a valid switch name"
		show_usage
		return $ex_usage
	}

	list_contains "$switches" "$switch" || {
		error "position: cannot find switch \"$switch\""
		show_usage
		return $ex_usage
	}

	get_current_position position "$switch" || return
	get_position_name position_name "$switch" "$position"

	echo "$position_name"
}

do_clean() {
	local switches
	local switch
	local state_file
	local switch_lock_dir

	check_prerequisites &&
	require_root &&
	load_switch_data no_load_from_cache no_save_to_cache || return

	get_switches switches

	for switch in $switches; do
		get_state_file state_file "$switch"
		get_lock_dir switch_lock_dir "$switch"

		debug "clean: removing state for switch \"$switch\" ($state_file)"
		rm -f "$state_file"

		debug "clean: removing lock for switch \"$switch\" ($switch_lock_dir)"
		[ -d "$switch_lock_dir" ] && rmdir --ignore-fail-on-non-empty "$switch_lock_dir"
	done

	debug "clean: removing $switches_cache"
	rm -f "$switches_cache"

	debug "clean: removing $state_dir, $run_dir, $lock_dir"
	rmdir --ignore-fail-on-non-empty "$state_dir" "$run_dir" "$lock_dir" 2>/dev/null

	return 0
}

do_boot() {
	local action="$1"

	require_root || return

	[ -x "$update_initd_script" ] || {
		error "boot: init.d script $update_initd_script not found or not executable"
		return $ex_unavailable
	}

	[ "x$action" = xenable ] || [ "x$action" = xdisable ] || {
		error "boot: invalid argument \"$action\""
		show_usage
		return $ex_usage
	}

	debug "boot: calling $update_initd_script $action"
	"$update_initd_script" "$action"
}

do_version() {
	echo "$me $version"
}
