package File::ShareDir::Install; # git description: v0.12-6-g29a6ff7

use strict;
use warnings;

use Carp ();

use File::Spec;
use IO::Dir;

our $VERSION = '0.13';

our @DIRS;
our %ALREADY;

require Exporter;

our @ISA = qw( Exporter );
our @EXPORT = qw( install_share delete_share );
our @EXPORT_OK = qw( postamble install_share delete_share );
our $INCLUDE_DOTFILES = 0;
our $INCLUDE_DOTDIRS = 0;

sub install_share
{
    my $dir  = @_ ? pop : 'share';
    my $type = @_ ? shift : 'dist';
    unless ( defined $type and
            ( $type =~ /^(module|dist)$/ ) ) {
        Carp::confess "Illegal or invalid share dir type '$type'";
    }

    if( $type eq 'dist' and @_ ) {
        Carp::confess "Too many parameters to install_share";
    }

    my $def = _mk_def( $type );
    _add_module( $def, $_[0] );

    _add_dir( $def, $dir );
}


sub delete_share
{
    my $dir  = @_ ? pop : '';
    my $type = @_ ? shift : 'dist';
    unless ( defined $type and
            ( $type =~ /^(module|dist)$/ ) ) {
        Carp::confess "Illegal or invalid share dir type '$type'";
    }

    if( $type eq 'dist' and @_ ) {
        Carp::confess "Too many parameters to delete_share";
    }

    my $def = _mk_def( "delete-$type" );
    _add_module( $def, $_[0] );
    _add_dir( $def, $dir );
}



sub _mk_def
{
    my( $type ) = @_;
    return { type=>$type,
             dotfiles => $INCLUDE_DOTFILES,
             dotdirs => $INCLUDE_DOTDIRS
           };
}

sub _add_module
{
    my( $def, $class ) =  @_;
    if( $def->{type} =~ /module$/ ) {
        my $module = _CLASS( $class );
        unless ( defined $module ) {
            Carp::confess "Missing or invalid module name '$_[0]'";
        }
        $def->{module} = $module;
    }
}

sub _add_dir
{
    my( $def, $dir ) = @_;

    $dir = [ $dir ] unless ref $dir;

    my $del = 0;
    $del = 1 if $def->{type} =~ /^delete-/;

    foreach my $d ( @$dir ) {
        unless ( $del or (defined $d and -d $d) ) {
            Carp::confess "Illegal or missing directory '$d'";
        }
        if( not $del and $ALREADY{ $d }++ ) {
            Carp::confess "Directory '$d' is already being installed";
        }
        push @DIRS, { %$def };
        $DIRS[-1]{dir} = $d;
    }
}


sub postamble
{
    my $self = shift;

    my @ret; # = $self->SUPER::postamble( @_ );
    foreach my $def ( @DIRS ) {
        push @ret, __postamble_share_dir( $self, $def );
    }
    return join "\n", @ret;
}

sub __postamble_share_dir
{
    my( $self, $def ) = @_;

    my $dir = $def->{dir};

    my( $idir );

    if( $def->{type} eq 'delete-dist' ) {
        $idir = File::Spec->catdir( _dist_dir(), $dir );
    }
    elsif( $def->{type} eq 'delete-module' ) {
        $idir = File::Spec->catdir( _module_dir( $def ), $dir );
    }
    elsif ( $def->{type} eq 'dist' ) {
        $idir = _dist_dir();
    }
    else {                                  # delete-share and share
        $idir = _module_dir( $def );
    }

    my @cmds;
    if( $def->{type} =~ /^delete-/ ) {
        @cmds = "\$(RM_RF) $idir";
    }
    else {
        my $autodir = '$(INST_LIB)';
        my $pm_to_blib = $self->oneliner(<<CODE, ['-MExtUtils::Install']);
pm_to_blib({\@ARGV}, '$autodir')
CODE

        my $files = {};
        _scan_share_dir( $files, $idir, $dir, $def );
        @cmds = $self->split_command( $pm_to_blib,
            map { ($self->quote_literal($_) => $self->quote_literal($files->{$_})) } sort keys %$files );
    }

    my $r = join '', map { "\t\$(NOECHO) $_\n" } @cmds;

    # Set up the install
    return "config::\n$r";
}

sub _dist_dir
{
    return File::Spec->catdir( '$(INST_LIB)',
                                    qw( auto share dist ),
                                    '$(DISTNAME)'
                                  );
}

sub _module_dir
{
    my( $def ) = @_;
    my $module = $def->{module};
    $module =~ s/::/-/g;
    return  File::Spec->catdir( '$(INST_LIB)',
                                    qw( auto share module ),
                                    $module
                                  );
}

sub _scan_share_dir
{
    my( $files, $idir, $dir, $def ) = @_;
    my $dh = IO::Dir->new( $dir ) or die "Unable to read $dir: $!";
    my $entry;
    while( defined( $entry = $dh->read ) ) {
        next if $entry =~ /(~|,v|#)$/;
        my $full = File::Spec->catfile( $dir, $entry );
        if( -f $full ) {
            next if not $def->{dotfiles} and $entry =~ /^\./;
            $files->{ $full } = File::Spec->catfile( $idir, $entry );
        }
        elsif( -d $full ) {
            if( $def->{dotdirs} ) {
                next if $entry eq '.' or $entry eq '..' or
                        $entry =~ /^\.(svn|git|cvs)$/;
            }
            else {
                next if $entry =~ /^\./;
            }
            _scan_share_dir( $files, File::Spec->catdir( $idir, $entry ), $full, $def );
        }
    }
}


sub _CLASS ($) {
    (
        defined $_[0]
        and
        ! ref $_[0]
        and
        $_[0] =~ m/^[^\W\d]\w*(?:::\w+)*$/s
    ) ? $_[0] : undef;
}

1;

__END__

