package Net::DNS::RR::SOA;

use strict;
use warnings;
our $VERSION = (qw$Id: SOA.pm 1875 2022-09-23 13:41:03Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Net::DNS::DomainName;
use Net::DNS::Mailbox;


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset, @opaque ) = @_;

	( $self->{mname}, $offset ) = Net::DNS::DomainName1035->decode(@_);
	( $self->{rname}, $offset ) = Net::DNS::Mailbox1035->decode( $data, $offset, @opaque );
	@{$self}{qw(serial refresh retry expire minimum)} = unpack "\@$offset N5", $$data;
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;
	my ( $offset, @opaque ) = @_;

	my $rname = $self->{rname};
	my $rdata = $self->{mname}->encode(@_);
	$rdata .= $rname->encode( $offset + length($rdata), @opaque );
	$rdata .= pack 'N5', $self->serial, @{$self}{qw(refresh retry expire minimum)};
	return $rdata;
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	my $mname  = $self->{mname}->string;
	my $rname  = $self->{rname}->string;
	my $serial = $self->serial;
	my $spacer = length "$serial" > 7 ? "" : "\t";
	return ($mname, $rname,
		join( "\n\t\t\t\t",
			"\t\t\t$serial$spacer\t;serial", "$self->{refresh}\t\t;refresh",
			"$self->{retry}\t\t;retry",	 "$self->{expire}\t\t;expire",
			"$self->{minimum}\t\t;minimum\n" ) );
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	$self->mname(shift);
	$self->rname(shift);
	$self->serial(shift);
	for (qw(refresh retry expire minimum)) {
		$self->$_( Net::DNS::RR::ttl( {}, shift ) ) if scalar @_;
	}
	return;
}


sub _defaults {				## specify RR attribute default values
	my $self = shift;

	$self->_parse_rdata(qw(. . 0 4h 1h 3w 1h));
	delete $self->{serial};
	return;
}


sub mname {
	my $self = shift;

	$self->{mname} = Net::DNS::DomainName1035->new(shift) if scalar @_;
	return $self->{mname} ? $self->{mname}->name : undef;
}


sub rname {
	my $self = shift;

	$self->{rname} = Net::DNS::Mailbox1035->new(shift) if scalar @_;
	return $self->{rname} ? $self->{rname}->address : undef;
}


sub serial {
	my $self = shift;

	return $self->{serial} || 0 unless scalar @_;		# current/default value

	my $value = shift;					# replace if in sequence
	return $self->{serial} = ( $value & 0xFFFFFFFF ) if _ordered( $self->{serial}, $value );

	# unwise to assume 64-bit arithmetic, or that 32-bit integer overflow goes unpunished
	my $serial = 0xFFFFFFFF & ( $self->{serial} || 0 );
	return $self->{serial} = 0x80000000 if $serial == 0x7FFFFFFF;	 # wrap
	return $self->{serial} = 0x00000000 if $serial == 0xFFFFFFFF;	 # wrap
	return $self->{serial} = $serial + 1;			# increment
}


sub refresh {
	my $self = shift;

	$self->{refresh} = 0 + shift if scalar @_;
	return $self->{refresh} || 0;
}


sub retry {
	my $self = shift;

	$self->{retry} = 0 + shift if scalar @_;
	return $self->{retry} || 0;
}


sub expire {
	my $self = shift;

	$self->{expire} = 0 + shift if scalar @_;
	return $self->{expire} || 0;
}


sub minimum {
	my $self = shift;

	$self->{minimum} = 0 + shift if scalar @_;
	return $self->{minimum} || 0;
}



sub _ordered() {			## irreflexive 32-bit partial ordering
	use integer;
	my ( $n1, $n2 ) = @_;

	return 0 unless defined $n2;				# ( any, undef )
	return 1 unless defined $n1;				# ( undef, any )

	# unwise to assume 64-bit arithmetic, or that 32-bit integer overflow goes unpunished
	if ( $n2 < 0 ) {					# fold, leaving $n2 non-negative
		$n1 = ( $n1 & 0xFFFFFFFF ) ^ 0x80000000;	# -2**31 <= $n1 < 2**32
		$n2 = ( $n2 & 0x7FFFFFFF );			#  0	 <= $n2 < 2**31
	}

	return $n1 < $n2 ? ( $n1 > ( $n2 - 0x80000000 ) ) : ( $n2 < ( $n1 - 0x80000000 ) );
}



1;
__END__


