package Net::DNS::RR::SMIMEA;

use strict;
use warnings;
our $VERSION = (qw$Id: SMIMEA.pm 1857 2021-12-07 13:38:02Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;

use constant BABBLE => defined eval { require Digest::BubbleBabble };


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $next = $offset + $self->{rdlength};

	@{$self}{qw(usage selector matchingtype)} = unpack "\@$offset C3", $$data;
	$offset += 3;
	$self->{certbin} = substr $$data, $offset, $next - $offset;
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	return pack 'C3 a*', @{$self}{qw(usage selector matchingtype certbin)};
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	$self->_annotation( $self->babble ) if BABBLE;
	my @cert  = split /(\S{64})/, $self->cert;
	my @rdata = ( $self->usage, $self->selector, $self->matchingtype, @cert );
	return @rdata;
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	$self->usage(shift);
	$self->selector(shift);
	$self->matchingtype(shift);
	$self->cert(@_);
	return;
}


sub usage {
	my $self = shift;

	$self->{usage} = 0 + shift if scalar @_;
	return $self->{usage} || 0;
}


sub selector {
	my $self = shift;

	$self->{selector} = 0 + shift if scalar @_;
	return $self->{selector} || 0;
}


sub matchingtype {
	my $self = shift;

	$self->{matchingtype} = 0 + shift if scalar @_;
	return $self->{matchingtype} || 0;
}


sub cert {
	my $self = shift;
	return unpack "H*", $self->certbin() unless scalar @_;
	return $self->certbin( pack "H*", join "", map { /^"*([\dA-Fa-f]*)"*$/ || croak("corrupt hex"); $1 } @_ );
}


sub certbin {
	my $self = shift;

	$self->{certbin} = shift if scalar @_;
	return $self->{certbin} || "";
}


sub certificate { return &cert; }


sub babble {
	return BABBLE ? Digest::BubbleBabble::bubblebabble( Digest => shift->certbin ) : '';
}


1;
__END__


