package Net::DNS::RR::IPSECKEY;

use strict;
use warnings;
our $VERSION = (qw$Id: IPSECKEY.pm 1857 2021-12-07 13:38:02Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;
use MIME::Base64;

use Net::DNS::DomainName;
use Net::DNS::RR::A;
use Net::DNS::RR::AAAA;


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $limit = $offset + $self->{rdlength};

	@{$self}{qw(precedence gatetype algorithm)} = unpack "\@$offset C3", $$data;
	$offset += 3;

	my $gatetype = $self->{gatetype};
	if ( not $gatetype ) {
		$self->{gateway} = undef;			# no gateway

	} elsif ( $gatetype == 1 ) {
		$self->{gateway} = unpack "\@$offset a4", $$data;
		$offset += 4;

	} elsif ( $gatetype == 2 ) {
		$self->{gateway} = unpack "\@$offset a16", $$data;
		$offset += 16;

	} elsif ( $gatetype == 3 ) {
		my $name;
		( $name, $offset ) = Net::DNS::DomainName->decode( $data, $offset );
		$self->{gateway} = $name;

	} else {
		die "unknown gateway type ($gatetype)";
	}

	$self->keybin( substr $$data, $offset, $limit - $offset );
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	my $gatetype   = $self->gatetype;
	my $gateway    = $self->{gateway};
	my $precedence = $self->precedence;
	my $algorithm  = $self->algorithm;
	my $keybin     = $self->keybin;

	if ( not $gatetype ) {
		return pack 'C3 a*', $precedence, $gatetype, $algorithm, $keybin;

	} elsif ( $gatetype == 1 ) {
		return pack 'C3 a4 a*', $precedence, $gatetype, $algorithm, $gateway, $keybin;

	} elsif ( $gatetype == 2 ) {
		return pack 'C3 a16 a*', $precedence, $gatetype, $algorithm, $gateway, $keybin;

	} elsif ( $gatetype == 3 ) {
		my $namebin = $gateway->encode;
		return pack 'C3 a* a*', $precedence, $gatetype, $algorithm, $namebin, $keybin;
	}
	die "unknown gateway type ($gatetype)";
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	my @params = map { $self->$_ } qw(precedence gatetype algorithm);
	my @base64 = split /\s+/, encode_base64( $self->keybin );
	my @rdata  = ( @params, $self->gateway, @base64 );
	return @rdata;
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	foreach (qw(precedence gatetype algorithm gateway)) { $self->$_(shift) }
	$self->key(@_);
	return;
}


sub precedence {
	my $self = shift;

	$self->{precedence} = 0 + shift if scalar @_;
	return $self->{precedence} || 0;
}


sub gatetype {
	return shift->{gatetype} || 0;
}


sub algorithm {
	my $self = shift;

	$self->{algorithm} = 0 + shift if scalar @_;
	return $self->{algorithm} || 0;
}


sub gateway {
	my $self = shift;

	for (@_) {
		/^\.*$/ && do {
			$self->{gatetype} = 0;
			$self->{gateway}  = undef;		# no gateway
			last;
		};
		/:.*:/ && do {
			$self->{gatetype} = 2;
			$self->{gateway}  = Net::DNS::RR::AAAA::address( {}, $_ );
			last;
		};
		/\.\d+$/ && do {
			$self->{gatetype} = 1;
			$self->{gateway}  = Net::DNS::RR::A::address( {}, $_ );
			last;
		};
		/\..+/ && do {
			$self->{gatetype} = 3;
			$self->{gateway}  = Net::DNS::DomainName->new($_);
			last;
		};
		croak 'unrecognised gateway type';
	}

	if ( defined wantarray ) {
		my $gatetype = $self->{gatetype};
		return wantarray ? '.' : undef unless $gatetype;
		my $gateway = $self->{gateway};
		for ($gatetype) {
			/^1$/ && return Net::DNS::RR::A::address( {address => $gateway} );
			/^2$/ && return Net::DNS::RR::AAAA::address( {address => $gateway} );
			/^3$/ && return wantarray ? $gateway->string : $gateway->name;
			die "unknown gateway type ($gatetype)";
		}
	}
	return;
}


sub key {
	my $self = shift;
	return MIME::Base64::encode( $self->keybin(), "" ) unless scalar @_;
	return $self->keybin( MIME::Base64::decode( join "", @_ ) );
}


sub keybin {
	my $self = shift;

	$self->{keybin} = shift if scalar @_;
	return $self->{keybin} || "";
}


sub pubkey { return &key; }


my $function = sub {			## sort RRs in numerically ascending order.
	return $Net::DNS::a->{'preference'} <=> $Net::DNS::b->{'preference'};
};

__PACKAGE__->set_rrsort_func( 'preference', $function );

__PACKAGE__->set_rrsort_func( 'default_sort', $function );


1;
__END__


