package Net::DNS::RR::HIP;

use strict;
use warnings;
our $VERSION = (qw$Id: HIP.pm 1857 2021-12-07 13:38:02Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;
use Net::DNS::DomainName;
use MIME::Base64;


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my ( $hitlen, $pklen ) = unpack "\@$offset Cxn", $$data;
	@{$self}{qw(algorithm hitbin keybin)} = unpack "\@$offset xCxx a$hitlen a$pklen", $$data;

	my $limit = $offset + $self->{rdlength};
	$offset += 4 + $hitlen + $pklen;
	$self->{servers} = [];
	while ( $offset < $limit ) {
		my $item;
		( $item, $offset ) = Net::DNS::DomainName->decode( $data, $offset );
		push @{$self->{servers}}, $item;
	}
	croak('corrupt HIP data') unless $offset == $limit;	# more or less FUBAR
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	my $hit = $self->hitbin;
	my $key = $self->keybin;
	my $nos = pack 'C2n a* a*', length($hit), $self->algorithm, length($key), $hit, $key;
	return join '', $nos, map { $_->encode } @{$self->{servers}};
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	my $base64 = MIME::Base64::encode( $self->{keybin}, '' );
	my @server = map { $_->string } @{$self->{servers}};
	my @rdata  = ( $self->algorithm, $self->hit, $base64, @server );
	return @rdata;
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	foreach (qw(algorithm hit key)) { $self->$_(shift) }
	$self->servers(@_);
	return;
}


sub algorithm {
	my $self = shift;

	$self->{algorithm} = 0 + shift if scalar @_;
	return $self->{algorithm} || 0;
}


sub hit {
	my $self = shift;
	return unpack "H*", $self->hitbin() unless scalar @_;
	return $self->hitbin( pack "H*", join "", map { /^"*([\dA-Fa-f]*)"*$/ || croak("corrupt hex"); $1 } @_ );
}


sub hitbin {
	my $self = shift;

	$self->{hitbin} = shift if scalar @_;
	return $self->{hitbin} || "";
}


sub key {
	my $self = shift;
	return MIME::Base64::encode( $self->keybin(), "" ) unless scalar @_;
	return $self->keybin( MIME::Base64::decode( join "", @_ ) );
}


sub keybin {
	my $self = shift;

	$self->{keybin} = shift if scalar @_;
	return $self->{keybin} || "";
}


sub servers {
	my $self = shift;

	my $servers = $self->{servers} ||= [];
	@$servers = map { Net::DNS::DomainName->new($_) } @_ if scalar @_;
	return defined(wantarray) ? map( { $_->name } @$servers ) : ();
}

sub rendezvousservers {			## historical
	$_[0]->_deprecate('prefer $rr->servers()');		# uncoverable pod
	my @servers = &servers;
	return \@servers;
}

sub pkalgorithm {			## historical
	return &algorithm;					# uncoverable pod
}

sub pubkey {				## historical
	return &key;						# uncoverable pod
}


1;
__END__


