package Net::DNS::RR::DS;

use strict;
use warnings;
our $VERSION = (qw$Id: DS.pm 1856 2021-12-02 14:36:25Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;

use constant BABBLE => defined eval { require Digest::BubbleBabble };

eval { require Digest::SHA };		## optional for simple Net::DNS RR
eval { require Digest::GOST12 };
eval { require Digest::GOST::CryptoPro };

my %digest = (
	'1' => scalar( eval { Digest::SHA->new(1) } ),
	'2' => scalar( eval { Digest::SHA->new(256) } ),
	'3' => scalar( eval { Digest::GOST::CryptoPro->new() } ),
	'4' => scalar( eval { Digest::SHA->new(384) } ),
	'5' => scalar( eval { Digest::GOST12->new() } ),
	);


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $rdata = substr $$data, $offset, $self->{rdlength};
	@{$self}{qw(keytag algorithm digtype digestbin)} = unpack 'n C2 a*', $rdata;
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	return pack 'n C2 a*', @{$self}{qw(keytag algorithm digtype digestbin)};
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	$self->_annotation( $self->babble ) if BABBLE && $self->{algorithm};
	my @param = @{$self}{qw(keytag algorithm digtype)};
	my @rdata = ( @param, split /(\S{64})/, $self->digest || '-' );
	return @rdata;
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	my $keytag = shift;		## avoid destruction by CDS algorithm(0)
	$self->algorithm(shift);
	$self->keytag($keytag);
	$self->digtype(shift);
	$self->digest(@_);
	return;
}


sub keytag {
	my $self = shift;

	$self->{keytag} = 0 + shift if scalar @_;
	return $self->{keytag} || 0;
}


sub algorithm {
	my ( $self, $arg ) = @_;

	unless ( ref($self) ) {		## class method or simple function
		my $argn = pop;
		return $argn =~ /[^0-9]/ ? _algbyname($argn) : _algbyval($argn);
	}

	return $self->{algorithm} unless defined $arg;
	return _algbyval( $self->{algorithm} ) if uc($arg) eq 'MNEMONIC';
	return $self->{algorithm} = _algbyname($arg) || die _algbyname('')    # disallow algorithm(0)
}


sub digtype {
	my ( $self, $arg ) = @_;

	unless ( ref($self) ) {		## class method or simple function
		my $argn = pop;
		return $argn =~ /[^0-9]/ ? _digestbyname($argn) : _digestbyval($argn);
	}

	return $self->{digtype} unless defined $arg;
	return _digestbyval( $self->{digtype} ) if uc($arg) eq 'MNEMONIC';
	return $self->{digtype} = _digestbyname($arg) || die _digestbyname('')	  # disallow digtype(0)
}


sub digest {
	my $self = shift;
	return unpack "H*", $self->digestbin() unless scalar @_;
	return $self->digestbin( pack "H*", join "", map { /^"*([\dA-Fa-f]*)"*$/ || croak("corrupt hex"); $1 } @_ );
}


sub digestbin {
	my $self = shift;

	$self->{digestbin} = shift if scalar @_;
	return $self->{digestbin} || "";
}


sub babble {
	return BABBLE ? Digest::BubbleBabble::bubblebabble( Digest => shift->digestbin ) : '';
}


sub create {
	my $class = shift;
	my $keyrr = shift;
	my %args  = @_;

	my ($type) = reverse split '::', $class;

	croak "Unable to create $type record for non-zone key" unless $keyrr->zone;
	croak "Unable to create $type record for revoked key" if $keyrr->revoke;
	croak "Unable to create $type record for invalid key" unless $keyrr->protocol == 3;

	my $self = Net::DNS::RR->new(
		owner	  => $keyrr->owner,			# per definition, same as keyrr
		type	  => $type,
		class	  => $keyrr->class,
		ttl	  => $keyrr->{ttl},
		keytag	  => $keyrr->keytag,
		algorithm => $keyrr->algorithm,
		digtype	  => 1,					# SHA1 by default
		%args
		);

	my $hash = $digest{$self->digtype};
	croak join ' ', 'digtype', $self->digtype('MNEMONIC'), 'not supported' unless $hash;
	my $clone = $hash->clone;
	$clone->add( $keyrr->{owner}->canonical );
	$clone->add( $keyrr->_encode_rdata );
	$self->digestbin( $clone->digest );

	return $self;
}


sub verify {
	my ( $self, $key ) = @_;
	my $verify = Net::DNS::RR::DS->create( $key, ( digtype => $self->digtype ) );
	return $verify->digestbin eq $self->digestbin;
}



{
	my @digestbyname = (
		'SHA-1'		    => 1,			# [RFC3658]
		'SHA-256'	    => 2,			# [RFC4509]
		'GOST-R-34.11-94'   => 3,			# [RFC5933]
		'SHA-384'	    => 4,			# [RFC6605]
		'GOST-R-34.11-2012' => 5,			# [RFC5933bis]
		);

	my @digestalias = (
		'SHA'	 => 1,
		'GOST94' => 3,
		'GOST12' => 5,
		);

	my %digestbyval = reverse @digestbyname;

	foreach (@digestbyname) { s/[\W_]//g; }			# strip non-alphanumerics
	my @digestrehash = map { /^\d/ ? ($_) x 3 : uc($_) } @digestbyname;
	my %digestbyname = ( @digestalias, @digestrehash );	# work around broken cperl

	sub _digestbyname {
		my $arg = shift;
		my $key = uc $arg;				# synthetic key
		$key =~ s/[\W_]//g;				# strip non-alphanumerics
		my $val = $digestbyname{$key};
		return $val if defined $val;
		return $key =~ /^\d/ ? $arg : croak qq[unknown algorithm $arg];
	}

	sub _digestbyval {
		my $value = shift;
		return $digestbyval{$value} || return $value;
	}
}


{
	my @algbyname = (
		'DELETE'	     => 0,			# [RFC4034][RFC4398][RFC8078]
		'RSAMD5'	     => 1,			# [RFC3110][RFC4034]
		'DH'		     => 2,			# [RFC2539]
		'DSA'		     => 3,			# [RFC3755][RFC2536]
					## Reserved	=> 4,	# [RFC6725]
		'RSASHA1'	     => 5,			# [RFC3110][RFC4034]
		'DSA-NSEC3-SHA1'     => 6,			# [RFC5155]
		'RSASHA1-NSEC3-SHA1' => 7,			# [RFC5155]
		'RSASHA256'	     => 8,			# [RFC5702]
					## Reserved	=> 9,	# [RFC6725]
		'RSASHA512'	     => 10,			# [RFC5702]
					## Reserved	=> 11,	# [RFC6725]
		'ECC-GOST'	     => 12,			# [RFC5933]
		'ECDSAP256SHA256'    => 13,			# [RFC6605]
		'ECDSAP384SHA384'    => 14,			# [RFC6605]
		'ED25519'	     => 15,			# [RFC8080]
		'ED448'		     => 16,			# [RFC8080]

		'INDIRECT'   => 252,				# [RFC4034]
		'PRIVATEDNS' => 253,				# [RFC4034]
		'PRIVATEOID' => 254,				# [RFC4034]
					## Reserved	=> 255,	# [RFC4034]
		);

	my %algbyval = reverse @algbyname;

	foreach (@algbyname) { s/[\W_]//g; }			# strip non-alphanumerics
	my @algrehash = map { /^\d/ ? ($_) x 3 : uc($_) } @algbyname;
	my %algbyname = @algrehash;				# work around broken cperl

	sub _algbyname {
		my $arg = shift;
		my $key = uc $arg;				# synthetic key
		$key =~ s/[\W_]//g;				# strip non-alphanumerics
		my $val = $algbyname{$key};
		return $val if defined $val;
		return $key =~ /^\d/ ? $arg : croak qq[unknown algorithm $arg];
	}

	sub _algbyval {
		my $value = shift;
		return $algbyval{$value} || return $value;
	}
}



1;
__END__


