package Net::DNS::RR;

use strict;
use warnings;

our $VERSION = (qw$Id: RR.pm 1864 2022-04-14 15:18:49Z willem $)[2];




use integer;
use Carp;

use constant LIB => grep { $_ ne '.' } grep { !ref($_) } @INC;

use Net::DNS::Parameters qw(%classbyname :class :type);
use Net::DNS::DomainName;



sub new {
	return eval {
		local $SIG{__DIE__};
		scalar @_ > 2 ? &_new_hash : &_new_string;
	} || do {
		my $class = shift || __PACKAGE__;
		my @param = map { defined($_) ? split /\s+/ : 'undef' } @_;
		my $stmnt = substr "$class->new( @param )", 0, 80;
		croak "${@}in $stmnt\n";
	};
}



my $PARSE_REGEX = q/("[^"]*")|;[^\n]*|[ \t\n\r\f()]+/;		# NB: *not* \s (matches Unicode white space)

sub _new_string {
	my $base;
	local $_;
	( $base, $_ ) = @_;
	croak 'argument absent or undefined' unless defined $_;
	croak 'non-scalar argument' if ref $_;

	# parse into quoted strings, contiguous non-whitespace and (discarded) comments
	s/\\\\/\\092/g;						# disguise escaped escape
	s/\\"/\\034/g;						# disguise escaped quote
	s/\\\(/\\040/g;						# disguise escaped bracket
	s/\\\)/\\041/g;						# disguise escaped bracket
	s/\\;/\\059/g;						# disguise escaped semicolon
	my ( $owner, @token ) = grep { defined && length } split /$PARSE_REGEX/o;

	croak 'unable to parse RR string' unless scalar @token;
	my $t1 = $token[0];
	my $t2 = $token[1];

	my ( $ttl, $class );
	if ( not defined $t2 ) {				# <owner> <type>
		@token = ('ANY') if $classbyname{uc $t1};	# <owner> <class>
	} elsif ( $t1 =~ /^\d/ ) {
		$ttl   = shift @token;				# <owner> <ttl> [<class>] <type>
		$class = shift @token if $classbyname{uc $t2} || $t2 =~ /^CLASS\d/i;
	} elsif ( $classbyname{uc $t1} || $t1 =~ /^CLASS\d/i ) {
		$class = shift @token;				# <owner> <class> [<ttl>] <type>
		$ttl   = shift @token if $t2 =~ /^\d/;
	}

	my $type      = shift(@token);
	my $populated = scalar @token;

	my $self = $base->_subclass( $type, $populated );	# create RR object
	$self->owner($owner);
	$self->class($class) if defined $class;			# specify CLASS
	$self->ttl($ttl)     if defined $ttl;			# specify TTL

	return $self unless $populated;				# empty RR

	if ( $#token && $token[0] =~ /^[\\]?#$/ ) {
		shift @token;					# RFC3597 hexadecimal format
		my $rdlen = shift(@token) || 0;
		my $rdata = pack 'H*', join( '', @token );
		croak 'length and hexadecimal data inconsistent' unless $rdlen == length $rdata;
		$self->rdata($rdata);				# unpack RDATA
	} else {
		$self->_parse_rdata(@token);			# parse arguments
	}

	$self->_post_parse();
	return $self;
}



my @core = qw(owner name type class ttl rdlength);

sub _new_hash {
	my $base = shift;

	my %attribute = ( owner => '.', type => 'NULL' );
	while ( my $key = shift ) {
		$attribute{lc $key} = shift;
	}

	my ( $owner, $name, $type, $class, $ttl ) = delete @attribute{@core};

	my $self = $base->_subclass( $type, scalar(%attribute) );
	$self->owner( $name ? $name : $owner );
	$self->class($class) if defined $class;			# optional CLASS
	$self->ttl($ttl)     if defined $ttl;			# optional TTL

	eval {
		while ( my ( $attribute, $value ) = each %attribute ) {
			$self->$attribute( ref($value) eq 'ARRAY' ? @$value : $value );
		}
	};
	die ref($self) eq __PACKAGE__ ? "type $type not implemented" : () if $@;

	$self->_post_parse();
	return $self;
}



use constant RRFIXEDSZ => length pack 'n2 N n', (0) x 4;

sub decode {
	my $base = shift;
	my ( $data, $offset, @opaque ) = @_;

	my ( $owner, $fixed ) = decode Net::DNS::DomainName1035(@_);

	my $index = $fixed + RRFIXEDSZ;
	die 'corrupt wire-format data' if length $$data < $index;
	my $self = $base->_subclass( unpack "\@$fixed n", $$data );
	$self->{owner} = $owner;
	@{$self}{qw(class ttl rdlength)} = unpack "\@$fixed x2 n N n", $$data;

	my $next = $index + $self->{rdlength};
	die 'corrupt wire-format data' if length $$data < $next;

	$self->{offset} = $offset || 0;
	$self->_decode_rdata( $data, $index, @opaque ) if $next > $index or $self->type eq 'OPT';
	delete $self->{offset};

	return wantarray ? ( $self, $next ) : $self;
}



sub encode {
	my $self = shift;
	my ( $offset, @opaque ) = scalar(@_) ? @_ : ( 0x4000, {} );

	my $owner = $self->{owner}->encode( $offset, @opaque );
	my ( $type, $class, $ttl ) = @{$self}{qw(type class ttl)};
	my $rdata = $self->_empty ? '' : $self->_encode_rdata( $offset + length($owner) + RRFIXEDSZ, @opaque );
	return pack 'a* n2 N n a*', $owner, $type, $class || 1, $ttl || 0, length $rdata, $rdata;
}



sub canonical {
	my $self = shift;

	my $owner = $self->{owner}->canonical;
	my ( $type, $class, $ttl ) = @{$self}{qw(type class ttl)};
	my $rdata = $self->_empty ? '' : $self->_encode_rdata( length($owner) + RRFIXEDSZ );
	return pack 'a* n2 N n a*', $owner, $type, $class || 1, $ttl || 0, length $rdata, $rdata;
}



sub print {
	print shift->string, "\n";
	return;
}



sub string {
	my $self = shift;

	my $name = $self->{owner}->string;
	my @ttl	 = grep {defined} $self->{ttl};
	my @core = ( $name, @ttl, $self->class, $self->type );

	my $empty = $self->_empty;
	my @rdata = $empty ? () : eval { $self->_format_rdata };
	carp $@ if $@;

	my $tab = length($name) < 72 ? "\t" : ' ';
	$self->_annotation('no data') if $empty;

	my @line = _wrap( join( $tab, @core, '(' ), @rdata, ')' );

	my $last = pop(@line);					# last or only line
	$last = join $tab, @core, "@rdata" unless scalar(@line);

	return join "\n\t", @line, _wrap( $last, map {"; $_"} $self->_annotation );
}



sub plain {
	return join ' ', shift->token;
}



sub token {
	my $self = shift;

	my @ttl	 = grep {defined} $self->{ttl};
	my @core = ( $self->{owner}->string, @ttl, $self->class, $self->type );

	# parse into quoted strings, contiguous non-whitespace and (discarded) comments
	local $_ = $self->_empty ? '' : join( ' ', $self->_format_rdata );
	s/\\\\/\\092/g;						# disguise escaped escape
	s/\\"/\\034/g;						# disguise escaped quote
	s/\\\(/\\040/g;						# disguise escaped bracket
	s/\\\)/\\041/g;						# disguise escaped bracket
	s/\\;/\\059/g;						# disguise escaped semicolon
	return ( @core, grep { defined && length } split /$PARSE_REGEX/o );
}



sub generic {
	my $self = shift;

	my @ttl	  = grep {defined} $self->{ttl};
	my @class = map	 {"CLASS$_"} grep {defined} $self->{class};
	my @core  = ( $self->{owner}->string, @ttl, @class, "TYPE$self->{type}" );
	my $data  = $self->rdata;
	my @data  = ( '\\#', length($data), split /(\S{32})/, unpack 'H*', $data );
	my @line  = _wrap( "@core (", @data, ')' );
	return join "\n\t", @line if scalar(@line) > 1;
	return join ' ', @core, @data;
}



sub owner {
	my $self = shift;
	$self->{owner} = Net::DNS::DomainName1035->new(shift) if scalar @_;
	return defined wantarray ? $self->{owner}->name : undef;
}

sub name { return &owner; }		## historical



sub type {
	my $self = shift;
	croak 'not possible to change RR->type' if scalar @_;
	return typebyval( $self->{type} );
}



sub class {
	my $self = shift;
	return $self->{class} = classbyname(shift) if scalar @_;
	return defined $self->{class} ? classbyval( $self->{class} ) : 'IN';
}



my %unit = ( W => 604800, D => 86400, H => 3600, M => 60, S => 1 );

sub ttl {
	my ( $self, $time ) = @_;

	return $self->{ttl} || 0 unless defined $time;		# avoid defining rr->{ttl}

	my $ttl	 = 0;
	my %time = reverse split /(\D)\D*/, $time . 'S';
	while ( my ( $u, $t ) = each %time ) {
		my $scale = $unit{uc $u} || die qq(bad time: $t$u);
		$ttl += $t * $scale;
	}
	return $self->{ttl} = $ttl;
}



sub _decode_rdata {			## decode rdata from wire-format octet string
	my ( $self, $data, $offset ) = @_;
	return $self->{rdata} = substr $$data, $offset, $self->{rdlength};
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	return shift->{rdata};
}


sub _format_rdata {			## format rdata portion of RR string
	my $rdata = shift->rdata;				# RFC3597 unknown RR format
	return ( '\\#', length($rdata), split /(\S{32})/, unpack 'H*', $rdata );
}


sub _parse_rdata {			## parse RR attributes in argument list
	my $self = shift;
	die join ' ', 'type', $self->type, 'not implemented' if ref($self) eq __PACKAGE__;
	die join ' ', 'no zone file representation defined for', $self->type;
}


sub _post_parse { }			## parser post processing


sub _defaults { }			## set attribute default values


sub dump {				## print internal data structure
	require Data::Dumper;					# uncoverable pod
	local $Data::Dumper::Maxdepth = $Data::Dumper::Maxdepth || 6;
	local $Data::Dumper::Sortkeys = $Data::Dumper::Sortkeys || 1;
	return print Data::Dumper::Dumper(@_);
}

sub rdatastr {				## historical RR subtype method
	return &rdstring;					# uncoverable pod
}



sub rdata {
	my $self = shift;

	return $self->_empty ? '' : eval { $self->_encode_rdata( 0x4000, {} ) } unless @_;

	my $data = shift || '';
	my $hash = {};
	$self->_decode_rdata( \$data, 0, $hash ) if ( $self->{rdlength} = length $data );
	croak 'compression pointer in rdata'	 if keys %$hash;
	return;
}



sub rdstring {
	my $self = shift;

	my @rdata = $self->_empty ? () : eval { $self->_format_rdata };
	carp $@ if $@;

	return join "\n\t", _wrap(@rdata);
}



sub rdlength {
	return length shift->rdata;
}




our %rrsortfunct;

sub set_rrsort_func {
	my $class     = shift;
	my $attribute = shift;
	my $function  = shift;

	my ($type) = $class =~ m/::([^:]+)$/;
	$rrsortfunct{$type}{$attribute} = $function;
	return;
}



my $default = sub { return $Net::DNS::a->canonical() cmp $Net::DNS::b->canonical(); };

sub get_rrsort_func {
	my $class     = shift;
	my $attribute = shift || 'default_sort';

	my ($type) = $class =~ m/::([^:]+)$/;

	return $rrsortfunct{$type}{$attribute} || return $default;
}



our %_MINIMAL = ( 255 => bless ['type' => 255], __PACKAGE__ );
our %_LOADED  = %_MINIMAL;

sub _subclass {
	my ( $class, $rrname, $default ) = @_;

	unless ( $_LOADED{$rrname} ) {
		my $rrtype = typebyname($rrname);

		unless ( $_LOADED{$rrtype} ) {			# load once only
			local @INC = LIB;

			my $identifier = typebyval($rrtype);
			$identifier =~ s/\W/_/g;		# kosher Perl identifier

			my $subclass = join '::', __PACKAGE__, $identifier;

			unless ( eval "require $subclass" ) {	## no critic ProhibitStringyEval
				my $perl = Net::DNS::Parameters::_typespec("$rrtype.RRTYPE");
				$subclass = join '::', __PACKAGE__, "TYPE$rrtype";
				push @INC, sub {		# see perldoc -f require
					my @line = split /\n/, $perl;
					return ( sub { defined( $_ = shift @line ) } );
				};
				eval "require $subclass";	## no critic ProhibitStringyEval
			}

			$subclass = __PACKAGE__ if $@;

			# cache pre-built minimal and populated default object images
			my @base = ( 'type' => $rrtype );
			$_MINIMAL{$rrtype} = bless [@base], $subclass;

			my $object = bless {@base}, $subclass;
			$object->_defaults;
			$_LOADED{$rrtype} = bless [%$object], $subclass;
		}

		$_MINIMAL{$rrname} = $_MINIMAL{$rrtype};
		$_LOADED{$rrname}  = $_LOADED{$rrtype};
	}

	my $prebuilt = $default ? $_LOADED{$rrname} : $_MINIMAL{$rrname};
	return bless {@$prebuilt}, ref($prebuilt);		# create object
}


sub _annotation {
	my $self = shift;
	$self->{annotation} = ["@_"] if scalar @_;
	return wantarray ? @{$self->{annotation} || []} : ();
}


my %warned;

sub _deprecate {
	my $msg = pop(@_);
	carp join ' ', 'deprecated method;', $msg unless $warned{$msg}++;
	return;
}


my %ignore = map { ( $_ => 1 ) } @core, 'annotation', '#';

sub _empty {
	my $self = shift;
	return not( $self->{'#'} ||= scalar grep { !$ignore{$_} } keys %$self );
}


sub _wrap {
	my @text = @_;
	my $cols = 80;
	my $coln = 0;

	my ( @line, @fill );
	foreach (@text) {
		s/\\034/\\"/g;					# tart up escaped "
		s/\\092/\\\\/g;					# tart up escaped escape
		$coln += ( length || next ) + 1;
		if ( $coln > $cols ) {				# start new line
			push( @line, join ' ', @fill ) if @fill;
			$coln = length;
			@fill = ();
		}
		$coln = $cols	  if chomp;			# force line break
		push( @fill, $_ ) if length;
	}
	push @line, join ' ', @fill;
	return @line;
}



our $AUTOLOAD;

sub DESTROY { }				## Avoid tickling AUTOLOAD (in cleanup)

sub AUTOLOAD {				## Default method
	my $self     = shift;
	my ($method) = reverse split /::/, $AUTOLOAD;

	for ($method) {			## tolerate mixed-case attribute name
		return $self->$_(@_) if tr [A-Z-] [a-z_];
	}

	no strict 'refs';		## no critic ProhibitNoStrict
	*{$AUTOLOAD} = sub {undef};	## suppress repetition and deep recursion
	my $oref = ref($self);
	croak qq[$self has no class method "$method"] unless $oref;

	my $string = $self->string;
	my @object = grep { defined($_) } $oref, $oref->VERSION;
	my $module = join '::', __PACKAGE__, $self->type;
	eval("require $module") if $oref eq __PACKAGE__;	## no critic ProhibitStringyEval

	@_ = ( <<"END" );
***  FATAL PROGRAM ERROR!!	Unknown instance method "$method"
***  which the program has attempted to call for the object:
***
$string
***
***  THIS IS A BUG IN THE CALLING SOFTWARE, which incorrectly assumes
***  that the object would be of a particular type.  The type of an
***  object should be checked before calling any of its methods.
***
@object
$@
END
	goto &{'Carp::confess'};
}


1;
__END__



