package Net::DNS::Header;

use strict;
use warnings;

our $VERSION = (qw$Id: Header.pm 1875 2022-09-23 13:41:03Z willem $)[2];




use integer;
use Carp;

use Net::DNS::Parameters qw(:opcode :rcode);



sub string {
	my $self = shift;

	my $id	   = $self->id;
	my $qr	   = $self->qr;
	my $opcode = $self->opcode;
	my $rcode  = $self->rcode;
	my $qd	   = $self->qdcount;
	my $an	   = $self->ancount;
	my $ns	   = $self->nscount;
	my $ar	   = $self->arcount;

	my $opt	 = $$self->edns;
	my $edns = $opt->_specified ? $opt->string : '';

	return <<END . $edns if $opcode eq 'UPDATE';
;;	id = $id
;;	qr = $qr		opcode = $opcode	rcode = $rcode
;;	zocount = $qd	prcount = $an	upcount = $ns	adcount = $ar
END

	my $aa = $self->aa;
	my $tc = $self->tc;
	my $rd = $self->rd;
	my $ra = $self->ra;
	my $zz = $self->z;
	my $ad = $self->ad;
	my $cd = $self->cd;
	my $do = $self->do;

	return <<END . $edns;
;;	id = $id
;;	qr = $qr	aa = $aa	tc = $tc	rd = $rd	opcode = $opcode
;;	ra = $ra	z  = $zz	ad = $ad	cd = $cd	rcode  = $rcode
;;	qdcount = $qd	ancount = $an	nscount = $ns	arcount = $ar
;;	do = $do
END
}



sub print {
	print &string;
	return;
}



my ( $cache1, $cache2, $limit ) = ( {}, {}, 50 );		# two part cache

sub id {
	my $self  = shift;
	my $ident = scalar(@_) ? ( $$self->{id} = shift ) : $$self->{id};
	return $ident if defined $ident;
	$cache2->{$ident = int rand(0xffff)}++;			# preserve recent uniqueness
	$cache2->{$ident = int rand(0xffff)}++ while $cache1->{$ident}++;
	( $cache1, $cache2, $limit ) = ( $cache2, {}, 50 ) unless $limit--;
	return $$self->{id} = $ident;
}



sub opcode {
	my ( $self, $arg ) = @_;
	my $opcode;
	for ( $$self->{status} ) {
		return opcodebyval( ( $_ >> 11 ) & 0x0f ) unless defined $arg;
		$opcode = opcodebyname($arg);
		$_	= ( $_ & 0x87ff ) | ( $opcode << 11 );
	}
	return $opcode;
}



sub rcode {
	my ( $self, $arg ) = @_;
	my $rcode;
	for ( $$self->{status} ) {
		my $opt = $$self->edns;
		unless ( defined $arg ) {
			return rcodebyval( $_ & 0x0f ) unless $opt->_specified;
			$rcode = ( $opt->rcode & 0xff0 ) | ( $_ & 0x00f );
			$opt->rcode($rcode);			# write back full 12-bit rcode
			return $rcode == 16 ? 'BADVERS' : rcodebyval($rcode);
		}
		$rcode = rcodebyname($arg);
		$opt->rcode($rcode);				# full 12-bit rcode
		$_ &= 0xfff0;					# low 4-bit rcode
		$_ |= ( $rcode & 0x000f );
	}
	return $rcode;
}



sub qr {
	return shift->_dnsflag( 0x8000, @_ );
}



sub aa {
	return shift->_dnsflag( 0x0400, @_ );
}



sub tc {
	return shift->_dnsflag( 0x0200, @_ );
}



sub rd {
	return shift->_dnsflag( 0x0100, @_ );
}



sub ra {
	return shift->_dnsflag( 0x0080, @_ );
}



sub z {
	return shift->_dnsflag( 0x0040, @_ );
}



sub ad {
	return shift->_dnsflag( 0x0020, @_ );
}



sub cd {
	return shift->_dnsflag( 0x0010, @_ );
}



our $warned;

sub qdcount {
	my $self = shift;
	return $$self->{count}[0] || scalar @{$$self->{question}} unless scalar @_;
	carp 'packet->header->qdcount attribute is read-only'	  unless $warned++;
	return;
}



sub ancount {
	my $self = shift;
	return $$self->{count}[1] || scalar @{$$self->{answer}} unless scalar @_;
	carp 'packet->header->ancount attribute is read-only'	unless $warned++;
	return;
}



sub nscount {
	my $self = shift;
	return $$self->{count}[2] || scalar @{$$self->{authority}} unless scalar @_;
	carp 'packet->header->nscount attribute is read-only'	   unless $warned++;
	return;
}



sub arcount {
	my $self = shift;
	return $$self->{count}[3] || scalar @{$$self->{additional}} unless scalar @_;
	carp 'packet->header->arcount attribute is read-only'	    unless $warned++;
	return;
}

sub zocount { return &qdcount; }
sub prcount { return &ancount; }
sub upcount { return &nscount; }
sub adcount { return &arcount; }



sub do {
	return shift->_ednsflag( 0x8000, @_ );
}



sub size {
	my $self = shift;
	return $$self->edns->size(@_);
}



sub edns {
	my $self = shift;
	return $$self->edns;
}



sub _dnsflag {
	my $self = shift;
	my $flag = shift;
	for ( $$self->{status} ) {
		my $set = $_ | $flag;
		$_ = (shift) ? $set : ( $set ^ $flag ) if scalar @_;
		$flag &= $_;
	}
	return $flag ? 1 : 0;
}


sub _ednsflag {
	my $self = shift;
	my $flag = shift;
	my $edns = $$self->edns;
	for ( $edns->flags ) {
		my $set = $_ | $flag;
		$edns->flags( $_ = (shift) ? $set : ( $set ^ $flag ) ) if scalar @_;
		$flag &= $_;
	}
	return $flag ? 1 : 0;
}


1;
__END__




