#!/usr/bin/perl

package NetAddr::IP::Lite;

use Carp;
use strict;
use NetAddr::IP::InetBase qw(
	inet_any2n
	isIPv4
	inet_n2dx
	inet_aton
	ipv6_aton
	ipv6_n2x
	fillIPv4
);	
use NetAddr::IP::Util qw(
	addconst
	sub128
	ipv6to4
	notcontiguous
	shiftleft
	hasbits
	bin2bcd
	bcd2bin
	mask4to6
	ipv4to6
	naip_gethostbyname
	havegethostbyname2
);

use vars qw(@ISA @EXPORT_OK $VERSION $Accept_Binary_IP $Old_nth $NoFQDN $AUTOLOAD *Zero);

$VERSION = do { my @r = (q$Revision: 1.57 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };

require Exporter;

@ISA = qw(Exporter);

@EXPORT_OK = qw(Zeros Zero Ones V4mask V4net);


$Accept_Binary_IP = 0;
$Old_nth = 0;
*Zero = \&Zeros;



my $parent = 'NetAddr::IP';


sub DESTROY {};

sub AUTOLOAD {
  no strict;
  my ($pkg,$func) = ($AUTOLOAD =~ /(.*)::([^:]+)$/);
  my $other = $parent .'::';

  if ($pkg =~ /^$other/o && exists ${$other}{$func}) {
    $other .= $func;
    goto &{$other};
  }

  my @stack = caller(0);

  if ( $pkg eq ref $_[0] ) {
    $other = qq|Can't locate object method "$func" via|;
  }
  else {
    $other = qq|Undefined subroutine \&$AUTOLOAD not found in|;
  }
  die $other . qq| package "$parent" or "$pkg" (did you forgot to load a module?) at $stack[1] line $stack[2].\n|;
}



my $_v4zero = pack('L',0);
my $_zero = pack('L4',0,0,0,0);
my $_ones = ~$_zero;
my $_v4mask = pack('L4',0xffffffff,0xffffffff,0xffffffff,0);
my $_v4net = ~ $_v4mask;
my $_ipv4FFFF = pack('N4',0,0,0xffff,0);

sub Zeros() {
  return $_zero;
}
sub Ones() {
  return $_ones;
}
sub V4mask() {
  return $_v4mask;
}
sub V4net() {
  return $_v4net;
}

				#############################################
				# These are the overload methods, placed here
				# for convenience.
				#############################################

use overload

    '+'		=> \&plus,

    '-'		=> \&minus,

    '++'	=> \&plusplus,

    '--'	=> \&minusminus,

    "="		=> \&copy,

    '""'	=> sub { $_[0]->cidr(); },

    'eq'	=> sub {
	my $a = (UNIVERSAL::isa($_[0],__PACKAGE__)) ? $_[0]->cidr : $_[0];
	my $b = (UNIVERSAL::isa($_[1],__PACKAGE__)) ? $_[1]->cidr : $_[1];
	$a eq $b;
    },

    'ne'	=> sub {
	my $a = (UNIVERSAL::isa($_[0],__PACKAGE__)) ? $_[0]->cidr : $_[0];
	my $b = (UNIVERSAL::isa($_[1],__PACKAGE__)) ? $_[1]->cidr : $_[1];
	$a ne $b;
    },

    '=='	=> sub {
	return 0 unless UNIVERSAL::isa($_[0],__PACKAGE__) && UNIVERSAL::isa($_[1],__PACKAGE__);
	$_[0]->cidr eq $_[1]->cidr;
    },

    '!='	=> sub {
	return 1 unless UNIVERSAL::isa($_[0],__PACKAGE__) && UNIVERSAL::isa($_[1],__PACKAGE__);
	$_[0]->cidr ne $_[1]->cidr;
    },

    '>'		=> sub {
	return &comp_addr_mask > 0 ? 1 : 0;
    },

    '<'		=> sub {
	return &comp_addr_mask < 0 ? 1 : 0;
    },

    '>='	=> sub {
	return &comp_addr_mask < 0 ? 0 : 1;
    },

    '<='	=> sub {
	return &comp_addr_mask > 0 ? 0 : 1;
    },

    '<=>'	=> \&comp_addr_mask,

    'cmp'	=> \&comp_addr_mask;

sub comp_addr_mask {
  my($c,$rv) = sub128($_[0]->{addr},$_[1]->{addr});
  return -1 unless $c;
  return 1 if hasbits($rv);
  ($c,$rv) = sub128($_[0]->{mask},$_[1]->{mask});
  return -1 unless $c;
  return hasbits($rv) ? 1 : 0;
}



sub copy {
	return _new($_[0],$_[0]->{addr}, $_[0]->{mask});
}


sub plus {
    my $ip	= shift;
    my $const	= shift;

    return $ip unless $const &&
		$const < 2147483648 &&
		$const > -2147483649;

    my $a = $ip->{addr};
    my $m = $ip->{mask};

    my $lo = $a & ~$m;
    my $hi = $a & $m;

    my $new = ((addconst($lo,$const))[1] & ~$m) | $hi;

    return _new($ip,$new,$m);
}


my $_smsk = pack('L3N',0xffffffff,0xffffffff,0xffffffff,0x80000000);

sub minus {
    my $ip	= shift;
    my $arg	= shift;
    unless (ref $arg) {
	return plus($ip, -$arg);
    }
    my($carry,$dif) = sub128($ip->{addr},$arg->{addr});
    if ($carry) {					# value is positive
	return undef if hasbits($dif & $_smsk);		# all sign bits should be 0's
	return (unpack('L3N',$dif))[3];
    } else {
	return undef if hasbits(($dif & $_smsk) ^ $_smsk);	# sign is 1's
	return (unpack('L3N',$dif))[3] - 4294967296;
    }
}

				# Auto-increment an object


sub plusplus {
    my $ip	= shift;

    my $a = $ip->{addr};
    my $m = $ip->{mask};

    my $lo = $a & ~ $m;
    my $hi = $a & $m;

    $ip->{addr} = ((addconst($lo,1))[1] & ~ $m) | $hi;
    return $ip;
}


sub minusminus {
    my $ip	= shift;

    my $a = $ip->{addr};
    my $m = $ip->{mask};

    my $lo = $a & ~$m;
    my $hi = $a & $m;

    $ip->{addr} = ((addconst($lo,-1))[1] & ~$m) | $hi;
    return $ip;
}

				#############################################
				# End of the overload methods.
				#############################################


				# This is a variant to ->new() that
				# creates and blesses a new object
				# without the fancy parsing of
				# IP formats and shorthands.

sub _new ($$$) {
  my $proto = shift;
  my $class = ref($proto) || die "reference required";
  $proto = $proto->{isv6};
  my $self = {
	addr	=> $_[0],
	mask	=> $_[1],
	isv6	=> $proto,
  };
  return bless $self, $class;
}


my $lbmask = inet_aton('255.0.0.0');
my $_p4broad	= inet_any2n('255.255.255.255');
my $_p4loop	= inet_any2n('127.0.0.1');
my $_p4mloop	= inet_aton('255.0.0.0');
   $_p4mloop	= mask4to6($_p4mloop);
my $_p6loop	= inet_any2n('::1');

my %fip4 = (
        default         => Zeros,
        any             => Zeros,
        broadcast       => $_p4broad,
        loopback        => $_p4loop,
	unspecified	=> undef,
);
my %fip4m = (
        default         => Zeros,
        any             => Zeros,
        broadcast       => Ones,
        loopback        => $_p4mloop,
	unspecified	=> undef,	# not applicable for ipV4
	host		=> Ones,
);

my %fip6 = (
	default         => Zeros,
	any             => Zeros,
	broadcast       => undef,	# not applicable for ipV6
	loopback        => $_p6loop,
	unspecified     => Zeros,
);

my %fip6m = (
	default         => Zeros,
	any             => Zeros,
	broadcast       => undef,	# not applicable for ipV6
	loopback        => Ones,
	unspecified     => Ones,
	host		=> Ones,
);

my $ff000000 = pack('L3N',0xffffffff,0xffffffff,0xffffffff,0xFF000000);
my $ffff0000 = pack('L3N',0xffffffff,0xffffffff,0xffffffff,0xFFFF0000);
my $ffffff00 = pack('L3N',0xffffffff,0xffffffff,0xffffffff,0xFFFFFF00);

sub _obits ($$) {
    my($lo,$hi) = @_;

    return 0xFF if $lo == $hi;
    return (~ ($hi ^ $lo)) & 0xFF;
}

sub new_no($;$$) {
  unshift @_, -1;
  goto &_xnew;
}

sub new($;$$) {
  unshift @_, 0;
  goto &_xnew;
}

sub new_from_aton($$) {
  my $proto     = shift;
  my $class = ref $proto || $proto || __PACKAGE__;
  my $ip = shift;
  return undef unless defined $ip;
  my $addrlen = length($ip);
  return undef unless $addrlen == 4;
  my $self = {
	addr    => ipv4to6($ip),
	mask    => &Ones,
	isv6    => 0,
  };
  return bless $self, $class;
}

sub new6($;$$) {
  unshift @_, 1;
  goto &_xnew;
}

sub new6FFFF($;$$) {
  my $ip = _xnew(1,@_);
  $ip->{addr} |= $_ipv4FFFF;
  return $ip;
}

sub new_cis($;$$) {
  my @in = @_;
  if ( $in[1] && $in[1] =~ m!^(.+)\s+(.+)$! ) {
    $in[1] = $1 .'/'. $2;
  }
  @_ = (0,@in);
  goto &_xnew;
}

sub new_cis6($;$$) {
  my @in = @_;
  if ( $in[1] && $in[1] =~ m!^(.+)\s+(.+)$! ) {
    $in[1] = $1 .'/'. $2;
  }
  @_ = (1,@in);
  goto &_xnew;
}

sub _no_octal {
  (my $rv = $_[0]) =~ s#\b0*([1-9]\d*/?|0/?)#$1#g;	# suppress leading zeros
  $rv;
}

sub _xnew($$;$$) {
  my $noctal	= 0;
  my $isV6	= shift;
  if ($isV6 < 0) {		# flag for no octal?
    $isV6	= 0;
    $noctal	= 1;
  }
  my $proto	= shift;
  my $class	= ref $proto || $proto || __PACKAGE__;
  my $ip	= shift;

  if ($ip && $noctal && $ip !~ m|(?:[^\s0123456789/. -])|) {		# octal suppression required if not an IPv4 address
    $ip = _no_octal($ip);
  }

  return undef if defined $ip && $ip eq '';

  $ip = 'default' unless defined $ip;
  $ip = _retMBIstring($ip)		# treat as big bcd string
	if ref $ip && ref $ip eq 'Math::BigInt';	# can /CIDR notation
  my $hasmask = 1;
  my($mask,$tmp);


  $ip = lc $ip;

  while (1) {
    unless (@_) {
      if ($ip !~ /\D/) {		# binary number notation
	$ip = bcd2bin($ip);
	$mask = Ones;
	last;
      }
      elsif ($ip =~ m!^([a-z0-9.:-]+)(?:/|\s+)([a-z0-9.:-]+)$! ||
	     $ip =~ m!^[\[]{1}([a-z0-9.:-]+)(?:/|\s+)([a-z0-9.:-]+)[\]]{1}$!) {
	$ip	= $1;
	$mask	= $2;
      } elsif (grep($ip eq $_,(qw(default any broadcast loopback unspecified)))) {
	$isV6 = 1 if $ip eq 'unspecified';
	if ($isV6) {
	  $mask = $fip6m{$ip};
	  return undef unless defined ($ip = $fip6{$ip});
	} else {
	  $mask	= $fip4m{$ip};
	  return undef unless defined ($ip = $fip4{$ip});
	}
	last;
      }
    }
    elsif (defined $_[0]) {
      if ($_[0] =~ /ipv6/i || $isV6) {
	if (grep($ip eq $_,(qw(default any loopback unspecified)))) {
	  $mask	= $fip6m{$ip};
	  $ip	= $fip6{$ip};
	  last;
	} else {
	  return undef unless $isV6;
        }
      }
      $mask = $_[0];
    }
    unless (defined $mask) {
      $hasmask	= 0;
      $mask	= 'host';
    }

    my $try;
    $isV6 = 1 if	# check big bcd and IPv6 rfc1884
	( $ip !~ /\D/ && 				  # ip is all decimal
	  (length($ip) > 3 || $ip > 255) &&		  # exclude a single digit in the range of zero to 255, could be funny IPv4
	  ($try = bcd2bin($ip)) && ! isIPv4($try)) ||	  # precedence so $try is not corrupted
	(index($ip,':') >= 0 && ($try = ipv6_aton($ip))); # fails if not an rfc1884 address


    $mask = _retMBIstring($mask)				# treat as big bcd string
        if ref $mask && ref $mask eq 'Math::BigInt';


    $mask = lc $mask;

    if ($mask !~ /\D/) {				# bcd or CIDR notation
      my $isCIDR = length($mask) < 4 && $mask < 129;
      if ($isV6) {
	if ($isCIDR) {
	  my($dq1,$dq2,$dq3,$dq4);
	  if ($ip =~ /^(\d+)(?:|\.(\d+)(?:|\.(\d+)(?:|\.(\d+))))$/ &&
	    do {$dq1 = $1;
		$dq2 = $2 || 0;
		$dq3 = $3 || 0;
		$dq4 = $4 || 0;
		1;
	    } &&
	    $dq1 >= 0 && $dq1 < 256 &&
	    $dq2 >= 0 && $dq2 < 256 &&
	    $dq3 >= 0 && $dq3 < 256 &&
	    $dq4 >= 0 && $dq4 < 256
	  ) {	# corner condition of IPv4 with isV6
	    $ip = join('.',$dq1,$dq2,$dq3,$dq4);
	    $try = ipv4to6(inet_aton($ip));
	    if ($mask < 32) {
	      $mask = shiftleft(Ones,32 -$mask);
	    }
	    elsif ($mask == 32) {
	      $mask = Ones;
	    } else {
	      return undef;			# undoubtably an error
	    }
	  }
	  elsif ($mask < 128) {
	    $mask = shiftleft(Ones,128 -$mask);	# small cidr
	  } else {
	    $mask = Ones();
	  }
	} else {
	  $mask = bcd2bin($mask);
	}
      }
      elsif ($isCIDR && $mask < 33) {		# is V4
	if ($mask < 32) {
	  $mask = shiftleft(Ones,32 -$mask);
	}
	elsif ( $mask == 32) {
	  $mask = Ones;
	} else {
	  $mask = bcd2bin($mask);
	  $mask |= $_v4mask;			# v4 always 
	}
      } else {					# also V4
	$mask = bcd2bin($mask);
	$mask |= $_v4mask;
      }
      if ($try) {				# is a big number
	$ip = $try;
	last;
      }
    } elsif ($mask =~ m/^\d+\.\d+\.\d+\.\d+$/) { # ipv4 form of mask
      $mask = _no_octal($mask) if $noctal;	# filter for octal
      return undef unless defined ($mask = inet_aton($mask));
      $mask = mask4to6($mask);
    } elsif (grep($mask eq $_,qw(default any broadcast loopback unspecified host))) {
      if (index($ip,':') < 0 && ! $isV6) {
	return undef unless defined ($mask = $fip4m{$mask});
      } else {
	return undef unless defined ($mask = $fip6m{$mask});
      }
    } else {
      return undef unless defined ($mask = ipv6_aton($mask));	# try ipv6 form of mask
    }


    if (index($ip,':') < 0) {				# ipv4 address
      if ($ip =~ m/^(\d+)\.(\d+)\.(\d+)\.(\d+)$/) {
	;	# the common case
      }
      elsif (grep($ip eq $_,(qw(default any broadcast loopback)))) {
	return undef unless defined ($ip = $fip4{$ip});
	last;
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)$/) {
	$ip = ($hasmask)
		? "${1}.${2}.0.0"
		: "${1}.0.0.${2}";
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)\.(\d+)$/) {
	$ip = ($hasmask)
		? "${1}.${2}.${3}.0"
		: "${1}.${2}.0.${3}";
      }
      elsif ($ip =~ /^(\d+)$/ && $hasmask && $1 >= 0 and $1 < 256) { # pure numeric
	$ip = sprintf("%d.0.0.0",$1);
      }
      elsif ($ip =~ /^\d+$/ ) {	# a big integer
	$ip = bcd2bin($ip);
	last;
      }
      elsif ($ip =~ /^0[xb]\d+$/ && $hasmask &&
		(($tmp = eval "$ip") || 1) &&
		$tmp >= 0 && $tmp < 256) {
        $ip = sprintf("%d.0.0.0",$tmp);
      }
      elsif ($ip =~ /^-?\d+$/) {
	$ip += 2 ** 32 if $ip < 0;
	$ip = pack('L3N',0,0,0,$ip);
	last;
      }
      elsif ($ip =~ /^-?0[xb]\d+$/) {
	$ip = eval "$ip";
	$ip = pack('L3N',0,0,0,$ip);
	last;
      }


      elsif ($ip =~ m/^(\d+)\.$/) {
	$ip = "${1}.0.0.0";
	$mask = $ff000000;
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)-(\d+)\.?$/ && $2 <= $3 && $3 < 256) {
	$ip = "${1}.${2}.0.0";
	$mask = pack('L3C4',0xffffffff,0xffffffff,0xffffffff,255,_obits($2,$3),0,0);
      }
      elsif ($ip =~ m/^(\d+)-(\d+)\.?$/ and $1 <= $2 && $2 < 256) {
	$ip = "${1}.0.0.0";
	$mask = pack('L3C4',0xffffffff,0xffffffff,0xffffffff,_obits($1,$2),0,0,0)
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)\.$/) {
	$ip = "${1}.${2}.0.0";
	$mask = $ffff0000;
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)\.(\d+)-(\d+)\.?$/ && $3 <= $4 && $4 < 256) {
	$ip = "${1}.${2}.${3}.0";
	$mask = pack('L3C4',0xffffffff,0xffffffff,0xffffffff,255,255,_obits($3,$4),0);
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)\.(\d+)\.$/) {
	$ip = "${1}.${2}.${3}.0";
	$mask = $ffffff00;
      }
      elsif ($ip =~ m/^(\d+)\.(\d+)\.(\d+)\.(\d+)-(\d+)$/ && $4 <= $5 && $5 < 256) {
	$ip = "${1}.${2}.${3}.${4}";
	$mask = pack('L3C4',0xffffffff,0xffffffff,0xffffffff,255,255,255,_obits($4,$5));
      }
      elsif ($ip =~ m/^(\d+\.\d+\.\d+\.\d+)
		\s*-\s*(\d+\.\d+\.\d+\.\d+)$/x) {
	  return undef unless ($ip = inet_aton($1));
	  return undef unless ($tmp = inet_aton($2));
	return undef if ($tmp = unpack('N',$tmp) - unpack('N',$ip)) < 0;
	$ip = ipv4to6($ip);
	$tmp = pack('L3N',0,0,0,$tmp);
	$mask = ~$tmp;
	return undef if notcontiguous($mask);
	return undef if hasbits($ip & $tmp);
	last;
      }
      elsif (! $NoFQDN && $ip !~ /[^a-zA-Z0-9\._-]/ && ($tmp = gethostbyname(fillIPv4($ip))) && $tmp ne $_v4zero && $tmp ne $_zero ) {
	$ip = ipv4to6($tmp);
	last;
      }
      elsif (! $NoFQDN && $ip !~ /[^a-zA-Z0-9\._-]/ && havegethostbyname2() && ($tmp = naip_gethostbyname($ip))) {
	$ip = $tmp;
	$isV6 = 1;
	last;
      }
      elsif ($Accept_Binary_IP && ! $hasmask) {
	if (length($ip) == 4) {
	  $ip = ipv4to6($ip);
	} elsif (length($ip) == 16) {
	  $isV6 = 1;
	} else {
	  return undef;
	}
	last;
      } else {
	return undef;
      }
      return undef unless defined ($ip = inet_aton($ip));
      $ip = ipv4to6($ip);
      last;
    }
    else {						# ipv6 address
      $isV6 = 1;
      $ip = $1 if $ip =~ /\[([^\]]+)\]/;		# transform URI notation
      if (defined ($tmp = ipv6_aton($ip))) {
	$ip = $tmp;
	last;
      }
      last if grep($ip eq $_,(qw(default any loopback unspecified))) &&
		defined ($ip = $fip6{$ip});
      return undef;
    }
  } # end while (1)
  return undef if notcontiguous($mask);			# invalid if not contiguous

  my $self = {
	addr	=> $ip,
	mask	=> $mask,
	isv6	=> $isV6,
  };
  return bless $self, $class;
}


sub broadcast ($) {
  my $ip = _new($_[0],$_[0]->{addr} | ~$_[0]->{mask},$_[0]->{mask});
  $ip->{addr} &= V4net unless $ip->{isv6};
  return $ip;
}


sub network ($) {
  return _new($_[0],$_[0]->{addr} & $_[0]->{mask},$_[0]->{mask});
}


sub addr ($) {
  return ($_[0]->{isv6})
	? ipv6_n2x($_[0]->{addr})
	: inet_n2dx($_[0]->{addr});
}


sub mask ($) {
  return ipv6_n2x($_[0]->{mask}) if $_[0]->{isv6};
  my $mask = isIPv4($_[0]->{addr})
	? $_[0]->{mask} & V4net
	: $_[0]->{mask};
  return inet_n2dx($mask);
}


sub masklen ($) {
  my $len = (notcontiguous($_[0]->{mask}))[1];
  return 0 unless $len;
  return $len if $_[0]->{isv6};
  return isIPv4($_[0]->{addr})
	? $len -96
	: $len;
}


sub bits {
  return $_[0]->{isv6} ? 128 : 32;
}


sub version {
  my $self = shift;
  return $self->{isv6} ? 6 : 4;
}


sub cidr ($) {
  return $_[0]->addr . '/' . $_[0]->masklen;
}


sub aton {
  return $_[0]->{addr} if $_[0]->{isv6};
  return isIPv4($_[0]->{addr})
	? ipv6to4($_[0]->{addr})
	: $_[0]->{addr};
}


sub range ($) {
  return $_[0]->network->addr . ' - ' . $_[0]->broadcast->addr;
}


sub numeric ($) {
  if (wantarray) {
    if (! $_[0]->{isv6} && isIPv4($_[0]->{addr})) {
      return (	sprintf("%u",unpack('N',ipv6to4($_[0]->{addr}))),
		sprintf("%u",unpack('N',ipv6to4($_[0]->{mask}))));
    }
    else {
      return (	bin2bcd($_[0]->{addr}),
		bin2bcd($_[0]->{mask}));
    }
  }
  return (! $_[0]->{isv6} && isIPv4($_[0]->{addr}))
    ? sprintf("%u",unpack('N',ipv6to4($_[0]->{addr})))
    : bin2bcd($_[0]->{addr});
}


my $biloaded;
my $bi2strng;
my $no_mbi_emu = 1;

sub _force_bi_emu {
  undef $biloaded;
  undef $bi2strng;
  $no_mbi_emu = 0;
  print STDERR "\n\n\tWARNING: test development mode, this
\tmessage SHOULD NEVER BE SEEN IN PRODUCTION!
set my \$no_mbi_emu = 1 in t/bigint.t to remove this warning\n\n";
}


sub _bi_stfy {
  "$_[0]" =~ /(\d+)/;		# stringify and remove '+' if present
  $1;
}

sub _fakebi2strg {
  ${$_[0]} =~ /(\d+)/;
  $1;
}

sub _bi_fake {
  bless \('+'. $_[1]), 'Math::BigInt';
}


sub _loadMBI {						# load Math::BigInt on demand
  if (eval {$no_mbi_emu && require Math::BigInt}) {	# any version should work, three known
    import Math::BigInt;
    $biloaded = \&Math::BigInt::new;
    $bi2strng = \&_bi_stfy;
  } else {
    $biloaded = \&_bi_fake;
    $bi2strng = \&_fakebi2strg;
  }
}

sub _retMBIstring {
  _loadMBI unless $biloaded;				# load Math::BigInt on demand
  $bi2strng->(@_);
}

sub _biRef {
  _loadMBI unless $biloaded;				# load Math::BigInt on demand
  $biloaded->('Math::BigInt',$_[0]);
}

sub bigint($) {
  my($addr,$mask);
  if (wantarray) {
    if (! $_[0]->{isv6} && isIPv4($_[0]->{addr})) {
      $addr = $_[0]->{addr}
	? sprintf("%u",unpack('N',ipv6to4($_[0]->{addr})))
	: 0;
      $mask = $_[0]->{mask}
	? sprintf("%u",unpack('N',ipv6to4($_[0]->{mask})))
	: 0;
    }
    else {
      $addr = $_[0]->{addr}
	? bin2bcd($_[0]->{addr})
	: 0;
      $mask = $_[0]->{mask}
	? bin2bcd($_[0]->{mask})
	: 0;
    }
    (_biRef($addr),_biRef($mask));

  } else {	# not wantarray

    if (! $_[0]->{isv6} && isIPv4($_[0]->{addr})) {
      $addr = $_[0]->{addr}
	? sprintf("%u",unpack('N',ipv6to4($_[0]->{addr})))
	: 0;
    } else {
      $addr = $_[0]->{addr}
	? bin2bcd($_[0]->{addr})
	: 0;
    }
    _biRef($addr);
  }
}


sub contains ($$) {
  return within(@_[1,0]);
}


sub within ($$) {
  return 1 unless hasbits($_[1]->{mask});	# 0x0 contains everything
  my $netme	= $_[0]->{addr} & $_[0]->{mask};
  my $brdme	= $_[0]->{addr} | ~ $_[0]->{mask};
  my $neto	= $_[1]->{addr} & $_[1]->{mask};
  my $brdo	= $_[1]->{addr} | ~ $_[1]->{mask};
  return (sub128($netme,$neto) && sub128($brdo,$brdme))
	? 1 : 0;
}


my $ip_10	= NetAddr::IP::Lite->new('10.0.0.0/8');
my $ip_10n	= $ip_10->{addr};               # already the right value
my $ip_10b	= $ip_10n | ~ $ip_10->{mask};

my $ip_172	= NetAddr::IP::Lite->new('172.16.0.0/12');
my $ip_172n	= $ip_172->{addr};              # already the right value
my $ip_172b	= $ip_172n | ~ $ip_172->{mask};

my $ip_192	= NetAddr::IP::Lite->new('192.168.0.0/16');
my $ip_192n	= $ip_192->{addr};              # already the right value
my $ip_192b	= $ip_192n | ~ $ip_192->{mask};

sub is_rfc1918 ($) {
  my $netme     = $_[0]->{addr} & $_[0]->{mask};
  my $brdme     = $_[0]->{addr} | ~ $_[0]->{mask};
  return 1 if (sub128($netme,$ip_10n) && sub128($ip_10b,$brdme));
  return 1 if (sub128($netme,$ip_192n) && sub128($ip_192b,$brdme));
  return (sub128($netme,$ip_172n) && sub128($ip_172b,$brdme))
        ? 1 : 0;
}


my $_lclhost6	= NetAddr::IP::Lite->new('::1');
my $_lclnet	= NetAddr::IP::Lite->new('127/8');

sub is_local ($) {
  return ($_[0]->{isv6})
	? $_[0] == $_lclhost6
	: $_[0]->within($_lclnet);
}


my $_cidr127 = pack('N4',0xffffffff,0xffffffff,0xffffffff,0xfffffffe);

sub first ($) {
  if (hasbits($_[0]->{mask} ^ $_cidr127)) {
    return $_[0]->network + 1;
  } else {
    return $_[0]->network;
  }
}


sub last ($) {
  if (hasbits($_[0]->{mask} ^ $_cidr127)) {
    return $_[0]->broadcast - 1;
  } else {
    return $_[0]->broadcast;
  }
}


sub nth ($$) {
  my $self    = shift;
  my $count   = shift;

  my $slash31 = ! hasbits($self->{mask} ^ $_cidr127);
  if ($Old_nth) {
    return undef if $slash31 && $count != 1;
    return undef if ($count < 1 or $count > $self->num ());
  }
  elsif ($slash31) {
    return undef if ($count && $count != 1);	# only index 0, 1 allowed for /31
  } else {
    ++$count;
    return undef if ($count < 1 or $count > $self->num ());
  }
  return $self->network + $count;
}


sub num ($) {
  if ($Old_nth) {
    my @net = unpack('L3N',$_[0]->{mask} ^ Ones);
    return 0xfffffffe if $net[0] || $net[1] || $net[2]; # 2**32 -1
    return $net[3] if $net[3];
  } else {	# returns 1 for /32 /128, 2 for /31 /127 else n-2 up to 2**32
    (undef, my $net) = addconst($_[0]->{mask},1);
    return 1 unless hasbits($net);	# ipV4/32 or ipV6/128
    $net = $net ^ Ones;
    return 2 unless hasbits($net);	# ipV4/31 or ipV6/127
    $net &= $_v4net unless $_[0]->{isv6};
    return bin2bcd($net);
  }
}



sub import {
  if (grep { $_ eq ':aton' } @_) {
    $Accept_Binary_IP = 1;
    @_ = grep { $_ ne ':aton' } @_;
  }
  if (grep { $_ eq ':old_nth' } @_) {
    $Old_nth = 1;
    @_ = grep { $_ ne ':old_nth' } @_;
  }
  if (grep { $_ eq ':lower' } @_)
  {
    NetAddr::IP::Util::lower();
    @_ = grep { $_ ne ':lower' } @_;
  }
  if (grep { $_ eq ':upper' } @_)
  {
    NetAddr::IP::Util::upper();
    @_ = grep { $_ ne ':upper' } @_;
  }
  if (grep { $_ eq ':nofqdn' } @_)
  {
    $NoFQDN = 1;
    @_ = grep { $_ ne ':nofqdn' } @_;
  }
  NetAddr::IP::Lite->export_to_level(1, @_);
}


1;
