#!/usr/bin/perl
package NetAddr::IP::InetBase;

use strict;

use vars qw($VERSION @EXPORT_OK @ISA %EXPORT_TAGS $Mode);
use AutoLoader qw(AUTOLOAD);
require Exporter;

@ISA = qw(Exporter);

$VERSION = do { my @r = (q$Revision: 0.08 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };

@EXPORT_OK = qw(
	inet_aton
	inet_ntoa
	ipv6_aton
	ipv6_ntoa
	ipv6_n2x
	ipv6_n2d
	inet_any2n
	inet_n2dx
	inet_n2ad
	inet_ntop
	inet_pton
	packzeros
	isIPv4
	isNewIPv4
	isAnyIPv4
	AF_INET
	AF_INET6
	fake_AF_INET6
	fillIPv4
);

%EXPORT_TAGS = (
	all     => [@EXPORT_OK],
	ipv4	=> [qw(
		inet_aton
		inet_ntoa
		fillIPv4
	)],
	ipv6	=> [qw(
		ipv6_aton
		ipv6_ntoa
		ipv6_n2x
		ipv6_n2d
		inet_any2n
		inet_n2dx
		inet_n2ad
		inet_pton
		inet_ntop
		packzeros
	)],
);

sub inet_ntoa;
sub ipv6_aton;
sub ipv6_ntoa;
sub inet_any2n($);
sub inet_n2dx($);
sub inet_n2ad($);
sub _inet_ntop;
sub _inet_pton;

my $emulateAF_INET6 = 0;

{ no warnings 'once';

*packzeros = \&_packzeros;


require Socket;

*AF_INET = \&Socket::AF_INET;

if (eval { AF_INET6() } ) {
  *AF_INET6 = \&Socket::AF_INET6;
  $emulateAF_INET6 = -1;			# have it, remind below
}
if (eval{ require Socket6 } ) {
  import Socket6 qw(
	inet_pton
	inet_ntop
  );
  unless ($emulateAF_INET6) {
    *AF_INET6 = \&Socket6::AF_INET6;
  }
  $emulateAF_INET6 = 0;				# clear, have it from elsewhere or here
} else {
  unless ($emulateAF_INET6) {	# unlikely at this point
    if ($^O =~ /(?:free|dragon.+)bsd/i) {	# FreeBSD, DragonFlyBSD
	$emulateAF_INET6 = 28;
    } elsif ($^O =~ /bsd/i) {		# other BSD flavors like NetBDS, OpenBSD, BSD
	$emulateAF_INET6 = 24;
    } elsif ($^O =~ /(?:darwin|mac)/i) {	# Mac OS X
	$emulateAF_INET6 = 30;
    } elsif ($^O =~ /win/i) {		# Windows
	$emulateAF_INET6 = 23;
    } elsif ($^O =~ /(?:solaris|sun)/i) {		# Sun box
	$emulateAF_INET6 = 26;
    } else {					# use linux default
	$emulateAF_INET6 = 10;
    }
    *AF_INET6 = sub { $emulateAF_INET6; };
  } else {
    $emulateAF_INET6 = 0;			# clear, have it from elsewhere
  }
  *inet_pton = \&_inet_pton;
  *inet_ntop = \&_inet_ntop;
}

} # end no warnings 'once'

sub fake_AF_INET6 {
  return $emulateAF_INET6;
}

BEGIN {
  use vars qw($n2x_format $n2d_format);
  $n2x_format = "%x:%x:%x:%x:%x:%x:%x:%x";
  $n2d_format = "%x:%x:%x:%x:%x:%x:%d.%d.%d.%d";
}

my $case = 0;	# default lower case

sub upper { $n2x_format = uc($n2x_format); $n2d_format = uc($n2d_format); $case = 1; }
sub lower { $n2x_format = lc($n2x_format); $n2d_format = lc($n2d_format); $case = 0; }

sub ipv6_n2x {
  die "Bad arg length for 'ipv6_n2x', length is ". length($_[0]) ." should be 16"
	unless length($_[0]) == 16;
  return sprintf($n2x_format,unpack("n8",$_[0]));
}

sub ipv6_n2d {
  die "Bad arg length for 'ipv6_n2d', length is ". length($_[0]) ." should be 16"
	unless length($_[0]) == 16;
  my @hex = (unpack("n8",$_[0]));
  $hex[9] = $hex[7] & 0xff;
  $hex[8] = $hex[7] >> 8;
  $hex[7] = $hex[6] & 0xff;
  $hex[6] >>= 8;
  return sprintf($n2d_format,@hex);
}



sub fillIPv4 {
  my $host = $_[0];
  return undef unless defined $host;
  if ($host =~ /^(\d+)(?:|\.(\d+)(?:|\.(\d+)(?:|\.(\d+))))$/) {
    if (defined $4) {
      return undef unless
        $1 >= 0 && $1 < 256 &&
        $2 >= 0 && $2 < 256 &&
        $3 >= 0 && $3 < 256 &&
        $4 >= 0 && $4 < 256;
      $host = $1.'.'.$2.'.'.$3.'.'.$4;
    } elsif (defined $3) {
      return undef unless  
        $1 >= 0 && $1 < 256 &&
        $2 >= 0 && $2 < 256 &&
        $3 >= 0 && $3 < 256;  
      $host = $1.'.'.$2.'.0.'.$3
    } elsif (defined $2) {
      return undef unless  
        $1 >= 0 && $1 < 256 &&
        $2 >= 0 && $2 < 256;  
      $host = $1.'.0.0.'.$2;
    } else {
      $host = '0.0.0.'.$1;
    }
  }
  $host;
} 	

sub inet_aton {
  my $host = fillIPv4($_[0]);
  return $host ? scalar gethostbyname($host) : undef;
}


my $_zero = pack('L4',0,0,0,0);
my $_ipv4mask = pack('L4',0xffffffff,0xffffffff,0xffffffff,0);

sub isIPv4 {
  if (length($_[0]) != 16) {
    my $sub = (caller(1))[3] || (caller(0))[3];
    die "Bad arg length for $sub, length is ". (length($_[0]) *8) .", should be 128";
  }
  return ($_[0] & $_ipv4mask) eq $_zero
	? 1 : 0;
}

my $_newV4compat = pack('N4',0,0,0xffff,0);

sub isNewIPv4 {
  my $naddr = $_[0] ^ $_newV4compat;
  return isIPv4($naddr);
}

sub isAnyIPv4 {
  my $naddr = $_[0];
  my $rv = isIPv4($_[0]);
  return $rv if $rv;
  return isNewIPv4($naddr);
}

sub DESTROY {};

sub import {
  if (grep { $_ eq ':upper' } @_) {
	upper();
	@_ = grep { $_ ne ':upper' } @_;
  }
  NetAddr::IP::InetBase->export_to_level(1,@_);
}

1;

__END__


sub inet_ntoa {
  die 'Bad arg length for '. __PACKAGE__ ."::inet_ntoa, length is ". length($_[0]) ." should be 4"
        unless length($_[0]) == 4;
  my @hex = (unpack("n2",$_[0]));
  $hex[3] = $hex[1] & 0xff;
  $hex[2] = $hex[1] >> 8;
  $hex[1] = $hex[0] & 0xff;
  $hex[0] >>= 8;
  return sprintf("%d.%d.%d.%d",@hex);
}


sub ipv6_aton {
  my($ipv6) = @_;
  return undef unless $ipv6;
  local($1,$2,$3,$4,$5);
  if ($ipv6 =~ /^(.*:)(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})$/) {	# mixed hex, dot-quad
    return undef if $2 > 255 || $3 > 255 || $4 > 255 || $5 > 255;
    $ipv6 = sprintf("%s%X%02X:%X%02X",$1,$2,$3,$4,$5);			# convert to pure hex
  }
  my $c;
  return undef if
	$ipv6 =~ /[^:0-9a-fA-F]/ ||			# non-hex character
	(($c = $ipv6) =~ s/::/x/ && $c =~ /(?:x|:):/) ||	# double :: ::?
	$ipv6 =~ /[0-9a-fA-F]{5,}/;			# more than 4 digits
  $c = $ipv6 =~ tr/:/:/;				# count the colons
  return undef if $c < 7 && $ipv6 !~ /::/;
  if ($c > 7) {						# strip leading or trailing ::
    return undef unless
	$ipv6 =~ s/^::/:/ ||
	$ipv6 =~ s/::$/:/;
    return undef if --$c > 7;
  }
  while ($c++ < 7) {					# expand compressed fields
    $ipv6 =~ s/::/:::/;
  }
  $ipv6 .= 0 if $ipv6 =~ /:$/;
  my @hex = split(/:/,$ipv6);
  foreach(0..$#hex) {
    $hex[$_] = hex($hex[$_] || 0);
  }
  pack("n8",@hex);
}


sub ipv6_ntoa {
  return inet_ntop(AF_INET6(),$_[0]);
}


sub inet_any2n($) {
  my($addr) = @_;
  $addr = '' unless $addr;
  $addr = '::' . $addr
	unless $addr =~ /:/;
  return ipv6_aton($addr);
}


sub inet_n2dx($) {
  my($nadr) = @_;
  if (isAnyIPv4($nadr)) {
    local $1;
    ipv6_n2d($nadr) =~ /([^:]+)$/;
    return $1;
  }
  return ipv6_n2x($nadr);
}


sub inet_n2ad($) {
  my($nadr) = @_;
  my $addr = ipv6_n2d($nadr);
  return $addr unless isAnyIPv4($nadr);
  local $1;
  $addr =~ /([^:]+)$/;
  return $1;
}


sub _inet_pton {
  my($af,$ip) = @_;
  die 'Bad address family for '. __PACKAGE__ ."::inet_pton, got $af"
	unless $af == AF_INET6() || $af == AF_INET();
  if ($af == AF_INET()) {
    inet_aton($ip);
  } else {
    ipv6_aton($ip);
  }
}


sub _inet_ntop {
  my($af,$naddr) = @_;
  die 'Unsupported address family for '. __PACKAGE__ ."::inet_ntop, af is $af"
	unless $af == AF_INET6() || $af == AF_INET();
  if ($af == AF_INET()) {
    inet_ntoa($naddr);
  } else {
    return ($case)
	? lc packzeros(ipv6_n2x($naddr))
	: _packzeros(ipv6_n2x($naddr));
  }
}


sub _packzeros {
  my $x6 = shift;
  if ($x6 =~ /\:\:/) {				# already contains ::
    $x6 = ($x6 =~ /\:\d+\.\d+\.\d+\.\d+/)	# ipv4 notation ?
	? ipv6_n2d(ipv6_aton($x6))
	: ipv6_n2x(ipv6_aton($x6));
  }
  $x6 = ':'. lc $x6;				# prefix : & always lower case
  my $d = '';
  if ($x6 =~ /(.+\:)(\d+\.\d+\.\d+\.\d+)/) {	# if contains dot quad
    $x6 = $1;					# save hex piece
    $d = $2;					# and dot quad piece
  }
  $x6 .= ':';					# suffix :
  $x6 =~ s/\:0+/\:0/g;				# compress strings of 0's to single '0'
  $x6 =~ s/\:0([1-9a-f]+)/\:$1/g;		# eliminate leading 0's in hex strings
  my @x = $x6 =~ /(?:\:0)*/g;			# split only strings of :0:0..."

  my $m = 0;
  my $i = 0;

  for (0..$#x) {				# find next longest pattern :0:0:0...
    my $len = length($x[$_]);
    next unless $len > $m;
    $m = $len;
    $i = $_;					# index to first longest pattern
  }

  if ($m > 2) {					# there was a string of 2 or more zeros
    $x6 =~ s/$x[$i]/\:/;	  		# replace first longest :0:0:0... with "::"
    unless ($i) {				# if it is the first match, $i = 0
      $x6 = substr($x6,0,-1);			# keep the leading ::, remove trailing ':'
    } else {
      $x6 = substr($x6,1,-1);			# else remove leading & trailing ':'
    }
    $x6 .= ':' unless $x6 =~ /\:\:/;		# restore ':' if match and we can't see it, implies trailing '::'
  } else {					# there was no match
    $x6 = substr($x6,1,-1);			# remove leading & trailing ':'
  }
  $x6 .= $d;					# append digits if any
  return $case
	? uc $x6
	: $x6;
}


1;
