#!/usr/bin/perl -w

package NetAddr::IP;

use strict;
use Carp;
use NetAddr::IP::Lite 1.57 qw(Zero Zeros Ones V4mask V4net);
use NetAddr::IP::Util 1.53 qw(
	sub128
	inet_aton
	inet_any2n
	ipv6_aton
	isIPv4
	ipv4to6
	mask4to6
	shiftleft
	addconst
	hasbits
	notcontiguous
);

use AutoLoader qw(AUTOLOAD);

use vars qw(
	@EXPORT_OK
	@EXPORT_FAIL
	@ISA
	$VERSION
	$_netlimit
	$rfc3021
);
require Exporter;

@EXPORT_OK = qw(Compact Coalesce Zero Zeros Ones V4mask V4net netlimit);
@EXPORT_FAIL = qw($_netlimit);

@ISA = qw(Exporter NetAddr::IP::Lite);

$VERSION = do { sprintf " %d.%03d", (q$Revision: 4.79 $ =~ /\d+/g) };

$rfc3021 = 0;


$_netlimit = 2 ** 16;			# default

sub netlimit($) {
  return undef unless $_[0];
  return undef if $_[0] =~ /\D/;
  return undef if $_[0] < 16;
  return undef if $_[0] > 24;
  $_netlimit = 2 ** $_[0];
};


				#############################################
				# These are the overload methods, placed here
				# for convenience.
				#############################################

use overload

    '@{}'	=> sub {
	return [ $_[0]->hostenum ];
    };


				#############################################
				# End of the overload methods.
				#############################################




my $full_format = "%04X:%04X:%04X:%04X:%04X:%04X:%D.%D.%D.%D";
my $full6_format = "%04X:%04X:%04X:%04X:%04X:%04X:%04X:%04X";

sub import
{
    if (grep { $_ eq ':old_storable' } @_) {
	@_ = grep { $_ ne ':old_storable' } @_;
    } else {
	*{STORABLE_freeze} = sub
	{
	    my $self = shift;
	    return $self->cidr();	# use stringification
	};
	*{STORABLE_thaw} = sub
	{
	    my $self	= shift;
	    my $cloning	= shift;	# Not used
	    my $serial	= shift;

	    my $ip = new NetAddr::IP $serial;
	    $self->{addr} = $ip->{addr};
	    $self->{mask} = $ip->{mask};
	    $self->{isv6} = $ip->{isv6};
	    return;
	};
    }

    if (grep { $_ eq ':aton' } @_)
    {
	$NetAddr::IP::Lite::Accept_Binary_IP = 1;
	@_ = grep { $_ ne ':aton' } @_;
    }
    if (grep { $_ eq ':old_nth' } @_)
    {
	$NetAddr::IP::Lite::Old_nth = 1;
	@_ = grep { $_ ne ':old_nth' } @_;
    }
    if (grep { $_ eq ':nofqdn'} @_)
    {
	$NetAddr::IP::NetAddr::IP::Lite::NoFQDN = 1;
	@_ = grep { $_ ne ':nofqdn' } @_;
    }
    if (grep { $_ eq ':lower' } @_)
    {
        $full_format = lc($full_format);
        $full6_format = lc($full6_format);
        NetAddr::IP::Util::lower();
	@_ = grep { $_ ne ':lower' } @_;
    }
    if (grep { $_ eq ':upper' } @_)
    {
        $full_format = uc($full_format);
        $full6_format = uc($full6_format);
        NetAddr::IP::Util::upper();
	@_ = grep { $_ ne ':upper' } @_;
    }
    if (grep { $_ eq ':rfc3021' } @_)
    {
	$rfc3021 = 1;
        @_ = grep { $_ ne ':rfc3021' } @_;
    }
    NetAddr::IP->export_to_level(1, @_);
}

sub compact {
    return (ref $_[0] eq 'ARRAY')
	? compactref($_[0])	# Compact(\@list)
	: @{compactref(\@_)};	# Compact(@list)  or ->compact(@list)
}

*Compact = \&compact;

sub Coalesce {
  return &coalesce;
}

sub hostenumref($) {
  my $r = _splitref(0,$_[0]);
  unless ((notcontiguous($_[0]->{mask}))[1] == 128 ||
	  ($rfc3021 && $_[0]->masklen == 31) ) {
    splice(@$r, 0, 1);
    splice(@$r, scalar @$r - 1, 1);
  }
  return $r;
}

sub splitref {
  unshift @_, 0;	# mark as no reverse
  &_splitref;
}

sub rsplitref {
  unshift @_, 1;	# mark as reversed
  &_splitref;
}

sub split {
  unshift @_, 0;	# mark as no reverse
  my $rv = &_splitref;
  return $rv ? @$rv : ();
}

sub rsplit {
  unshift @_, 1;	# mark as reversed
  my $rv = &_splitref;
  return $rv ? @$rv : ();
}

sub full($) {
  if (! $_[0]->{isv6} && isIPv4($_[0]->{addr})) {
    my @hex = (unpack("n8",$_[0]->{addr}));
    $hex[9] = $hex[7] & 0xff;
    $hex[8] = $hex[7] >> 8;
    $hex[7] = $hex[6] & 0xff;
    $hex[6] >>= 8;
    return sprintf($full_format,@hex);
  } else {
    &full6;
  }
}

sub full6($) {
  my @hex = (unpack("n8",$_[0]->{addr}));
  return sprintf($full6_format,@hex);
}

sub full6m($) {
  my @hex = (unpack("n8",$_[0]->{mask}));
  return sprintf($full6_format,@hex);
}

sub DESTROY {};

1;
__END__

sub do_prefix ($$$) {
    my $mask	= shift;
    my $faddr	= shift;
    my $laddr	= shift;

    if ($mask > 24) {
	return "$faddr->[0].$faddr->[1].$faddr->[2].$faddr->[3]-$laddr->[3]";
    }
    elsif ($mask == 24) {
	return "$faddr->[0].$faddr->[1].$faddr->[2].";
    }
    elsif ($mask > 16) {
	return "$faddr->[0].$faddr->[1].$faddr->[2]-$laddr->[2].";
    }
    elsif ($mask == 16) {
	return "$faddr->[0].$faddr->[1].";
    }
    elsif ($mask > 8) {
	return "$faddr->[0].$faddr->[1]-$laddr->[1].";
    }
    elsif ($mask == 8) {
	return "$faddr->[0].";
    }
    else {
	return "$faddr->[0]-$laddr->[0]";
    }
}


sub prefix($) {
    return undef if $_[0]->{isv6};
    my $mask = (notcontiguous($_[0]->{mask}))[1];
    return $_[0]->addr if $mask == 128;
    $mask -= 96;
    my @faddr = split (/\./, $_[0]->first->addr);
    my @laddr = split (/\./, $_[0]->broadcast->addr);
    return do_prefix $mask, \@faddr, \@laddr;
}


sub nprefix($) {
    return undef if $_[0]->{isv6};
    my $mask = (notcontiguous($_[0]->{mask}))[1];
    return $_[0]->addr if $mask == 128;
    $mask -= 96;
    my @faddr = split (/\./, $_[0]->first->addr);
    my @laddr = split (/\./, $_[0]->last->addr);
    return do_prefix $mask, \@faddr, \@laddr;
}


sub wildcard($) {
  my $copy = $_[0]->copy;
  $copy->{addr} = ~ $copy->{mask};
  $copy->{addr} &= V4net unless $copy->{isv6};
  if (wantarray) {
    return ($_[0]->addr, $copy->addr);
  }
  return $copy->addr;
}


sub _compact_v6 ($) {
    my $addr = shift;

    my @o = split /:/, $addr;
    return $addr unless @o and grep { $_ =~ m/^0+$/ } @o;

    my @candidates	= ();
    my $start		= undef;

    for my $i (0 .. $#o)
    {
	if (defined $start)
	{
	    if ($o[$i] !~ m/^0+$/)
	    {
		push @candidates, [ $start, $i - $start ];
		$start = undef;
	    }
	}
	else
	{
	    $start = $i if $o[$i] =~ m/^0+$/;
	}
    }

    push @candidates, [$start, 8 - $start] if defined $start;

    my $l = (sort { $b->[1] <=> $a->[1] } @candidates)[0];

    return $addr unless defined $l;

    $addr = $l->[0] == 0 ? '' : join ':', @o[0 .. $l->[0] - 1];
    $addr .= '::';
    $addr .= join ':', @o[$l->[0] + $l->[1] .. $#o];
    $addr =~ s/(^|:)0{1,3}/$1/g;

    return $addr;
}



sub _compV6 ($) {
    my $ip = shift;
    return $ip unless my @candidates = $ip =~ /((?:^|:)0(?::0)+(?::|$))/g;
    my $longest = (sort { length($b) <=> length($a) } @candidates)[0];
    $ip =~ s/$longest/::/;
    return $ip;
}

sub short($) {
  my $addr = $_[0]->addr;
  if (! $_[0]->{isv6} && isIPv4($_[0]->{addr})) {
    my @o = split(/\./, $addr, 4);
    splice(@o, 1, 2) if $o[1] == 0 and $o[2] == 0;
    return join '.', @o;
  }
  return _compV6($addr);
}


sub canon($) {
  my $addr = $_[0]->addr;
  return $_[0]->{isv6} ? lc _compV6($addr) : $addr;
}


sub _splitplan {
  my($ip,@bits) = @_;
  my $addr = $ip->addr();
  my $isV6 = $ip->{isv6};
  unless (@bits) {
    $bits[0] = $isV6 ? 128 : 32;
  }
  my $basem = $ip->masklen();

  my(%nets,$dif);
  my $denom = 0;

  my($x,$maddr);
  foreach(@bits) {
    if (ref $_) {	# is a NetAddr::IP
      $x = $_->{isv6} ? $_->{addr} : $_->{addr} | V4mask;
      ($x,$maddr) = notcontiguous($x);
      return () if $x;	# spurious bits
      $_ = $isV6 ? $maddr : $maddr - 96;
    }
    elsif ( $_ =~ /^d+$/ ) {		# is a negative number of the form -nnnn
	;
    }
    elsif ($_ = NetAddr::IP->new($addr,$_,$isV6)) { # will be undefined if bad mask and will fall into oops!
      $_ = $_->masklen();
    }
    else {
      return ();	# oops!
    }
    $dif = $_ - $basem;			# for normalization
    return () if $dif < 0;		# overange nets not allowed
    return (\@bits,undef) unless ($dif || $#bits);	# return if original net = mask alone
    $denom = $dif if $dif > $denom;
    next if exists $nets{$_};
    $nets{$_} = $_ - $basem;		# for normalization
  }

  my %masks;					# calculate masks
  my $maskbase = $isV6 ? 128 : 32;
  foreach( keys %nets ) {
    $nets{$_} = 2 ** ($denom - $nets{$_});
    $masks{$_} = shiftleft(Ones, $maskbase - $_);
  }

  my @plan;
  my $idx = 0;
  $denom = 2 ** $denom;
  PLAN:
  while ($denom > 0) {				# make a net plan
    my $nexmask = ($idx < $#bits) ? $bits[$idx] : $bits[$#bits];
    ++$idx;
    unless (($denom -= $nets{$nexmask}) < 0) {
      return () if (push @plan, $nexmask) > $_netlimit;
      next;
    }
    $denom += $nets{$nexmask};			# restore mistake
  TRY:
    foreach (sort { $a <=> $b } keys %nets) {
      next TRY if $nexmask > $_;
      do {
	next TRY if $denom - $nets{$_} < 0;
	return () if (push @plan, $_) > $_netlimit;
	$denom -= $nets{$_};
      } while $denom;
    }
    die 'ERROR: miscalculated weights' if $denom;
  }
  return () if $idx < @bits;			# overrange original subnet request
  return (\@plan,\%masks);
}

sub _splitref {
  my $rev = shift;
  my($plan,$masks) = &_splitplan;
  croak("netmask error: overrange or spurious bits") unless defined $plan;
  my $net = $_[0]->network();
  return [$net] unless $masks;
  my $addr = $net->{addr};
  my $isV6 = $net->{isv6};
  my @plan = $rev ? reverse @$plan : @$plan;

  my @ret;
  while ($_ = shift @plan) {
    my $mask = $masks->{$_};
    push @ret, $net->_new($addr,$mask,$isV6);
    last unless @plan;
    $addr = (sub128($addr,$mask))[1];
  }
  return \@ret;
}


sub hostenum ($) {
    return @{$_[0]->hostenumref};
}


sub compactref($) {

  my @r;
  {
    my $unr  = [];
    my $args = $_[0];

    if (ref $_[0] eq __PACKAGE__ and ref $_[1] eq 'ARRAY') {
      # ->compactref(\@list)
      #
      $unr = [$_[0], @{$_[1]}]; # keeping structures intact
    }
    else {
      # Compact(@list) or ->compact(@list) or Compact(\@list)
      #
      $unr = $args;
    }

    return [] unless @$unr;

    foreach(@$unr) {
      $_->{addr} = $_->network->{addr};
    }

    @r = sort @$unr;
  }

  my $changed;
  do {
    $changed = 0;
    for(my $i=0; $i <= $#r -1;$i++) {
      if ($r[$i]->contains($r[$i +1])) {
        splice(@r,$i +1,1);
        ++$changed;
        --$i;
      }
      elsif ((notcontiguous($r[$i]->{mask}))[1] == (notcontiguous($r[$i +1]->{mask}))[1]) {		# masks the same
        if (hasbits($r[$i]->{addr} ^ $r[$i +1]->{addr})) {	# if not the same netblock
          my $upnet = $r[$i]->copy;
          $upnet->{mask} = shiftleft($upnet->{mask},1);
          if ($upnet->contains($r[$i +1])) {					# adjacent nets in next net up
      $r[$i] = $upnet;
      splice(@r,$i +1,1);
      ++$changed;
      --$i;
          }
        } else {									# identical nets
          splice(@r,$i +1,1);
          ++$changed;
          --$i;
        }
      }
    }
  } while $changed;
  return \@r;
}


sub coalesce
{
    my $masklen	= shift;
    if (ref $masklen && ref $masklen eq __PACKAGE__ ) {	# if called as a method
      push @_,$masklen;
      $masklen = shift;
    }

    my $number	= shift;

    # Addresses are at @_
    return [] unless @_;
    my %ret = ();
    my $type = $_[0]->{isv6};
    return [] unless defined $type;

    for my $ip (@_)
    {
	return [] unless $ip->{isv6} == $type;
	$type = $ip->{isv6};
	my $n = NetAddr::IP->new($ip->addr . '/' . $masklen)->network;
	if ($ip->masklen > $masklen)
	{
	    $ret{$n} += $ip->num + $NetAddr::IP::Lite::Old_nth;
	}
    }

    my @ret = ();

    # Add to @ret any arguments with netmasks longer than our argument
    for my $c (sort { $a->masklen <=> $b->masklen }
	       grep { $_->masklen <= $masklen } @_)
    {
	next if grep { $_->contains($c) } @ret;
	push @ret, $c->network;
    }

    # Now add to @ret all the subnets with more than $number hits
    for my $c (map { new NetAddr::IP $_ }
	       grep { $ret{$_} >= $number }
	       keys %ret)
    {
	next if grep { $_->contains($c) } @ret;
	push @ret, $c;
    }

    return \@ret;
}


sub re ($)
{
    return &re6 unless isIPv4($_[0]->{addr});
    my $self = shift->network;	# Insure a "zero" host part
    my ($addr, $mlen) = ($self->addr, $self->masklen);
    my @o = split('\.', $addr, 4);

    my $octet= '(?:[0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])';
    my @r = @o;
    my $d;


    if ($mlen != 32)
    {
	if ($mlen > 24)
	{
	     $d	= 2 ** (32 - $mlen) - 1;
	     $r[3] = '(?:' . join('|', ($o[3]..$o[3] + $d)) . ')';
	}
	else
	{
	    $r[3] = $octet;
	    if ($mlen > 16)
	    {
		$d = 2 ** (24 - $mlen) - 1;
		$r[2] = '(?:' . join('|', ($o[2]..$o[2] + $d)) . ')';
	    }
	    else
	    {
		$r[2] = $octet;
		if ($mlen > 8)
		{
		    $d = 2 ** (16 - $mlen) - 1;
		    $r[1] = '(?:' . join('|', ($o[1]..$o[1] + $d)) . ')';
		}
		else
		{
		    $r[1] = $octet;
		    if ($mlen > 0)
		    {
			$d = 2 ** (8 - $mlen) - 1;
			$r[0] = '(?:' . join('|', ($o[0] .. $o[0] + $d)) . ')';
		    }
		    else { $r[0] = $octet; }
		}
	    }
	}
    }

    ### no digit before nor after (look-behind, look-ahead)
    return "(?:(?<![0-9])$r[0]\\.$r[1]\\.$r[2]\\.$r[3](?![0-9]))";
}


sub re6($) {
  my @net = split('',sprintf("%04X%04X%04X%04X%04X%04X%04X%04X",unpack('n8',$_[0]->network->{addr})));
  my @brd = split('',sprintf("%04X%04X%04X%04X%04X%04X%04X%04X",unpack('n8',$_[0]->broadcast->{addr})));

  my @dig;

  foreach(0..$#net) {
    my $n = $net[$_];
    my $b = $brd[$_];
    my $m;
    if ($n.'' eq $b.'') {
      if ($n =~ /\d/) {
	push @dig, $n;
      } else {
	push @dig, '['.(lc $n).$n.']';
      }
    } else {
      my $n = $net[$_];
      my $b = $brd[$_];
      if ($n.'' eq 0 && $b =~ /F/) {
	push @dig, 'x';
      }
      elsif ($n =~ /\d/ && $b =~ /\d/) {
	push @dig, '['.$n.'-'.$b.']';
      }
      elsif ($n =~ /[A-F]/ && $b =~ /[A-F]/) {
	$n .= '-'.$b;
	push @dig, '['.(lc $n).$n.']';
      }
      elsif ($n =~ /\d/ && $b =~ /[A-F]/) {
	$m = ($n == 9) ? 9 : $n .'-9';
	if ($b =~ /A/) {
	  $m .= 'aA';
	} else {
	  $b = 'A-'. $b;
	  $m .= (lc $b). $b;
	}
	push @dig, '['.$m.']';
      }
      elsif ($n =~ /[A-F]/ && $b =~ /\d/) {
	if ($n =~ /A/) {
	  $m = 'aA';
	} else {
	  $n .= '-F';
	  $m = (lc $n).$n;
	}
	if ($b == 9) {
	  $m .= 9;
	} else {
	  $m .= $b .'-9';
	}
	push @dig, '['.$m.']';
      }
    }
  }
  my @grp;
  do {
    my $grp = join('',splice(@dig,0,4));
    if ($grp =~ /^(0+)/) {
      my $l = length($1);
      if ($l == 4) {
	$grp = '0{1,4}';
      } else {
	$grp =~ s/^${1}/0\{0,$l\}/;
      }
    }
    if ($grp =~ /(x+)$/) {
      my $l = length($1);
      if ($l == 4) {
	$grp = '[0-9a-fA-F]{1,4}';
      } else {
	$grp =~ s/x+/\[0\-9a\-fA\-F\]\{$l\}/;
      }
    }
    push @grp, $grp;
  } while @dig > 0;
  return '('. join(':',@grp) .')';
}

sub mod_version {
  return $VERSION;
  &Compact;			# suppress warnings about these symbols
  &Coalesce;
  &STORABLE_freeze;
  &STORABLE_thaw;
}


1;
