package Net::DNS::Resolver::Recurse;

use strict;
use warnings;
our $VERSION = (qw$Id: Recurse.pm 1856 2021-12-02 14:36:25Z willem $)[2];




use base qw(Net::DNS::Resolver);



my @hints;
my $root = [];

sub hints {
	shift;
	return @hints unless scalar @_;
	$root  = [];
	@hints = @_;
	return;
}



sub send {
	my $self = shift;
	my @conf = ( recurse => 0, udppacketsize => 1024 );	# RFC8109
	return bless( {persistent => {'.' => $root}, %$self, @conf}, ref($self) )->_send(@_);
}


sub query_dorecursion {			## historical
	my ($self) = @_;					# uncoverable pod
	$self->_deprecate('prefer  $resolver->send(...)');
	return &send;
}


sub _send {
	my $self  = shift;
	my $query = $self->_make_query_packet(@_);

	unless ( scalar(@$root) ) {
		$self->_diag("resolver priming query");
		$self->nameservers( scalar(@hints) ? @hints : $self->_hints );
		my $packet = $self->SUPER::send(qw(. NS));
		$self->_callback($packet);
		$self->_referral($packet);
		$root = $self->{persistent}->{'.'};
	}

	return $self->_recurse( $query, '.' );
}


sub _recurse {
	my ( $self, $query, $apex ) = @_;
	$self->_diag("using cached nameservers for $apex");
	my $nslist = $self->{persistent}->{$apex};
	$self->nameservers(@$nslist);
	$query->header->id(undef);
	my $reply = $self->SUPER::send($query);
	$self->_callback($reply);
	return unless $reply;
	my $qname = lc( ( $query->question )[0]->qname );
	my $zone  = $self->_referral($reply) || return $reply;
	return $reply if grep { lc( $_->owner ) eq $qname } $reply->answer;
	return $self->_recurse( $query, $zone );
}


sub _referral {
	my ( $self, $packet ) = @_;
	return unless $packet;
	my @auth = grep { $_->type eq 'NS' } $packet->answer, $packet->authority;
	return unless scalar(@auth);
	my $owner = lc( $auth[0]->owner );
	my $cache = $self->{persistent}->{$owner};
	return $owner if $cache && scalar(@$cache);
	my @addr = grep { $_->can('address') } $packet->additional;
	my @ip;
	my @ns = map { lc( $_->nsdname ) } @auth;

	foreach my $ns (@ns) {
		push @ip, map { $_->address } grep { $ns eq lc( $_->owner ) } @addr;
	}
	$self->_diag("resolving glue for $owner")   unless scalar(@ip);
	@ip = $self->nameservers( $ns[0], $ns[-1] ) unless scalar(@ip);
	$self->_diag("caching nameservers for $owner");
	$self->{persistent}->{$owner} = \@ip;
	return $owner;
}



sub callback {
	my $self = shift;

	( $self->{callback} ) = grep { ref($_) eq 'CODE' } @_;
	return;
}

sub _callback {
	my $callback = shift->{callback};
	$callback->(@_) if $callback;
	return;
}

sub recursion_callback {		## historical
	my ($self) = @_;					# uncoverable pod
	$self->_deprecate('prefer  $resolver->callback(...)');
	&callback;
	return;
}


1;

__END__



