package Net::DNS::RR::TKEY;

use strict;
use warnings;
our $VERSION = (qw$Id: TKEY.pm 1857 2021-12-07 13:38:02Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;

use Net::DNS::Parameters qw(:class :type);
use Net::DNS::DomainName;

use constant ANY  => classbyname qw(ANY);
use constant TKEY => typebyname qw(TKEY);


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $limit = $offset + $self->{rdlength};

	( $self->{algorithm}, $offset ) = Net::DNS::DomainName->decode(@_);

	@{$self}{qw(inception expiration mode error)} = unpack "\@$offset N2n2", $$data;
	$offset += 12;

	my $key_size = unpack "\@$offset n", $$data;
	$self->{key} = substr $$data, $offset + 2, $key_size;
	$offset += $key_size + 2;

	my $other_size = unpack "\@$offset n", $$data;
	$self->{other} = substr $$data, $offset + 2, $other_size;
	$offset += $other_size + 2;

	croak('corrupt TKEY data') unless $offset == $limit;	# more or less FUBAR
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	return '' unless defined $self->{algorithm};
	my $rdata = $self->{algorithm}->encode;

	$rdata .= pack 'N2n2', $self->inception, $self->expiration, $self->mode, $self->error;

	my $key = $self->key;					# RFC2930(2.7)
	$rdata .= pack 'na*', length $key, $key;

	my $other = $self->other;				# RFC2930(2.8)
	$rdata .= pack 'na*', length $other, $other;
	return $rdata;
}


sub class {				## overide RR method
	return 'ANY';
}

sub encode {				## overide RR method
	my $self = shift;

	my $owner = $self->{owner}->encode();
	my $rdata = eval { $self->_encode_rdata() } || '';
	return pack 'a* n2 N n a*', $owner, TKEY, ANY, 0, length $rdata, $rdata;
}


sub algorithm {
	my $self = shift;

	$self->{algorithm} = Net::DNS::DomainName->new(shift) if scalar @_;
	return $self->{algorithm} ? $self->{algorithm}->name : undef;
}


sub inception {
	my $self = shift;

	$self->{inception} = 0 + shift if scalar @_;
	return $self->{inception} || 0;
}


sub expiration {
	my $self = shift;

	$self->{expiration} = 0 + shift if scalar @_;
	return $self->{expiration} || 0;
}


sub mode {
	my $self = shift;

	$self->{mode} = 0 + shift if scalar @_;
	return $self->{mode} || 0;
}


sub error {
	my $self = shift;

	$self->{error} = 0 + shift if scalar @_;
	return $self->{error} || 0;
}


sub key {
	my $self = shift;

	$self->{key} = shift if scalar @_;
	return $self->{key} || "";
}


sub other {
	my $self = shift;

	$self->{other} = shift if scalar @_;
	return $self->{other} || "";
}


sub other_data { return &other; }				# uncoverable pod


1;
__END__


