package Net::DNS::RR::OPT;

use strict;
use warnings;
our $VERSION = (qw$Id: OPT.pm 1864 2022-04-14 15:18:49Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Carp;
use Net::DNS::Parameters qw(:rcode :ednsoption);

use constant CLASS_TTL_RDLENGTH => length pack 'n N n', (0) x 3;

use constant OPT => Net::DNS::Parameters::typebyname qw(OPT);

require Net::DNS::DomainName;
require Net::DNS::RR::A;
require Net::DNS::RR::AAAA;
require Net::DNS::Text;


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset ) = @_;

	my $index = $offset - CLASS_TTL_RDLENGTH;		# OPT redefines class and TTL fields
	@{$self}{qw(size rcode version flags)} = unpack "\@$index n C2 n", $$data;
	@{$self}{rcode} = @{$self}{rcode} << 4;
	delete @{$self}{qw(class ttl)};

	my $limit = $offset + $self->{rdlength} - 4;

	while ( $offset <= $limit ) {
		my ( $code, $length ) = unpack "\@$offset nn", $$data;
		my $value = unpack "\@$offset x4 a$length", $$data;
		$self->{option}{$code} = $value;
		$offset += $length + 4;
	}
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;

	my $option = $self->{option} || {};
	return join '', map { pack( 'nna*', $_, length $option->{$_}, $option->{$_} ) } keys %$option;
}


sub encode {				## overide RR method
	my $self = shift;

	my $data = $self->_encode_rdata;
	my $size = $self->size;
	my @xttl = ( $self->rcode >> 4, $self->version, $self->flags );
	return pack 'C n n C2n n a*', 0, OPT, $size, @xttl, length($data), $data;
}


sub string {				## overide RR method
	my $self = shift;

	my $edns   = $self->version;
	my $flags  = sprintf '%04x', $self->flags;
	my $rcode  = $self->rcode;
	my $size   = $self->size;
	my @option = map { join( "\n;;\t\t\t\t", $self->_format_option($_) ) } $self->options;
	my @format = join "\n;;\t\t", @option;

	$rcode = 0 if $rcode < 16;				# weird: 1 .. 15 not EDNS codes!!

	my $rc = exists( $self->{rdlength} ) && $rcode ? "$rcode + [4-bits]" : rcodebyval($rcode);

	$rc = 'BADVERS' if $rcode == 16;			# code 16 unambiguous here

	return <<"QQ";
;; EDNS version $edns
;;	flags:	$flags
;;	rcode:	$rc
;;	size:	$size
;;	option: @format
QQ
}


sub class {				## overide RR method
	my $self = shift;
	$self->_deprecate(qq[please use "size()"]);
	return $self->size(@_);
}

sub ttl {				## overide RR method
	my $self = shift;
	$self->_deprecate(qq[please use "flags()" or "rcode()"]);
	my @rcode = map { unpack( 'C',	 pack 'N', $_ ) } @_;
	my @flags = map { unpack( 'x2n', pack 'N', $_ ) } @_;
	return pack 'C2n', $self->rcode(@rcode), $self->version, $self->flags(@flags);
}


sub version {
	my $self = shift;

	$self->{version} = 0 + shift if scalar @_;
	return $self->{version} || 0;
}


sub size {
	my $self = shift;
	$self->{size} = shift if scalar @_;
	return ( $self->{size} || 0 ) > 512 ? $self->{size} : 512;
}


sub rcode {
	my $self = shift;
	return $self->{rcode} || 0 unless scalar @_;
	delete $self->{rdlength};				# (ab)used to signal incomplete value
	my $val = shift || 0;
	return $self->{rcode} = $val < 16 ? 0 : $val;		# discard non-EDNS rcodes 1 .. 15
}


sub flags {
	my $self = shift;
	$self->{flags} = shift if scalar @_;
	return $self->{flags} || 0;
}


sub options {
	my ($self) = @_;
	my $option = $self->{option} || {};
	my @option = sort { $a <=> $b } keys %$option;
	return @option;
}

sub option {
	my $self   = shift;
	my $number = ednsoptionbyname(shift);
	return $self->_get_option($number) unless scalar @_;
	return $self->_set_option( $number, @_ );
}



sub _format_option {
	my ( $self, $number ) = @_;
	my $option  = ednsoptionbyval($number);
	my $options = $self->{option} || {};
	my $payload = $options->{$number};
	return () unless defined $payload;
	my $package = join '::', __PACKAGE__, $option;
	$package =~ s/-/_/g;
	my $defined = length($payload) && $package->can('_image');
	my @element = $defined ? eval { $package->_image($payload) } : unpack 'H*', $payload;
	my $protect = pop(@element);
	return Net::DNS::RR::_wrap( "$option\t=> (", map( {"$_,"} @element ), "$protect )" );
}


sub _get_option {
	my ( $self, $number ) = @_;

	my $options = $self->{option} || {};
	my $payload = $options->{$number};
	return $payload unless wantarray;
	return ()	unless $payload;
	my $package = join '::', __PACKAGE__, ednsoptionbyval($number);
	$package =~ s/-/_/g;
	return ( 'OPTION-DATA' => $payload ) unless $package->can('_decompose');
	return eval { $package->_decompose($payload) };
}


sub _set_option {
	my ( $self, $number, $value, @etc ) = @_;

	my $options = $self->{option} ||= {};
	delete $options->{$number};
	return unless defined $value;
	if ( ref($value) || scalar(@etc) || $value !~ /\D/ ) {
		my @arg = ( $value, @etc );
		@arg = @$value if ref($value) eq 'ARRAY';
		@arg = %$value if ref($value) eq 'HASH';
		if ( $arg[0] eq 'OPTION-DATA' ) {
			$value = $arg[1];
		} else {
			my $option  = ednsoptionbyval($number);
			my $package = join '::', __PACKAGE__, $option;
			$package =~ s/-/_/g;
			if ( $package->can('_compose') ) {
				$value = $package->_compose(@arg);
			} elsif ( scalar(@etc) ) {
				croak "unable to compose option $option";
			}
		}
	}
	return $options->{$number} = $value;
}


sub _specified {
	my $self = shift;
	return scalar grep { $self->{$_} } qw(size flags rcode option);
}


package Net::DNS::RR::OPT::DAU;					# RFC6975

sub _compose {
	shift;
	return pack 'C*', @_;
}

sub _decompose {
	my @payload = unpack 'C*', $_[1];
	return @payload;
}

sub _image { return &_decompose; }


package Net::DNS::RR::OPT::DHU;					# RFC6975
our @ISA = qw(Net::DNS::RR::OPT::DAU);

package Net::DNS::RR::OPT::N3U;					# RFC6975
our @ISA = qw(Net::DNS::RR::OPT::DAU);


package Net::DNS::RR::OPT::CLIENT_SUBNET;			# RFC7871

my %family = qw(0 Net::DNS::RR::AAAA	1 Net::DNS::RR::A	2 Net::DNS::RR::AAAA);
my @field8 = qw(FAMILY SOURCE-PREFIX-LENGTH SCOPE-PREFIX-LENGTH ADDRESS);

sub _compose {
	my ( $class, %argument ) = ( map( ( $_ => 0 ), @field8 ), @_ );
	my $address = bless( {}, $family{$argument{FAMILY}} )->address( $argument{ADDRESS} );
	my $bitmask = $argument{'SOURCE-PREFIX-LENGTH'};
	return pack "a* B$bitmask", pack( 'nC2', @argument{@field8} ), unpack 'B*', $address;
}

sub _decompose {
	my %hash;
	@hash{@field8} = unpack 'nC2a*', $_[1];
	$hash{ADDRESS} = bless( {address => $hash{ADDRESS}}, $family{$hash{FAMILY}} )->address;
	my @payload = map( ( $_ => $hash{$_} ), @field8 );
	return @payload;
}

sub _image {
	my %hash  = &_decompose;
	my @image = map "$_ => $hash{$_}", @field8;
	return @image;
}


package Net::DNS::RR::OPT::EXPIRE;				# RFC7314

sub _compose {
	return pack 'N', pop @_;
}

sub _decompose {
	my @payload = ( 'EXPIRE-TIMER' => unpack 'N', $_[1] );
	return @payload;
}

sub _image { return join ' => ', &_decompose; }


package Net::DNS::RR::OPT::COOKIE;				# RFC7873

my @field10 = qw(VERSION RESERVED TIMESTAMP HASH);

sub _compose {
	my ( $class, %argument ) = ( VERSION => 1, RESERVED => '', @_ );
	return pack 'a8', $argument{'CLIENT-COOKIE'} if $argument{'CLIENT-COOKIE'};
	return pack 'Ca3Na*', map $_, @argument{@field10};
}

sub _decompose {
	my ( $class, $argument ) = @_;
	return ( 'CLIENT-COOKIE', $argument ) unless length($argument) > 8;
	my %hash;
	@hash{@field10} = unpack 'Ca3Na*', $argument;
	my @payload = map( ( $_ => $hash{$_} ), @field10 );
	return @payload;
}

sub _image {
	my %hash = &_decompose;
	return unpack 'H*', $hash{'CLIENT-COOKIE'} if $hash{'CLIENT-COOKIE'};
	for (qw(RESERVED HASH)) { $hash{$_} = unpack 'H*', $hash{$_} }
	my @image = map "$_ => $hash{$_}", @field10;
	return @image;
}


package Net::DNS::RR::OPT::TCP_KEEPALIVE;			# RFC7828

sub _compose {
	return pack 'n', pop @_;
}

sub _decompose {
	my @payload = ( 'TIMEOUT' => unpack 'n', $_[1] );
	return @payload;
}

sub _image { return join ' => ', &_decompose; }


package Net::DNS::RR::OPT::PADDING;				# RFC7830

sub _compose {
	my $size = pop @_;
	return pack "x$size";
}

sub _decompose {
	my @payload = ( 'OPTION-LENGTH' => length( $_[1] ) );
	return @payload;
}

sub _image { return join ' => ', &_decompose; }


package Net::DNS::RR::OPT::CHAIN;				# RFC7901

sub _compose {
	my ( $class, %argument ) = @_;
	my ($trust_point) = values %argument;
	return Net::DNS::DomainName->new($trust_point)->encode;
}

sub _decompose {
	my ( $class, $payload ) = @_;
	my $fqdn    = Net::DNS::DomainName->decode( \$payload )->string;
	my @payload = ( 'CLOSEST-TRUST-POINT' => $fqdn );
	return @payload;
}

sub _image { return join ' => ', &_decompose; }


package Net::DNS::RR::OPT::KEY_TAG;				# RFC8145

sub _compose {
	shift;
	return pack 'n*', @_;
}

sub _decompose {
	my @payload = unpack 'n*', $_[1];
	return @payload;
}

sub _image { return &_decompose; }


package Net::DNS::RR::OPT::EXTENDED_ERROR;			# RFC8914

my @field15 = qw(INFO-CODE EXTRA-TEXT);

sub _compose {
	my ( $class, %argument ) = ( 'INFO-CODE' => 0, 'EXTRA-TEXT' => '', @_ );
	my ( $code,  $text )	 = @argument{@field15};
	return pack 'na*', $code, Net::DNS::Text->new($text)->raw;
}

sub _decompose {
	my ( $code, $text ) = unpack 'na*', $_[1];
	my @payload = (
		'INFO-CODE'  => $code,
		'EXTRA-TEXT' => Net::DNS::Text->decode( \$text, 0, length $text )->string
		);
	return @payload;
}

sub _image {
	my %hash  = &_decompose;
	my @image = map "$_ => $hash{$_}", @field15;
}



1;
__END__


