package Net::DNS::RR::NAPTR;

use strict;
use warnings;
our $VERSION = (qw$Id: NAPTR.pm 1857 2021-12-07 13:38:02Z willem $)[2];

use base qw(Net::DNS::RR);



use integer;

use Net::DNS::DomainName;
use Net::DNS::Text;


sub _decode_rdata {			## decode rdata from wire-format octet string
	my $self = shift;
	my ( $data, $offset, @opaque ) = @_;

	@{$self}{qw(order preference)} = unpack "\@$offset n2", $$data;
	( $self->{flags},   $offset ) = Net::DNS::Text->decode( $data, $offset + 4 );
	( $self->{service}, $offset ) = Net::DNS::Text->decode( $data, $offset );
	( $self->{regexp},  $offset ) = Net::DNS::Text->decode( $data, $offset );
	$self->{replacement} = Net::DNS::DomainName2535->decode( $data, $offset, @opaque );
	return;
}


sub _encode_rdata {			## encode rdata as wire-format octet string
	my $self = shift;
	my ( $offset, @opaque ) = @_;

	my $rdata = pack 'n2', @{$self}{qw(order preference)};
	$rdata .= $self->{flags}->encode;
	$rdata .= $self->{service}->encode;
	$rdata .= $self->{regexp}->encode;
	$rdata .= $self->{replacement}->encode( $offset + length($rdata), @opaque );
	return $rdata;
}


sub _format_rdata {			## format rdata portion of RR string.
	my $self = shift;

	my @order = @{$self}{qw(order preference)};
	my @rdata = ( @order, map { $_->string } @{$self}{qw(flags service regexp replacement)} );
	return @rdata;
}


sub _parse_rdata {			## populate RR from rdata in argument list
	my $self = shift;

	foreach (qw(order preference flags service regexp replacement)) { $self->$_(shift) }
	return;
}


sub order {
	my $self = shift;

	$self->{order} = 0 + shift if scalar @_;
	return $self->{order} || 0;
}


sub preference {
	my $self = shift;

	$self->{preference} = 0 + shift if scalar @_;
	return $self->{preference} || 0;
}


sub flags {
	my $self = shift;

	$self->{flags} = Net::DNS::Text->new(shift) if scalar @_;
	return $self->{flags} ? $self->{flags}->value : undef;
}


sub service {
	my $self = shift;

	$self->{service} = Net::DNS::Text->new(shift) if scalar @_;
	return $self->{service} ? $self->{service}->value : undef;
}


sub regexp {
	my $self = shift;

	$self->{regexp} = Net::DNS::Text->new(shift) if scalar @_;
	return $self->{regexp} ? $self->{regexp}->value : undef;
}


sub replacement {
	my $self = shift;

	$self->{replacement} = Net::DNS::DomainName2535->new(shift) if scalar @_;
	return $self->{replacement} ? $self->{replacement}->name : undef;
}


my $function = sub {
	my ( $a, $b ) = ( $Net::DNS::a, $Net::DNS::b );
	return $a->{order} <=> $b->{order}
			|| $a->{preference} <=> $b->{preference};
};

__PACKAGE__->set_rrsort_func( 'order', $function );

__PACKAGE__->set_rrsort_func( 'default_sort', $function );


1;
__END__


