package Net::DNS::Question;

use strict;
use warnings;

our $VERSION = (qw$Id: Question.pm 1855 2021-11-26 11:33:48Z willem $)[2];




use integer;
use Carp;

use Net::DNS::Parameters qw(%classbyname %typebyname :class :type);
use Net::DNS::Domain;
use Net::DNS::DomainName;



sub new {
	my $self   = bless {}, shift;
	my $qname  = shift;
	my $qtype  = shift || '';
	my $qclass = shift || '';

	# tolerate (possibly unknown) type and class in zone file order
	unless ( exists $classbyname{$qclass} ) {
		( $qtype, $qclass ) = ( $qclass, $qtype ) if exists $classbyname{$qtype};
		( $qtype, $qclass ) = ( $qclass, $qtype ) if $qtype =~ /CLASS/;
	}
	unless ( exists $typebyname{$qtype} ) {
		( $qtype, $qclass ) = ( $qclass, $qtype ) if exists $typebyname{$qclass};
		( $qtype, $qclass ) = ( $qclass, $qtype ) if $qclass =~ /TYPE/;
	}

	# if argument is an IP address, do appropriate reverse lookup
	if ( defined $qname and $qname =~ m/:|\d$/ ) {
		if ( my $reverse = _dns_addr($qname) ) {
			$qname = $reverse;
			$qtype ||= 'PTR';
		}
	}

	$self->{qname}	= Net::DNS::DomainName1035->new($qname);
	$self->{qtype}	= typebyname( $qtype   || 'A' );
	$self->{qclass} = classbyname( $qclass || 'IN' );

	return $self;
}



use constant QFIXEDSZ => length pack 'n2', (0) x 2;

sub decode {
	my $self = bless {}, shift;
	my ( $data, $offset ) = @_;

	( $self->{qname}, $offset ) = Net::DNS::DomainName1035->decode(@_);

	my $next = $offset + QFIXEDSZ;
	die 'corrupt wire-format data' if length $$data < $next;
	@{$self}{qw(qtype qclass)} = unpack "\@$offset n2", $$data;

	return wantarray ? ( $self, $next ) : $self;
}



sub encode {
	my $self = shift;

	return pack 'a* n2', $self->{qname}->encode(@_), @{$self}{qw(qtype qclass)};
}



sub string {
	my $self = shift;

	return join "\t", $self->{qname}->string, $self->qclass, $self->qtype;
}



sub print {
	print &string, "\n";
	return;
}



sub name {
	my $self = shift;

	croak 'immutable object: argument invalid' if scalar @_;
	return $self->{qname}->xname;
}



sub qname {
	my $self = shift;

	croak 'immutable object: argument invalid' if scalar @_;
	return $self->{qname}->name;
}

sub zname { return &qname; }



sub type {
	my $self = shift;

	croak 'immutable object: argument invalid' if scalar @_;
	return typebyval( $self->{qtype} );
}

sub qtype { return &type; }
sub ztype { return &type; }



sub class {
	my $self = shift;

	croak 'immutable object: argument invalid' if scalar @_;
	return classbyval( $self->{qclass} );
}

sub qclass { return &class; }
sub zclass { return &class; }



sub _dns_addr {				## Map IP address into reverse lookup namespace
	local $_ = shift;

	# IP address must contain address characters only
	s/[%].+$//;						# discard RFC4007 scopeid
	return unless m#^[a-fA-F0-9:./]+$#;

	my ( $address, $pfxlen ) = split m#/#;

	# map IPv4 address to in-addr.arpa space
	if (m#^\d*[.\d]*\d(/\d+)?$#) {
		my @parse = split /\./, $address;
		$pfxlen = scalar(@parse) << 3 unless $pfxlen;
		my $last = $pfxlen > 24 ? 3 : ( $pfxlen - 1 ) >> 3;
		return join '.', reverse( ( @parse, (0) x 3 )[0 .. $last] ), 'in-addr.arpa.';
	}

	# map IPv6 address to ip6.arpa space
	return unless m#^[:\w]+:([.\w]*)(/\d+)?$#;
	my $rhs = $1 || '0';
	return _dns_addr($rhs) if m#^[:0]*:0*:[fF]{4}:[^:]+$#;	# IPv4
	$rhs = sprintf '%x%0.2x:%x%0.2x', map { $_ || 0 } split( /\./, $rhs, 4 ) if /\./;
	$address =~ s/:[^:]*$/:0$rhs/;
	my @parse = split /:/, ( reverse "0$address" ), 9;
	my @xpand = map { /./ ? $_ : ('0') x ( 9 - @parse ) } @parse;	 # expand ::
	$pfxlen = ( scalar(@xpand) << 4 ) unless $pfxlen;	# implicit length if unspecified
	my $len = $pfxlen > 124 ? 32 : ( $pfxlen + 3 ) >> 2;
	my $hex = pack 'A4' x 8, map { $_ . '000' } ('0') x ( 8 - @xpand ), @xpand;
	return join '.', split( //, substr( $hex, -$len ) ), 'ip6.arpa.';
}


1;
__END__



