package Net::DNS::Packet;

use strict;
use warnings;

our $VERSION = (qw$Id: Packet.pm 1865 2022-05-21 09:57:49Z willem $)[2];




use integer;
use Carp;

use Net::DNS::Parameters qw(:dsotype);
use constant UDPSZ => 512;

BEGIN {
	require Net::DNS::Header;
	require Net::DNS::Question;
	require Net::DNS::RR;
}



sub new {
	return &decode if ref $_[1];
	my $class = shift;

	my $self = bless {
		status	   => 0,
		question   => [],
		answer	   => [],
		authority  => [],
		additional => [],
		}, $class;

	$self->{question} = [Net::DNS::Question->new(@_)] if scalar @_;

	return $self;
}




use constant HEADER_LENGTH => length pack 'n6', (0) x 6;

sub decode {
	my $class = shift;					# uncoverable pod
	my $data  = shift;
	my $debug = shift || 0;

	my $offset = 0;
	my $self;
	eval {
		local $SIG{__DIE__};
		my $length = length $$data;
		die 'corrupt wire-format data' if $length < HEADER_LENGTH;

		# header section
		my ( $id, $status, @count ) = unpack 'n6', $$data;
		my ( $qd, $an, $ns, $ar ) = @count;

		$self = bless {
			id	   => $id,
			status	   => $status,
			count	   => [@count],
			question   => [],
			answer	   => [],
			authority  => [],
			additional => [],
			replysize  => $length
			}, $class;

		# question/zone section
		my $hash = {};
		my $record;
		$offset = HEADER_LENGTH;
		while ( $qd-- ) {
			( $record, $offset ) = decode Net::DNS::Question( $data, $offset, $hash );
			CORE::push( @{$self->{question}}, $record );
		}

		# RR sections
		while ( $an-- ) {
			( $record, $offset ) = decode Net::DNS::RR( $data, $offset, $hash );
			CORE::push( @{$self->{answer}}, $record );
		}

		while ( $ns-- ) {
			( $record, $offset ) = decode Net::DNS::RR( $data, $offset, $hash );
			CORE::push( @{$self->{authority}}, $record );
		}

		while ( $ar-- ) {
			( $record, $offset ) = decode Net::DNS::RR( $data, $offset, $hash );
			CORE::push( @{$self->{additional}}, $record );
		}

		return unless $offset == HEADER_LENGTH;
		return unless $self->header->opcode eq 'DSO';

		$self->{dso} = [];
		my $limit = $length - 4;
		while ( $offset < $limit ) {
			my ( $t, $l, $v ) = unpack "\@$offset n2a*", $$data;
			CORE::push( @{$self->{dso}}, [$t, substr( $v, 0, $l )] );
			$offset += ( $l + 4 );
		}
	};

	if ($debug) {
		local $@ = $@;
		print $@ if $@;
		$self->print if $self;
	}

	return wantarray ? ( $self, $offset ) : $self;
}



sub data {
	return &encode;
}

sub encode {
	my ( $self, $size ) = @_;				# uncoverable pod

	my $edns = $self->edns;					# EDNS support
	my @addl = grep { !$_->isa('Net::DNS::RR::OPT') } @{$self->{additional}};
	$self->{additional} = [$edns, @addl] if $edns->_specified;

	return $self->truncate($size) if $size;

	my @part = qw(question answer authority additional);
	my @size = map { scalar @{$self->{$_}} } @part;
	my $data = pack 'n6', $self->header->id, $self->{status}, @size;
	$self->{count} = [];

	my $hash = {};						# packet body
	foreach my $component ( map { @{$self->{$_}} } @part ) {
		$data .= $component->encode( length $data, $hash, $self );
	}

	return $data;
}



sub header {
	my $self = shift;
	return bless \$self, q(Net::DNS::Header);
}



sub edns {
	my $self = shift;
	my $link = \$self->{xedns};
	($$link) = grep { $_->isa(qw(Net::DNS::RR::OPT)) } @{$self->{additional}} unless $$link;
	$$link = Net::DNS::RR->new( type => 'OPT' ) unless $$link;
	return $$link;
}



sub reply {
	my $query  = shift;
	my $UDPmax = shift;
	my $qheadr = $query->header;
	croak 'erroneous qr flag in query packet' if $qheadr->qr;

	my $reply  = Net::DNS::Packet->new();
	my $header = $reply->header;
	$header->qr(1);						# reply with same id, opcode and question
	$header->id( $qheadr->id );
	$header->opcode( $qheadr->opcode );
	my @question = $query->question;
	$reply->{question} = [@question];

	$header->rcode('FORMERR');				# no RCODE considered sinful!

	$header->rd( $qheadr->rd );				# copy these flags into reply
	$header->cd( $qheadr->cd );

	return $reply unless grep { $_->isa('Net::DNS::RR::OPT') } @{$query->{additional}};

	my $edns = $reply->edns();
	CORE::push( @{$reply->{additional}}, $edns );
	$edns->size($UDPmax);
	return $reply;
}



sub question {
	my @qr = @{shift->{question}};
	return @qr;
}

sub zone { return &question }



sub answer {
	my @rr = @{shift->{answer}};
	return @rr;
}

sub pre		 { return &answer }
sub prerequisite { return &answer }



sub authority {
	my @rr = @{shift->{authority}};
	return @rr;
}

sub update { return &authority }



sub additional {
	my @rr = @{shift->{additional}};
	return @rr;
}



sub print {
	print &string;
	return;
}



sub string {
	my $self = shift;

	my $header = $self->header;
	my $opcode = $header->opcode;
	my $server = $self->{replyfrom};
	my $length = $self->{replysize};
	my $origin = $server ? ";; Response received from $server ($length octets)\n" : "";
	my @record = ( "$origin;; HEADER SECTION", $header->string );

	if ( $opcode eq 'DSO' ) {
		CORE::push( @record, ";; DSO SECTION" );
		foreach ( @{$self->{dso}} ) {
			my ( $t, $v ) = @$_;
			CORE::push( @record, pack 'a* A18 a*', ";;\t", dsotypebyval($t), unpack( 'H*', $v ) );
		}
		return join "\n", @record, "\n";
	}

	my @section  = $opcode eq 'UPDATE' ? qw(ZONE PREREQUISITE UPDATE) : qw(QUESTION ANSWER AUTHORITY);
	my @question = $self->question;
	my $qdcount  = scalar @question;
	my $qds	     = $qdcount != 1 ? 's' : '';
	CORE::push( @record, ";; $section[0] SECTION ($qdcount record$qds)", map { ';; ' . $_->string } @question );

	my @answer  = $self->answer;
	my $ancount = scalar @answer;
	my $ans	    = $ancount != 1 ? 's' : '';
	CORE::push( @record, "\n;; $section[1] SECTION ($ancount record$ans)", map { $_->string } @answer );

	my @authority = $self->authority;
	my $nscount   = scalar @authority;
	my $nss	      = $nscount != 1 ? 's' : '';
	CORE::push( @record, "\n;; $section[2] SECTION ($nscount record$nss)", map { $_->string } @authority );

	my @additional = $self->additional;
	my $arcount    = scalar @additional;
	my $ars	       = $arcount != 1 ? 's' : '';
	CORE::push( @record, "\n;; ADDITIONAL SECTION ($arcount record$ars)", map { $_->string } @additional );

	return join "\n", @record, "\n";
}



sub from {
	my $self = shift;

	$self->{replyfrom} = shift if scalar @_;
	return $self->{replyfrom};
}

sub answerfrom { return &from; }				# uncoverable pod



sub size {
	return shift->{replysize};
}

sub answersize { return &size; }				# uncoverable pod



sub push {
	my $self = shift;
	my $list = $self->_section(shift);
	return CORE::push( @$list, grep { ref($_) } @_ );
}



sub unique_push {
	my $self = shift;
	my $list = $self->_section(shift);
	my @rr	 = grep { ref($_) } @_;

	my %unique = map { ( bless( {%$_, ttl => 0}, ref $_ )->canonical => $_ ) } @rr, @$list;

	return scalar( @$list = values %unique );
}



sub pop {
	my $self = shift;
	my $list = $self->_section(shift);
	return CORE::pop(@$list);
}


my %_section = (			## section name abbreviation table
	'ans' => 'answer',
	'pre' => 'answer',
	'aut' => 'authority',
	'upd' => 'authority',
	'add' => 'additional'
	);

sub _section {				## returns array reference for section
	my $self = shift;
	my $name = shift;
	my $list = $_section{unpack 'a3', $name} || $name;
	return $self->{$list} ||= [];
}



sub sign_tsig {
	my $self = shift;

	return eval {
		local $SIG{__DIE__};
		require Net::DNS::RR::TSIG;
		my $tsig = Net::DNS::RR::TSIG->create(@_);
		$self->push( 'additional' => $tsig );
		return $tsig;
	} || return croak "$@\nTSIG: unable to sign packet";
}



sub verify {
	my $self = shift;

	my $sig = $self->sigrr;
	return $sig ? $sig->verify( $self, @_ ) : shift;
}

sub verifyerr {
	my $self = shift;

	my $sig = $self->sigrr;
	return $sig ? $sig->vrfyerrstr : 'not signed';
}



sub sign_sig0 {
	my $self = shift;
	my $karg = shift;

	return eval {
		local $SIG{__DIE__};

		my $sig0;
		if ( ref($karg) eq 'Net::DNS::RR::SIG' ) {
			$sig0 = $karg;

		} else {
			require Net::DNS::RR::SIG;
			$sig0 = Net::DNS::RR::SIG->create( '', $karg );
		}

		$self->push( 'additional' => $sig0 );
		return $sig0;
	} || return croak "$@\nSIG0: unable to sign packet";
}



sub sigrr {
	my $self = shift;

	my ($sig) = reverse $self->additional;
	return unless $sig;
	return $sig if $sig->type eq 'TSIG';
	return $sig if $sig->type eq 'SIG';
	return;
}






sub truncate {
	my $self = shift;
	my $size = shift || UDPSZ;

	my $sigrr = $self->sigrr;
	$size = UDPSZ unless $size > UDPSZ;
	$size -= $sigrr->_size if $sigrr;

	my $data = pack 'x' x HEADER_LENGTH;			# header placeholder
	$self->{count} = [];

	my $tc;
	my $hash = {};
	foreach my $section ( map { $self->{$_} } qw(question answer authority) ) {
		my @list;
		foreach my $item (@$section) {
			my $component = $item->encode( length $data, $hash );
			last if length($data) + length($component) > $size;
			last if $tc;
			$data .= $component;
			CORE::push @list, $item;
		}
		$tc++ if scalar(@list) < scalar(@$section);
		@$section = @list;
	}
	$self->header->tc(1) if $tc;				# only set if truncated here

	my %rrset;
	my @order;
	foreach my $item ( grep { ref($_) ne ref($sigrr) } $self->additional ) {
		my $name  = $item->{owner}->canonical;
		my $class = $item->{class} || 0;
		my $key	  = pack 'nna*', $class, $item->{type}, $name;
		CORE::push @order, $key unless $rrset{$key};
		CORE::push @{$rrset{$key}}, $item;
	}

	my @list;
	foreach my $key (@order) {
		my $component = '';
		my @item      = @{$rrset{$key}};
		foreach my $item (@item) {
			$component .= $item->encode( length $data, $hash );
		}
		last if length($data) + length($component) > $size;
		$data .= $component;
		CORE::push @list, @item;
	}

	if ($sigrr) {
		$data .= $sigrr->encode( length $data, $hash, $self );
		CORE::push @list, $sigrr;
	}
	$self->{'additional'} = \@list;

	my @part = qw(question answer authority additional);
	my @size = map { scalar @{$self->{$_}} } @part;
	return pack 'n6 a*', $self->header->id, $self->{status}, @size, substr( $data, HEADER_LENGTH );
}



sub dump {				## print internal data structure
	require Data::Dumper;					# uncoverable pod
	local $Data::Dumper::Maxdepth = $Data::Dumper::Maxdepth || 3;
	local $Data::Dumper::Sortkeys = $Data::Dumper::Sortkeys || 1;
	print Data::Dumper::Dumper(@_);
	return;
}


1;
__END__



