package Net::DNS::DomainName;

use strict;
use warnings;

our $VERSION = (qw$Id: DomainName.pm 1855 2021-11-26 11:33:48Z willem $)[2];




use base qw(Net::DNS::Domain);

use integer;
use Carp;



sub canonical {
	my @label = shift->_wire;
	for (@label) {
		tr /\101-\132/\141-\172/;
	}
	return join '', map { pack 'C a*', length($_), $_ } @label, '';
}



sub decode {
	my $label  = [];
	my $self   = bless {label => $label}, shift;
	my $buffer = shift;					# reference to data buffer
	my $offset = shift || 0;				# offset within buffer
	my $cache  = shift || {};				# hashed objectref by offset

	my $buflen = length $$buffer;
	my $index  = $offset;

	while ( $index < $buflen ) {
		my $header = unpack( "\@$index C", $$buffer )
				|| return wantarray ? ( $self, ++$index ) : $self;

		if ( $header < 0x40 ) {				# non-terminal label
			push @$label, substr( $$buffer, ++$index, $header );
			$index += $header;

		} elsif ( $header < 0xC0 ) {			# deprecated extended label types
			croak 'unimplemented label type';

		} else {					# compression pointer
			my $link = 0x3FFF & unpack( "\@$index n", $$buffer );
			croak 'corrupt compression pointer' unless $link < $offset;

			# uncoverable condition false
			$self->{origin} = $cache->{$link} ||= Net::DNS::DomainName->decode( $buffer, $link, $cache );
			return wantarray ? ( $self, $index + 2 ) : $self;
		}
	}
	croak 'corrupt wire-format data';
}



sub encode {
	return join '', map { pack 'C a*', length($_), $_ } shift->_wire, '';
}



package Net::DNS::DomainName1035;	## no critic ProhibitMultiplePackages
our @ISA = qw(Net::DNS::DomainName);


sub encode {
	my $self   = shift;
	my $offset = shift || 0;				# offset in data buffer
	my $hash   = shift || return $self->canonical;		# hashed offset by name

	my @labels = $self->_wire;
	my $data   = '';
	while (@labels) {
		my $name = join( '.', @labels );

		return $data . pack( 'n', 0xC000 | $hash->{$name} ) if defined $hash->{$name};

		my $label  = shift @labels;
		my $length = length $label;
		$data .= pack( 'C a*', $length, $label );

		next unless $offset < 0x4000;
		$hash->{$name} = $offset;
		$offset += 1 + $length;
	}
	return $data .= pack 'x';
}



package Net::DNS::DomainName2535;	## no critic ProhibitMultiplePackages
our @ISA = qw(Net::DNS::DomainName);


sub encode {
	my ( $self, $offset, $hash ) = @_;
	return $self->canonical unless defined $hash;
	return join '', map { pack 'C a*', length($_), $_ } $self->_wire, '';
}

1;
__END__




