package autodie::Scope::GuardStack;

use strict;
use warnings;

use autodie::Scope::Guard;

our $VERSION = '2.29'; # VERSION

my $H_KEY_STEM = __PACKAGE__ . '/guard';
my $COUNTER = 0;


sub new {
    my ($class) = @_;

    return bless([], $class);
}

sub push_hook {
    my ($self, $hook) = @_;
    my $h_key = $H_KEY_STEM . ($COUNTER++);
    my $size = @{$self};
    $^H{$h_key} = autodie::Scope::Guard->new(sub {
        # Pop the stack until we reach the right size
        # - this may seem weird, but it is to avoid relying
        #   on "destruction order" of keys in %^H.
        #
        # Example:
        #  {
        #     use autodie;  # hook 1
        #     no autodie;   # hook 2
        #     use autodie;  # hook 3
        #  }
        #
        #  Here we want call hook 3, then hook 2 and finally hook 1.
        #  Any other order could have undesired consequences.
        #
        #  Suppose hook 2 is destroyed first, it will pop hook 3 and
        #  then hook 2.  hook 3 will then be destroyed, but do nothing
        #  since its "frame" was already popped and finally hook 1
        #  will be popped and take its own frame with it.
        #
        #  We need to check that $self still exists since things can get weird
        #  during global destruction.
        $self->_pop_hook while $self && @{$self} > $size;
    });
    push(@{$self}, [$hook, $h_key]);
    return;
}

sub _pop_hook {
    my ($self) = @_;
    my ($hook, $key) = @{ pop(@{$self}) };
    my $ref = delete($^H{$key});
    $hook->();
    return;
}

sub DESTROY {
    my ($self) = @_;

    # To be honest, I suspect @{$self} will always be empty here due
    # to the subs in %^H having references to the stack (which would
    # keep the stack alive until those have been destroyed).  Anyhow,
    # it never hurt to be careful.
    $self->_pop_hook while @{$self};
    return;
}

1;

__END__

