package merlin

import (
	"bufio"
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"

	"github.com/nextdns/nextdns/config"
	"github.com/nextdns/nextdns/router/internal"
)

type Router struct {
	DNSMasqPath     string
	ListenPort      string
	ClientReporting bool
	savedParams     []string
	johnFork        bool
}

func New() (*Router, bool) {
	b, err := exec.Command("uname", "-o").Output()
	if err != nil || !strings.HasPrefix(string(b), "ASUSWRT-Merlin") {
		return nil, false
	}
	return &Router{
		DNSMasqPath: "/jffs/configs/dnsmasq.conf.add",
		ListenPort:  "5342",
		johnFork:    strings.HasPrefix(string(b), "ASUSWRT-Merlin-LTS"),
	}, true
}

func (r *Router) Configure(c *config.Config) error {
	c.Listen = "127.0.0.1:" + r.ListenPort
	r.ClientReporting = c.ReportClientInfo
	return nil
}

func nvKeys(vals []string) []string {
	keys := make([]string, 0, len(vals))
	for _, val := range vals {
		if idx := strings.IndexByte(val, '='); idx != -1 {
			keys = append(keys, val[:idx])
		} else {
			keys = append(keys, val)
		}
	}
	return keys
}

func (r *Router) Setup() error {
	if err := internal.WriteTemplate(r.DNSMasqPath, tmpl, r, 0644); err != nil {
		return err
	}
	// Configure the firmware so:
	// * Disable WAN DNS so dnsmasq does not set conflicting upstream servers.
	// * DNS rebinding is disabled, as DNS blocking uses 0.0.0.0 to block domains.
	//   The rebinding protection can be setup and enforced at NextDNS level.
	// * DNSSEC validation is disabled as when a DNSSEC supported domain is blocked,
	//   the validation will fail as blocking alters the response. NextDNS takes care
	//   of DNS validation for non blocked queries.
	// * DNS over TLS is disabled so stubby does not run for nothing.
	// * Reset the DHCP DNS settings to point back to the router (if was set to something else).
	nvRamReset := []string{
		"wan_dnsenable_x=0",
		"dns_norebind=0",
		"dnssec_enable=0",
		"dhcp_dns1_x=",
		"dhcp_dns2_x=",
	}
	if r.johnFork {
		nvRamReset = append(nvRamReset, "stubby_proxy=0")
	} else {
		nvRamReset = append(nvRamReset, "dnspriv_enable=0")
	}
	// Save nvram values so we can restore them.
	var err error
	if r.savedParams, err = internal.NVRAM(nvKeys(nvRamReset)...); err != nil {
		return err
	}
	if err := internal.SetNVRAM(nvRamReset...); err != nil {
		return err
	}

	if err := commentServer(r.dnsmasqFile(), false); err != nil {
		return err
	}

	// Restart dnsmasq service to apply changes.
	if err := exec.Command("service", "restart_dnsmasq").Run(); err != nil {
		return fmt.Errorf("service restart_dnsmasq: %v", err)
	}

	return nil
}

func (r *Router) Restore() error {
	// Restore previous settings.
	if err := internal.SetNVRAM(r.savedParams...); err != nil {
		return err
	}
	if err := commentServer(r.dnsmasqFile(), true); err != nil {
		return err
	}

	// Remove the custom dnsmasq config
	_ = os.Remove(r.DNSMasqPath)

	// Restart dnsmasq service to apply changes.
	if err := exec.Command("service", "restart_dnsmasq").Run(); err != nil {
		return fmt.Errorf("service restart_dnsmasq: %v", err)
	}
	return nil
}

func (r *Router) dnsmasqFile() string {
	if r.johnFork {
		return "/tmp/resolv.conf"
	}
	return "/tmp/resolv.dnsmasq"
}

func commentServer(file string, restore bool) error {
	const marker = "#nextdns#"
	f, err := os.Open(file)
	if err != nil {
		return err
	}
	s := bufio.NewScanner(f)
	var out []byte
	for s.Scan() {
		line := s.Text()
		if restore {
			if strings.HasPrefix(line, marker+"server=") {
				line = line[len(marker):]
			}
		} else {
			if strings.HasPrefix(line, "server=") && !strings.HasPrefix(line, "server=/") {
				out = append(out, marker...)
			}
		}
		out = append(out, line...)
		out = append(out, '\n')
	}
	if err := s.Err(); err != nil {
		return err
	}
	f.Close()
	return ioutil.WriteFile(file, out, 0644)
}

var tmpl = `# Configuration generated by NextDNS
no-resolv
server=127.0.0.1#{{.ListenPort}}
{{- if .ClientReporting}}
add-mac
add-subnet=32,128
{{- end}}
`
