package autodie::exception;
use 5.008;
use strict;
use warnings;
use Carp qw(croak);

our $VERSION = '2.26'; # VERSION: Generated by DZP::OurPkg:Version

our $DEBUG = 0;

use overload
    q{""} => "stringify",
    # Overload smart-match only if we're using 5.10 or up
    ($] >= 5.010 ? ('~~'  => "matches") : ()),
    fallback => 1
;

my $PACKAGE = __PACKAGE__;  # Useful to have a scalar for hash keys.



sub args        { return $_[0]->{$PACKAGE}{args}; }


sub function   { return $_[0]->{$PACKAGE}{function};  }


sub file        { return $_[0]->{$PACKAGE}{file};  }


sub package     { return $_[0]->{$PACKAGE}{package}; }


sub caller      { return $_[0]->{$PACKAGE}{caller};  }


sub line        { return $_[0]->{$PACKAGE}{line};  }



sub context     { return $_[0]->{$PACKAGE}{context} }


sub return      { return $_[0]->{$PACKAGE}{return} }



sub errno       { return $_[0]->{$PACKAGE}{errno}; }


sub eval_error { return $_[0]->{$PACKAGE}{eval_error}; }


{
    my (%cache);

    sub matches {
        my ($this, $that) = @_;

        # TODO - Handle references
        croak "UNIMPLEMENTED" if ref $that;

        my $sub = $this->function;

        if ($DEBUG) {
            my $sub2 = $this->function;
            warn "Smart-matching $that against $sub / $sub2\n";
        }

        # Direct subname match.
        return 1 if $that eq $sub;
        return 1 if $that !~ /:/ and "CORE::$that" eq $sub;
        return 0 if $that !~ /^:/;

        # Cached match / check tags.
        require Fatal;

        if (exists $cache{$sub}{$that}) {
            return $cache{$sub}{$that};
        }

        # This rather awful looking line checks to see if our sub is in the
        # list of expanded tags, caches it, and returns the result.

        return $cache{$sub}{$that} = grep { $_ eq $sub } @{ $this->_expand_tag($that) };
    }
}


sub _expand_tag {
    my ($this, @args) = @_;

    return Fatal->_expand_tag(@args);
}



my %formatter_of = (
    'CORE::close'    => \&_format_close,
    'CORE::open'     => \&_format_open,
    'CORE::dbmopen'  => \&_format_dbmopen,
    'CORE::flock'    => \&_format_flock,
    'CORE::read'     => \&_format_readwrite,
    'CORE::sysread'  => \&_format_readwrite,
    'CORE::syswrite' => \&_format_readwrite,
);


sub _format_flock {
    my ($this) = @_;

    require Fcntl;

    my $filehandle = $this->args->[0];
    my $raw_mode   = $this->args->[1];

    my $mode_type;
    my $lock_unlock;

    if ($raw_mode & Fcntl::LOCK_EX() ) {
        $lock_unlock = "lock";
        $mode_type = "for exclusive access";
    }
    elsif ($raw_mode & Fcntl::LOCK_SH() ) {
        $lock_unlock = "lock";
        $mode_type = "for shared access";
    }
    elsif ($raw_mode & Fcntl::LOCK_UN() ) {
        $lock_unlock = "unlock";
        $mode_type = "";
    }
    else {
        # I've got no idea what they're trying to do.
        $lock_unlock = "lock";
        $mode_type = "with mode $raw_mode";
    }

    my $cooked_filehandle;

    if ($filehandle and not ref $filehandle) {

        # A package filehandle with a name!

        $cooked_filehandle = " $filehandle";
    }
    else {
        # Otherwise we have a scalar filehandle.

        $cooked_filehandle = '';

    }

    local $! = $this->errno;

    return "Can't $lock_unlock filehandle$cooked_filehandle $mode_type: $!";

}

sub _format_dbmopen {
    my ($this) = @_;
    my @args   = @{$this->args};

    # TODO: Presently, $args flattens out the (usually empty) hash
    # which is passed as the first argument to dbmopen.  This is
    # a bug in our args handling code (taking a reference to it would
    # be better), but for the moment we'll just examine the end of
    # our arguments list for message formatting.

    my $mode = $args[-1];
    my $file = $args[-2];

    # If we have a mask, then display it in octal, not decimal.
    # We don't do this if it already looks octalish, or doesn't
    # look like a number.

    if ($mode =~ /^[^\D0]\d+$/) {
        $mode = sprintf("0%lo", $mode);
    };

    local $! = $this->errno;

    return "Can't dbmopen(%hash, '$file', $mode): '$!'";
}


sub _format_close {
    my ($this) = @_;
    my $close_arg = $this->args->[0];

    local $! = $this->errno;

    # If we've got an old-style filehandle, mention it.
    if ($close_arg and not ref $close_arg) {
        return "Can't close filehandle '$close_arg': '$!'";
    }

    # TODO - This will probably produce an ugly error.  Test and fix.
    return "Can't close($close_arg) filehandle: '$!'";

}

sub _format_readwrite {
    my ($this) = @_;
    my $call = $this->function;
    local $! = $this->errno;

    # Trim package name off dying sub for error messages.
    $call =~ s/.*:://;

    # These subs receive the following arguments (in order):
    #
    # * FILEHANDLE
    # * SCALAR (buffer, we do not want to write this)
    # * LENGTH (optional for syswrite)
    # * OFFSET (optional for all)
    my (@args) = @{$this->args};
    my $arg_name = $args[1];
    if (defined($arg_name)) {
        if (ref($arg_name)) {
            my $name = blessed($arg_name) || ref($arg_name);
            $arg_name = "<${name}>";
        } else {
            $arg_name = '<BUFFER>';
        }
    } else {
        $arg_name = '<UNDEF>';
    }
    $args[1] = $arg_name;

    return "Can't $call(" . join(q{, }, @args) . "): $!";
}


use constant _FORMAT_OPEN => "Can't open '%s' for %s: '%s'";

sub _format_open_with_mode {
    my ($this, $mode, $file, $error) = @_;

    my $wordy_mode;

    if    ($mode eq '<')  { $wordy_mode = 'reading';   }
    elsif ($mode eq '>')  { $wordy_mode = 'writing';   }
    elsif ($mode eq '>>') { $wordy_mode = 'appending'; }

    $file = '<undef>' if not defined $file;

    return sprintf _FORMAT_OPEN, $file, $wordy_mode, $error if $wordy_mode;

    Carp::confess("Internal autodie::exception error: Don't know how to format mode '$mode'.");

}

sub _format_open {
    my ($this) = @_;

    my @open_args = @{$this->args};

    # Use the default formatter for single-arg and many-arg open
    if (@open_args <= 1 or @open_args >= 4) {
        return $this->format_default;
    }

    # For two arg open, we have to extract the mode
    if (@open_args == 2) {
        my ($fh, $file) = @open_args;

        if (ref($fh) eq "GLOB") {
            $fh = '$fh';
        }

        my ($mode) = $file =~ m{
            ^\s*                # Spaces before mode
            (
                (?>             # Non-backtracking subexp.
                    <           # Reading
                    |>>?        # Writing/appending
                )
            )
            [^&]                # Not an ampersand (which means a dup)
        }x;

        if (not $mode) {
            # Maybe it's a 2-arg open without any mode at all?
            # Detect the most simple case for this, where our
            # file consists only of word characters.

            if ( $file =~ m{^\s*\w+\s*$} ) {
                $mode = '<'
            }
            else {
                # Otherwise, we've got no idea what's going on.
                # Use the default.
                return $this->format_default;
            }
        }

        # Localising $! means perl makes it a pretty error for us.
        local $! = $this->errno;

        return $this->_format_open_with_mode($mode, $file, $!);
    }

    # Here we must be using three arg open.

    my $file = $open_args[2];

    local $! = $this->errno;

    my $mode = $open_args[1];

    local $@;

    my $msg = eval { $this->_format_open_with_mode($mode, $file, $!); };

    return $msg if $msg;

    # Default message (for pipes and odd things)

    return "Can't open '$file' with mode '$open_args[1]': '$!'";
}


sub register {
    my ($class, $symbol, $handler) = @_;

    croak "Incorrect call to autodie::register" if @_ != 3;

    $formatter_of{$symbol} = $handler;

}



sub add_file_and_line {
    my ($this) = @_;

    return sprintf(" at %s line %d\n", $this->file, $this->line);
}


sub stringify {
    my ($this) = @_;

    my $call        =  $this->function;
    my $msg;

    if ($DEBUG) {
        my $dying_pkg   = $this->package;
        my $sub   = $this->function;
        my $caller = $this->caller;
        warn "Stringifing exception for $dying_pkg :: $sub / $caller / $call\n";
    }

    # TODO - This isn't using inheritance.  Should it?
    if ( my $sub = $formatter_of{$call} ) {
        $msg = $sub->($this) . $this->add_file_and_line;
    } else {
        $msg = $this->format_default . $this->add_file_and_line;
    }
    $msg .=  $this->{$PACKAGE}{_stack_trace}
        if $Carp::Verbose;

    return $msg;
}



sub format_default {
    my ($this) = @_;

    my $call        =  $this->function;

    local $! = $this->errno;

    # TODO: This is probably a good idea for CORE, is it
    # a good idea for other subs?

    # Trim package name off dying sub for error messages.
    $call =~ s/.*:://;

    # Walk through all our arguments, and...
    #
    #   * Replace undef with the word 'undef'
    #   * Replace globs with the string '$fh'
    #   * Quote all other args.

    my @args = @{ $this->args() };

    foreach my $arg (@args) {
       if    (not defined($arg))   { $arg = 'undef' }
       elsif (ref($arg) eq "GLOB") { $arg = '$fh'   }
       else                        { $arg = qq{'$arg'} }
    }

    # Format our beautiful error.

    return "Can't $call(".  join(q{, }, @args) . "): $!" ;

    # TODO - Handle user-defined errors from hash.

    # TODO - Handle default error messages.

}


sub new {
    my ($class, @args) = @_;

    my $this = {};

    bless($this,$class);

    # I'd love to use EVERY here, but it causes our code to die
    # because it wants to stringify our objects before they're
    # initialised, causing everything to explode.

    $this->_init(@args);

    return $this;
}

sub _init {

    my ($this, %args) = @_;

    # Capturing errno here is not necessarily reliable.
    my $original_errno = $!;

    our $init_called = 1;

    my $class = ref $this;

    # We're going to walk up our call stack, looking for the
    # first thing that doesn't look like our exception
    # code, autodie/Fatal, or some whacky eval.

    my ($package, $file, $line, $sub);

    my $depth = 0;

    while (1) {
        $depth++;

        ($package, $file, $line, $sub) = CORE::caller($depth);

        # Skip up the call stack until we find something outside
        # of the Fatal/autodie/eval space.

        next if $package->isa('Fatal');
        next if $package->isa($class);
        next if $package->isa(__PACKAGE__);

        # Anything with the 'autodie::skip' role wants us to skip it.
        # https://github.com/pjf/autodie/issues/15

        next if ($package->can('DOES') and $package->DOES('autodie::skip'));

        next if $file =~ /^\(eval\s\d+\)$/;

        last;

    }

    # We now have everything correct, *except* for our subroutine
    # name.  If it's __ANON__ or (eval), then we need to keep on
    # digging deeper into our stack to find the real name.  However we
    # don't update our other information, since that will be correct
    # for our current exception.

    my $first_guess_subroutine = $sub;

    while (defined $sub and $sub =~ /^\(eval\)$|::__ANON__$/) {
        $depth++;

        $sub = (CORE::caller($depth))[3];
    }

    # If we end up falling out the bottom of our stack, then our
    # __ANON__ guess is the best we can get.  This includes situations
    # where we were called from the top level of a program.

    if (not defined $sub) {
        $sub = $first_guess_subroutine;
    }

    $this->{$PACKAGE}{package} = $package;
    $this->{$PACKAGE}{file}    = $file;
    $this->{$PACKAGE}{line}    = $line;
    $this->{$PACKAGE}{caller}  = $sub;

    # Tranks to %Carp::CarpInternal all Fatal, autodie and
    # autodie::exception stack frames are filtered already, but our
    # nameless wrapper is still present, so strip that.

    my $trace = Carp::longmess();
    $trace =~ s/^\s*at \(eval[^\n]+\n//;

    # And if we see an __ANON__, then we'll replace that with the actual
    # name of our autodying function.

    my $short_func = $args{function};
    $short_func =~ s/^CORE:://;
    $trace =~ s/(\s*[\w:]+)__ANON__/$1$short_func/;

    # And now we just fill in all our attributes.

    $this->{$PACKAGE}{_stack_trace} = $trace;

    $this->{$PACKAGE}{errno}   = $args{errno} || 0;

    $this->{$PACKAGE}{context} = $args{context};
    $this->{$PACKAGE}{return}  = $args{return};
    $this->{$PACKAGE}{eval_error}  = $args{eval_error};

    $this->{$PACKAGE}{args}    = $args{args} || [];
    $this->{$PACKAGE}{function}= $args{function} or
              croak("$class->new() called without function arg");

    return $this;

}

1;

__END__

