package Device::USB::Device;

require 5.006;
use warnings;
use strict;
use Carp;

use constant MAX_BUFFER_SIZE => 256;


our $VERSION=0.36;



sub DESTROY
{
    my $self = shift;
    Device::USB::libusb_close( $self->{handle} ) if $self->{handle};
    return;
}

sub _assert_open
{
    my $self = shift;

    if(!defined $self->{handle})
    {
        $self->open() or croak "Cannot open device: $!\n";
    }
    return;
}


sub _make_descr_accessor
{
    my $name = shift;
    ## no critic (ProhibitStringyEval)

    return eval <<"EOE";
sub $name
        {
            my \$self = shift;
            return \$self->{descriptor}->{$name};
        }
EOE
}


sub filename
{
    my $self = shift;
    return $self->{filename};
}


sub config
{
    my $self = shift;
    return wantarray ? @{$self->{config}} : $self->{config};
}


sub configurations
{
    my $self = shift;
    return wantarray ? @{$self->{config}} : $self->{config};
}


sub get_configuration
{
    my $self = shift;
    my $index = shift || 0;
    return $self->configurations()->[$index];
}


_make_descr_accessor( 'bcdUSB' );
_make_descr_accessor( 'bDeviceClass' );
_make_descr_accessor( 'bDeviceSubClass' );
_make_descr_accessor( 'bDeviceProtocol' );
_make_descr_accessor( 'bMaxPacketSize0' );
_make_descr_accessor( 'idVendor' );
_make_descr_accessor( 'idProduct' );
_make_descr_accessor( 'bcdDevice' );
_make_descr_accessor( 'iManufacturer' );
_make_descr_accessor( 'iProduct' );
_make_descr_accessor( 'iSerialNumber' );
_make_descr_accessor( 'bNumConfigurations' );


sub manufacturer
{
    my $self = shift;

    return $self->get_string_simple( $self->iManufacturer() );
}


sub product
{
    my $self = shift;

    return $self->get_string_simple( $self->iProduct() );
}


sub serial_number
{
    my $self = shift;

    return $self->get_string_simple( $self->iSerialNumber() );
}


sub open  ## no critic (ProhibitBuiltinHomonyms)
{
    my $self = shift;
    Device::USB::libusb_close( $self->{handle} ) if $self->{handle};
    local $! = 0;
    $self->{handle} = Device::USB::libusb_open( $self->{device} );

    return 0 == $!;
}


sub set_configuration
{
    my $self = shift;
    my $configuration = shift;
    $self->_assert_open();

    return Device::USB::libusb_set_configuration( $self->{handle}, $configuration );
}


sub set_altinterface
{
    my $self = shift;
    my $alternate = shift;
    $self->_assert_open();

    return Device::USB::libusb_set_altinterface( $self->{handle}, $alternate );
}


sub clear_halt
{
    my $self = shift;
    my $ep = shift;
    $self->_assert_open();

    return Device::USB::libusb_clear_halt( $self->{handle}, $ep );
}


sub reset  ## no critic (ProhibitBuiltinHomonyms)
{
    my $self = shift;

    return 0 unless defined $self->{handle};

    my $ret = Device::USB::libusb_reset( $self->{handle} );
    delete $self->{handle} unless $ret;

    return $ret;
}


sub claim_interface
{
    my $self = shift;
    my $interface = shift;
    $self->_assert_open();

    return Device::USB::libusb_claim_interface( $self->{handle}, $interface );
}


sub release_interface
{
    my $self = shift;
    my $interface = shift;
    $self->_assert_open();

    return Device::USB::libusb_release_interface( $self->{handle}, $interface );
}


sub control_msg
{
    my $self = shift;
    ## no critic (RequireArgUnpacking)
    my ($requesttype, $request, $value, $index, $bytes, $size, $timeout) = @_;
    $bytes = q{} unless defined $bytes;
    $self->_assert_open();

    my ($retval, $out) = Device::USB::libusb_control_msg(
            $self->{handle}, $requesttype, $request, $value,
            $index, $bytes, $size, $timeout
       );
    # replace the input string in $bytes.
    $_[4] = $out if defined $_[4];
    return $retval;
}


sub get_string
{
    my $self = shift;
    my $index = shift;
    my $langid = shift;

    $self->_assert_open();

    my $buf = "\0" x MAX_BUFFER_SIZE;

    my $retlen = Device::USB::libusb_get_string(
        $self->{handle}, $index, $langid, $buf, MAX_BUFFER_SIZE
    );

    return if $retlen < 0;

    return substr( $buf, 0, $retlen );
}


sub get_string_simple
{
    my $self = shift;
    my $index = shift;
    $self->_assert_open();

    my $buf = "\0" x MAX_BUFFER_SIZE;

    my $retlen = Device::USB::libusb_get_string_simple(
        $self->{handle}, $index, $buf, MAX_BUFFER_SIZE
    );

    return if $retlen < 0;

    return substr( $buf, 0, $retlen );
}


sub get_descriptor
{
    my $self = shift;
    my $type = shift;
    my $index = shift;
    $self->_assert_open();

    my $buf = "\0" x MAX_BUFFER_SIZE;

    my $retlen = Device::USB::libusb_get_descriptor(
        $self->{handle}, $type, $index, $buf, MAX_BUFFER_SIZE
    );

    return if $retlen < 0;

    return substr( $buf, 0, $retlen );
}


sub get_descriptor_by_endpoint
{
    my $self = shift;
    my $ep = shift;
    my $type = shift;
    my $index = shift;

    $self->_assert_open();

    my $buf = "\0" x MAX_BUFFER_SIZE;

    my $retlen = Device::USB::libusb_get_descriptor_by_endpoint(
        $self->{handle}, $ep, $type, $index, $buf, MAX_BUFFER_SIZE
    );

    return if $retlen < 0;

    return substr( $buf, 0, $retlen );
}


sub bulk_read
{
    my $self = shift;
    # Don't change to shifts, I need to write back to $bytes.
    my ($ep, $bytes, $size, $timeout) = @_;
    $bytes = q{} unless defined $bytes;

    $self->_assert_open();

    if(length $bytes < $size)
    {
        $bytes .= "\0" x ($size - length $bytes);
    }

    my $retlen = Device::USB::libusb_bulk_read(
        $self->{handle}, $ep, $bytes, $size, $timeout
    );

    # stick back in the bytes parameter.
    $_[1] = substr( $bytes, 0, $retlen );

    return $retlen;
}


sub interrupt_read
{
    my $self = shift;
    # Don't change to shifts, I need to write back to $bytes.
    my ($ep, $bytes, $size, $timeout) = @_;
    $bytes = q{} unless defined $bytes;

    $self->_assert_open();

    if(length $bytes < $size)
    {
        $bytes .= "\0" x ($size - length $bytes);
    }

    my $retlen = Device::USB::libusb_interrupt_read(
        $self->{handle}, $ep, $bytes, $size, $timeout
    );

    # stick back in the bytes parameter.
    $_[1] = substr( $bytes, 0, $retlen );

    return $retlen;
}


sub bulk_write
{
    my $self = shift;
    my $ep = shift;
    my $bytes = shift;
    my $timeout = shift;

    $self->_assert_open();

    return Device::USB::libusb_bulk_write(
        $self->{handle}, $ep, $bytes, length $bytes, $timeout
    );
}


sub interrupt_write
{
    my $self = shift;
    my $ep = shift;
    my $bytes = shift;
    my $timeout = shift;

    $self->_assert_open();

    return Device::USB::libusb_interrupt_write(
        $self->{handle}, $ep, $bytes, length $bytes, $timeout
    );
}


sub get_driver_np
{
    my $self = shift;
    my $interface = shift;
    my $name = shift;

    $self->_assert_open();

    my $buf = "\0" x MAX_BUFFER_SIZE;

    my $retlen = Device::USB::libusb_get_driver_np(
        $self->{handle}, $interface, $buf, MAX_BUFFER_SIZE
    );

    return if $retlen < 0;

    return substr( $buf, 0, $retlen );
}



sub detach_kernel_driver_np
{
    my $self = shift;
    my $interface = shift;
    $self->_assert_open();

    return Device::USB::libusb_detach_kernel_driver_np(
        $self->{handle}, $interface
    );
}


1;
