package JoeDog::Config;

use strict;
use vars qw( $VERSION $LOCK_EX $LOCK_UN );

$VERSION = '1.2';
$LOCK_EX = 2;
$LOCK_UN = 8;

=head1 SYNOPSIS

  JoeDog::Config - Perl extension for parsing configuration files.

  use JoeDog::Config;
  my $config = new JoeDog::Config( filename );
  my @array  = $config->getColumn();
  my @array  = $config->getColumns( sep );
  my %hash   = $config->getHash( sep );

=head1 METHODS

=item $config = new JoeDog::Config( filename );

Constructor for Config. Returns a reference to a JoeDog::Config object.

=cut

sub new( $ )
{
  my ( $this, $file )   =  @_;
  my $class             =  ref( $this ) || $this;
  my $self              =  {};
  $self->{"file"}       =  $file;
  bless( $self, $class );
  return $self;
}

=item my @array = $config->getColumn();

Reads the configuration file referenced by the constructor and 
returns an array populated with all the lines in the file. 

  Example:
  my @array = $config->getColumn();
  for ( my $x = 0; $x <= $#array; $x++ ){
    print $array[$x][$y] . "\n";
  }

=cut 

sub getColumn
{
  my $this  = shift;
  my @lines;
  if( open ( FILE, "<" . $this->{"file"} )){
    flock( FILE, $LOCK_EX );
    @lines = grep(!/^\s*#/, grep(!/^$/, <FILE>));
    flock( FILE, $LOCK_UN );
    close( FILE );
  }
  return @lines;
}

=item my @array = $config->getColumns( sep );

Reads the configuration file referenced by the constructor and
returns a multi-dimensional array of columns which were separated
by 'sep' in the configuration file.

  Example:
  my @array = $config->getColumns( "," );
  for ( my $x = 0; $x <= $#array; $x++ ){
    for( my $y = 0; $y <= $#{$array[$x]}; $y++ ){
      print $array[$x][$y];
    }
    print "\n";
  } 

=cut

sub getColumns
{
  my $this  = shift;
  my ($sep) = @_;
  my @lines;
  my( @list, @cols );
  if( open ( FILE, "<" . $this->{"file"} )){
    flock( FILE, $LOCK_EX );
    @lines = grep(!/^\s*#/, grep(!/^$/, <FILE>));
    flock( FILE, $LOCK_UN );
    close( FILE );
  }
  my $x = 0;

  foreach my $thing ( @lines ){
    push @{$cols[$x]}, (split ( /$sep/, $thing) ); 
    $x ++;
  }

  for( my $x = 0; $x <= $#cols; $x++ ){
    for( my $y = 0; $y <= $#{$cols[$x]}; $y++ ){
      $cols[$x][$y] =~ s/^\s+//;
      $cols[$x][$y] =~ s/\s+$//; 
    }
  } 
  return @cols;
}

=item my %hash = $config->getHash( sep ); 

Reads a key value pair into memory, builds and returns a 
dynamic hash table where the key is the string before the 
separator and the value is the string after it.

  Example:
  $config = new JoeDog::Config("$ENV{HOME}/.popcheckrc")
    || warn( "couldn't open $ENV{HOME}/.popcheckrc $!" );
  %config = $config->getHash("="); 
  my $username = $config{'username'};  
  my $password = $config{'password'};

=cut

sub getHash
{
  my $this  = shift;
  my ($sep,$bad_programmer) = @_;
  my (%hash, $hash);
  my $lines = "";
  my( @list, @cols );
  my($left,$right);

  if( open ( FILE, "<" . $this->{"file"} )){
    flock( FILE, $LOCK_EX );
    while( <FILE> ){
      next if /^$/;
      next if /^\s*#/;
      $lines .= $_;
    }
    flock( FILE, $LOCK_UN );
    close( FILE );
  }

  foreach my $thing ( split( /\n/, $lines ) ){
    ($left,$right) = split( /$sep/, $thing );
    # Trim begnining and trailing whitespace
    $left=~s/^\s+//;
    $right=~s/^\s+//;
    $left=~s/\s+$//;
    $right=~s/\s+$//;
    $hash{$left}=$right;
  }

  return %hash;
} 

=item $my str = getFileName();

Returns the name of the file with with JoeDog::Config
was instantiated.

=cut 

sub getFileName
{
  my $this = shift;
  return $this->{"file"};
}

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

# Below is the stub of documentation for your module. You better edit it!

=head1 DESCRIPTION

JoeDog::Config is a module which reads a configuration file and
various data types, arrays, multi-dimensional arrays and hashes.

=head1 AUTHOR

Jeffrey Fulmer, jdfulmer@armstrong.com

=head1 SEE ALSO

perl(1).

=cut
