/*
Copyright (C) 2016, 2017 Siep Kroonenberg

This file is part of TLaunch.

TLaunch is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

TLaunch is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with TLaunch.  If not, see <http://www.gnu.org/licenses/>.
*/

/////////////////////////////////////////////////////////
// editor choice

// The editors dialogue is handled here, except that some of its
// functions are invoked from tlaunch.c and tla_parse.c.

// A globally defined button ed_control has as progid the global variable
// ed_pgid, which is set in initialize_editors in this file.

// Note on icon handles: even if an external icon is used multiple times,
// each widget will get its own handle.

#include "tla_utils.h"
#include "resource.h"
#include "tlaunch.h"

static wchar_t * edsel_class = L"edsel_class";

/////////////////////////////////////////////////////////
// data structures

typedef struct {
  Pgid * pgid;
  // per-editor GUI items for editor selection window
  HWND hw; // radio button; includes text label
  HWND hwic; // icon label
  HICON hic; // icon itself
  BOOL icon_from_resource; // if true, never call DestroyIcon
  UINT_PTR id;
} Editor;
static Editor editors[MAX_EDITORS + 1]; // one spare slot for custom editor
static short neds, current_edno, cust_edno, old_edno;
static Pgid old_cust_pgid; // for backing up the cust_pgid fields

///////////////////////////////////////////////////////
// a couple of candidates for tla_utils, except that we need them only here

static wchar_t * name_from_cmd( wchar_t * cmd ) {
  wchar_t * prog, * p, * name;
  if( !cmd ) return NULL;
  prog = prog_from_cmd( cmd ); // this is a purely syntactic operation
  if( !prog ) {
    return NULL;
  } else if( !prog[0] ) {
    FREE0( prog );
    return NULL;
  }
  p = wcsrchr( prog, L'.' );
  if( p ) *p = L'\0'; // now prog probably has no extension any more
  p = wcsrchr( prog, L'\\' );
  if( !p ) p = prog;
  else p++;
  name = tl_wcsncpy( p, wcslen( p ));
  name[0] = towupper( name[0] );
  FREE0( prog );
  return name;
} // name_from_cmd

static wchar_t * command_from_shell_cmd( wchar_t * shell_cmd ) {
  int l;
  wchar_t * prog, * command;
  if( !shell_cmd || !shell_cmd[0] ) return NULL;
  // assume shell_cmd was a registry value written by the launcher
  // as the result of the user selecting a custom editor.
  // Just lop off the final bit to get at the command.
  // Otherwise we would have to do a lot more parsing, or
  // store more information.
  prog = prog_from_cmd( shell_cmd );
  // first, existence check
  if( prog && prog[0] && FILE_EXISTS( prog )) {
    l = wcslen( shell_cmd );
    if( l > 5 && !wcscmp( shell_cmd + l - 5, L" \"%1\"" ))
      command = tl_wcsncpy( shell_cmd, l - 5 );
    else if( l > 8 && !wcscmp( shell_cmd + l - 8, L" \"%1\" %*" ))
      command = tl_wcsncpy( shell_cmd, l - 8 );
    else
      command = tl_wcscpy( shell_cmd );
    l = wcslen( command );
    // remove trailing spaces
    while( l > 0 && command[ l - 1 ] == L' ' ) command[ --l ] = L'\0';
    if ( !command[0] ) FREE0( command );
  } else {
    command = NULL;
  }
  FREE0( prog );
  return command;
} // command_from_shell_cmd

void free_pgid_data( Pgid * pgid ) {
  FREE0( pgid->command );
  FREE0( pgid->basename );
  FREE0( pgid->shell_cmd );
  FREE0( pgid->name );
  FREE0( pgid->iconspec );
} // free_pgid_data

void copy_pgid_data( Pgid * dest, Pgid * src ) {
  free_pgid_data( dest );
  dest->command = tl_wcscpy( src->command );
  dest->basename = tl_wcscpy( src->basename );
  dest->shell_cmd = tl_wcscpy( src->shell_cmd );
  dest->name = tl_wcscpy( src->name );
  dest->iconspec = tl_wcscpy( src->iconspec );
  dest->primary = src->primary;
  dest->path_prefix = src->path_prefix;
} // copy_pgid_data

/////////////////////////////////////////////////////////
// exported: default editor control in the main window

// menu item and/or button for default editor on main window
void update_mainwin_ed_control() {
  Control * ctrl = NULL;
  int i;

  // run through all controls and check for type E
  for( i=0; i<ncontrols; i++ ) {
    ctrl = controls[i];
    if( ctrl && ctrl->type && ctrl->type == L'E' ) {
      ctrl->pgid = ed_pgid;
      ctrl->enabled = ed_pgid->command ? TRUE : FALSE;
      EnableWindow( ctrl->hw, ctrl->enabled );

      if( i>=first_button && i<first_button + nbuttons ) {
        set_icon( ctrl );
      } // if button
    } // if control for default editor
  } // for controls
} // update_mainwin_ed_control

///////////////////////////////////////////////////////
// exported:
// build editors data structure and add custom editor, both as editor and
// as pgid; called by parse_ini when it starts with menu- and button specs
// i.e. when it is finished with filetypes.
// the GUI stuff here is limited to initializing the respective structure
// members to 0 / NULL, which is not really necessary for a static array
void initialize_editors() {

  int i;
  wchar_t * shcmd, * prog, * progi, *icf, *nm;
  wchar_t * exts, * p;
  Pgid * pgid;

  // collect editors from pgids.
  // remember first and default editor
  first_ed = NULL;
  for( i=0; i<nprogs; i++ ) {
    pgid = pgids[i];
    if( !pgid->extensions || !pgid->command ) continue;
    // tex among extensions?
    exts = pgid->extensions;
    p = wcsstr( exts, L".tex" );
    if( p && ( p == exts || *( p - 1 ) == L' ' ) &&
        (*( p + 4 ) == L'\0'  || *( p + 4 ) == L' ' )) {
      editors[neds].pgid = pgid;
      if( pgid->primary && !first_ed ) first_ed = pgid;
      // create name if there is none
      if( !pgid->name )
          pgid->name = name_from_cmd( pgid->command );
      if( ++neds >= MAX_EDITORS ) break;
    }
  }
  if( neds > 0 && !first_ed ) first_ed = editors[0].pgid;
  // add pgid and editor for custom editor. Note that
  // the pgids and editors arrays each have at least one spare slot
  cust_pgid = calloc_fatal( 1, sizeof( Pgid ));
  pgids[nprogs++] = cust_pgid;
  // in case the ini file defined CUSTOMED_PROGID:
  cust_pgid->progid = get_env( L"CUSTOMED_PROGID" );
  // otherwise use the #define-d value
  if( !cust_pgid->progid ) cust_pgid->progid = CUSTOMED_PROGID;
  cust_pgid->extensions = lx_exts;
  cust_edno = neds++;
  editors[cust_edno].pgid = cust_pgid;

  // a working cust_pgid in the registry gets priority
  get_assoc_data_progid( cust_pgid->progid, &cust_pgid->shell_cmd,
      &cust_pgid->iconspec, &cust_pgid->name );
  // command_from_shell_cmd also checks for existence of the program
  cust_pgid->command = command_from_shell_cmd( cust_pgid->shell_cmd );
  if( !cust_pgid->command ) {
    FREE0( cust_pgid->shell_cmd );
    FREE0( cust_pgid->iconspec );
    FREE0( cust_pgid->name );
  } else {
    if( !cust_pgid->name )
      cust_pgid->name = name_from_cmd( cust_pgid->command );
    cust_pgid->basename = get_basename( cust_pgid->command );
    // no need to synthesize an iconspec;
    // if necessary, an icon may be retrieved directly from command
  }

  // currently configured editor
  current_edno = -1;
  get_assoc_data_ext( L".tex", NULL, &shcmd, &icf, &nm );
  if( shcmd ) {
    prog = prog_from_cmd( shcmd );
    if( FILE_EXISTS( prog )) {
      // is this a configured editor?
      for( i=0; i<neds; i++ ) {
        pgid = editors[i].pgid;
        if( !pgid->command ) continue; // might only happen with cust_edno
        progi = prog_from_cmd( pgid->command );
        if( same_file( prog, progi )) {
          current_edno = i;
          FREE0( progi );
          FREE0( shcmd );
          FREE0( nm );
          FREE0( icf );
          break;
        } else {
          FREE0( progi );
        }
      } // end for
      if( current_edno < 0 ) { // prog exists, but not as configured editor
        current_edno = cust_edno;
        cust_pgid->shell_cmd = shcmd;
        cust_pgid->command = command_from_shell_cmd( shcmd );
        // cust_pgid->progid already set
        // use specified icon, or create default
        if( icf ) cust_pgid->iconspec = icf;
        if( nm ) cust_pgid->name = nm;
        else cust_pgid->name = name_from_cmd( shcmd );
        cust_pgid->basename = get_basename( cust_pgid->command );
      }
      FREE0( prog );
    } else { // prog does not exist: ignore
      FREE0( prog );
      FREE0( shcmd );
      FREE0( icf );
      FREE0( nm );
      current_edno = 0;
    }
  } else { // nothing useful configured
    FREE0( icf );
    FREE0( nm );
    current_edno = 0;
  }
  ed_pgid = editors[current_edno].pgid; // may be unconfigured cust_pgid

  // initialize the GUI members of the Editor structures to NULL/0
  // (not really necessary, since the editors array is static)
  for( i=0; i<neds; i++ ) {
    editors[i].hw = NULL;
    editors[i].hwic = NULL;
    editors[i].hic = NULL;
    editors[i].id = IDC_STATIC;
    editors[i].icon_from_resource = FALSE;
  }

  ed_initdone = TRUE;
} // initialize_editors

// select new editor via file browser an update cust_pgid accordingly.
// The editor-selection window is updated elsewhere.
static BOOL pick_custom_ed( HWND hwnd ) {
  OPENFILENAME ofn;
  BOOL chosen;
  ZeroMemory( &ofn, sizeof( OPENFILENAME ));
  ofn.lStructSize = sizeof( OPENFILENAME );
  ofn.hwndOwner = hwnd;
  ofn.lpstrFilter =
      L"Programs (*.exe, *.bat, *.cmd)\0*.exe;*.bat;*.cmd)\0\0";
  ofn.nFilterIndex = 1;
  ofn.lpstrFile = calloc_fatal( MAX_PATH, sizeof( wchar_t ));
  ofn.nMaxFile = MAX_PATH;
  ofn.lpstrTitle = L"Select custom editor";
  ofn.Flags = OFN_FILEMUSTEXIST;
  ofn.lpstrDefExt = L"exe";
  chosen = GetOpenFileName( &ofn );
  if( chosen ) {
    // we already made a backup of the old data
    current_edno = cust_edno;
    free_pgid_data( cust_pgid );
    if( wcschr( ofn.lpstrFile, L' ' ))
      cust_pgid->command = tl_wcsnconcat( 3, L"\"", ofn.lpstrFile, L"\"" );
    else
      cust_pgid->command = tl_wcscpy( ofn.lpstrFile );
    cust_pgid->basename = get_basename( ofn.lpstrFile );
    cust_pgid->shell_cmd = tl_wcsnconcat( 2, cust_pgid->command,
        L" \"%1\"" );
    cust_pgid->iconspec = tl_wcsnconcat( 2, ofn.lpstrFile, L",0" );
    cust_pgid->name = name_from_cmd( ofn.lpstrFile );
    cust_pgid->primary = 1;
    cust_pgid->path_prefix = 0;
    // The path prefix registry value is tied to the basename rather than
    // to the full path, and there may be other files with this basename.
    // Nevertheless, we register the basename even though it might already
    // be registered for another file, because otherwise
    // user choice association may not work right.
    // But defining a path prefix too is going a bit far.
  }
  FREE0( ofn.lpstrFile );
  return chosen;
} // pick_custom_ed

/////////////////////////////////////////////////////////
// building GUI

static HWND add_editsel_child( wchar_t * cls, wchar_t * text, DWORD style,
    DWORD x, DWORD y, DWORD w, DWORD h, HWND parent, LONG id ) {
  HWND hC;
  hC = CreateWindow( cls, text,
      WS_CHILD | WS_VISIBLE | style, x, y, w, h,
      parent, (HMENU)id, GetModuleHandle( NULL ), NULL );
  if( !( hC )) tldie( parent, L"Failure to create child" );
  SendMessage( hC, WM_SETFONT, ( WPARAM )hfnt, TRUE );
  return hC;
} // add_editsel_child

static void update_ed_item( short ied ) {
  // text and icon;
  // the icon label and radio button have already been created
  wchar_t * iconfile = NULL;
  UINT iconindex = 0;
  Editor * ed = NULL;
  Pgid * pgid = NULL;

  ed = editors + ied; // *( editors[ied] )
  pgid = ed->pgid;
  // text
  if( pgid->name )
    SetWindowText( ed->hw, pgid->name );
  else
    SetWindowText( ed->hw, L"(No custom defined)" );
  // icon: discard previous icon, if any
  if( ed->hic ) {
    SendMessage( ed->hwic, STM_SETIMAGE, ( WPARAM )IMAGE_ICON,
        ( LPARAM )NULL );
    if( !ed->icon_from_resource ) DestroyIcon( ed->hic );
    ed->hic = NULL;
  }
  // icon: find new one
  if( pgid->iconspec ) {
    parse_iconspec( pgid->iconspec, &iconfile, &iconindex );
    if( iconfile ) {
      ed->hic = ExtractIcon( hInst, iconfile, iconindex );
      FREE0( iconfile );
    }
  }
  if( ed->hic ) {
    ed->icon_from_resource = FALSE;
  } else {
    ed->hic = LoadIcon( hInst, MAKEINTRESOURCE( IDI_DEFLT ));
    ed->icon_from_resource = TRUE;
  }
  // icon: apply it
  SendMessage( ed->hwic, STM_SETIMAGE, ( WPARAM )IMAGE_ICON,
      ( LPARAM )ed->hic );

} // update_ed_item

// update default editor and states of radio buttons
static void update_selected ( ) {
  int i;
  ed_pgid = editors[current_edno].pgid;
  for( i=0; i < neds; i++ ) {
    if( i == current_edno ) {
      SendMessage( editors[i].hw, BM_SETCHECK, 1, 0 );
      ed_pgid = editors[i].pgid;
    } else {
      SendMessage( editors[i].hw, BM_SETCHECK, 0, 0 );
    }
  } // end for
  if( !cust_pgid || !cust_pgid->command || neds == 1 ) {
    EnableWindow( editors[cust_edno].hw, FALSE );
  } else {
    EnableWindow( editors[cust_edno].hw, TRUE );
  }
} // update_selected

// exported; this is the function that invokes the editor choice dialog
void ed_sel_tl( HWND hwnd ) {
  HWND hEds;
  hEds = CreateWindowEx(
      WS_EX_TOPMOST,
      edsel_class,
      L"Editor Choice",
      WS_POPUP | WS_OVERLAPPED | WS_CAPTION | WS_THICKFRAME |
          WS_VISIBLE,
      CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
      hMain, NULL, NULL, NULL );
  if ( hEds ) {
    // editor data structures already created
    // back up some current values before proceeding
    old_edno = current_edno;
    copy_pgid_data( &old_cust_pgid, cust_pgid );

    ShowWindow( hEds, SW_SHOW);
    UpdateWindow( hEds);
    EnableWindow( hwnd, FALSE );
  } else {
    tlwarn( hwnd, L"Creation Editor Window Failed" );
  }
} // ed_sel_tl

static void eds_finishup ( DWORD id ) {
  int i;
  EnableWindow( hMain, TRUE );
  // no need to deallocate text labels: they are either constants
  // or members of persistent data structures.
  // But we do need to deallocate icon handles
  for( i=0; i<neds; i++ ) {
    if( editors[i].hic ) {
      // make sure the icon is no longer in use
      SendMessage( editors[i].hwic, BM_SETIMAGE,
          ( WPARAM )IMAGE_ICON, ( LPARAM )NULL );
      if( !editors[i].icon_from_resource ) DestroyIcon( editors[i].hic );
      editors[i].hic = NULL;
      editors[i].icon_from_resource = FALSE;
    } // if
  } // for
  if( id == IDOK ) {
    free_pgid_data( &old_cust_pgid );
    if( assoc_mode ) {
      for( i=0; i<NEXTS; i++ )
          set_assoc_abs( latex_exts[i], ed_pgid->progid );
    }
    register_progid( cust_pgid->progid, cust_pgid->shell_cmd, REG_SZ,
        cust_pgid->iconspec, cust_pgid->name );
    register_prog( cust_pgid->command, NULL );
    update_mainwin_ed_control();
  } else {
    current_edno = old_edno;
    copy_pgid_data( cust_pgid, &old_cust_pgid );
    free_pgid_data( &old_cust_pgid );
  }
} // eds_finishup

LRESULT CALLBACK EdsProc( HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam ) {

  int i, j;
  HWND hOk, hCancel, hCustom;
  UINT_PTR id_next;
  Pgid * pgid;
  switch( msg ) {
    case WM_CREATE: {
      int y, w, h;
      RECT rect = { -1, -1, -1, -1 };
      int lh = cy / 2 + ( iconsize > ( 3 * cy / 2 ) ? iconsize : 3 * cy / 2 );

      if( GetWindowRect( hMain, &rect ))
        SetWindowPos( hwnd, HWND_TOP, rect.left + 30, rect.top + 30,
            40 * cx, (( neds + 3 ) * lh + cy ),
            SWP_NOACTIVATE | SWP_NOZORDER );
      else
        SetWindowPos( hwnd, HWND_TOP, 0, 0,
            60 * cx, (( neds + 3 ) * lh + cy ),
            SWP_NOACTIVATE | SWP_NOMOVE | SWP_NOZORDER );

      // create some radio buttons with static icon labels
      y = cy / 2;
      w = 24 * cx;
      h = lh;
      id_next = ID_FIRST_EDS_CHILD;
      for( i=0; i<neds; i++ ) {
        editors[i].id = id_next++;
        editors[i].hw = add_editsel_child( L"button", NULL, BS_RADIOBUTTON,
            4 * cx + iconsize, y, w, h, hwnd, editors[i].id );
        // also create a static item with an editor icon
        editors[i].hwic =  add_editsel_child( L"static", NULL, SS_ICON,
            2 * cx, y, iconsize, iconsize, hwnd, IDC_STATIC );
        update_ed_item( i );
        y += lh;
      } // for
        update_selected( );
        y += cy;

      // custom selection button
      hCustom = add_editsel_child( L"button", L"Custom editor...",
          BS_PUSHBUTTON | WS_TABSTOP, 2 * cx, y, w, 2 * cy, hwnd,
          ID_CUSTOM );
      if( !hCustom ) tldie( hwnd, L"Failure to create button" );
      y += 5 * cy / 2;

      // ok button
      hOk = add_editsel_child( L"button", L"Ok",
          BS_PUSHBUTTON | WS_TABSTOP, 2 * cx, y, 6 * cx, 2 * cy, hwnd,
          IDOK );
      if( !hOk ) tldie( hwnd, L"Failure to create button" );

      // cancel button
      hCancel = add_editsel_child( L"button", L"Cancel",
          BS_PUSHBUTTON | WS_TABSTOP, 10 * cx, y, 10 * cx, 2 * cy, hwnd,
          IDCANCEL );
      if( !hCancel ) tldie( hwnd, L"Failure to create button" );

      return 0;
    } // end WM_CREATE

    case WM_CLOSE: {
      DestroyWindow( hwnd );
      return 0;
    }
    case WM_COMMAND: {
      DWORD id = LOWORD( wParam );
      // editor radio button?
      if( id >= editors[0].id && id <= editors[ neds - 1 ].id ) {
        for( j=0; j<neds; j++ ) if( id == editors[j].id ) {
          pgid = editors[j].pgid;
          if( pgid->command ) {
            current_edno = j;
            break;
          } // if pgid->command
        } // for j if id
        update_selected( );
        return 0;
      } else if( id == ID_CUSTOM ) {
        // file browser
        if( !pick_custom_ed( hwnd )) return 0;
        if( cust_pgid->command ) {
          current_edno = cust_edno;
          ed_pgid = cust_pgid;
        }
        update_ed_item( cust_edno );
        update_selected( );
        return 0;
      } else if( id == IDOK ) {
        free_pgid_data( &old_cust_pgid );
        eds_finishup( id );
        PostMessage( hwnd, WM_CLOSE, 0, 0 );
        return 0;
      } else if( id == IDCANCEL ) {
        // restore from backup
        current_edno = old_edno;
        copy_pgid_data( cust_pgid, &old_cust_pgid );
        // this includes freeing new strings
        free_pgid_data( &old_cust_pgid );
        eds_finishup( id );
        PostMessage( hwnd, WM_CLOSE, 0, 0 );
        return 0;
      }
      break;
    } // end WM_COMMAND
  } // switch msg
  return DefWindowProc( hwnd, msg, wParam, lParam );
} // EdsProc

void register_eds_wnd_class( ) {
  WNDCLASSEX wedselc;

  wedselc.cbSize = sizeof(WNDCLASSEX);
  wedselc.style = 0; // set by CreateWindow
  wedselc.lpfnWndProc = EdsProc;
  wedselc.cbClsExtra = 0;
  wedselc.cbWndExtra = 0;
  wedselc.hInstance = hInst;
  wedselc.hIcon =
      LoadIcon( GetModuleHandle( NULL ), MAKEINTRESOURCE( IDI_TL ));
  wedselc.hCursor = NULL;
  wedselc.hbrBackground = ( HBRUSH )( COLOR_BTNFACE + 1 );
  wedselc.lpszMenuName = NULL;
  wedselc.lpszClassName = edsel_class;
  wedselc.hIconSm = NULL;

  if(!RegisterClassEx( &wedselc ))
      tldie( NULL, L"Edsel Window Registration Failed!" );
}
