/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.paimon.flink.action;

import org.apache.paimon.utils.TimeUtils;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;

/** Factory to create {@link ReplaceTagAction} or {@link ReplaceTagAction}. */
public abstract class CreateOrReplaceTagActionFactory implements ActionFactory {

    private static final String TAG_NAME = "tag_name";
    private static final String SNAPSHOT = "snapshot";
    private static final String TIME_RETAINED = "time_retained";

    @Override
    public Optional<Action> create(MultipleParameterToolAdapter params) {
        Map<String, String> catalogConfig = catalogConfigMap(params);
        String tagName = params.getRequired(TAG_NAME);

        Long snapshot = null;
        if (params.has(SNAPSHOT)) {
            snapshot = Long.parseLong(params.get(SNAPSHOT));
        }

        Duration timeRetained = null;
        if (params.has(TIME_RETAINED)) {
            timeRetained = TimeUtils.parseDuration(params.get(TIME_RETAINED));
        }

        return Optional.of(
                createOrReplaceTagAction(
                        params.getRequired(DATABASE),
                        params.getRequired(TABLE),
                        catalogConfig,
                        tagName,
                        snapshot,
                        timeRetained));
    }

    abstract Action createOrReplaceTagAction(
            String database,
            String table,
            Map<String, String> catalogConfig,
            String tagName,
            Long snapshot,
            Duration timeRetained);
}
