# -*- coding: utf-8 -*-
# Description:
# Setup for the OSP nmap Server
#
# Authors:
# Jan-Oliver Wagner <Jan-Oliver.Wagner@greenbone.net>
#
# Copyright:
# Copyright (C) 2015 Greenbone Networks GmbH
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.

from ospd.ospd import OSPDaemon
from ospd.misc import main as daemon_main
from ospd_nmap import __version__
import xml.etree.ElementTree as ET
import subprocess

OSPD_DESC = """
This scanner runs the tool 'nmap' to scan the target hosts.

This tool is availble for most operating systems and identifies open ports,
probes the services, operating systems and even can run more sophisticated detection
routines.

For more details about nmap see the nmap homepage:
http://nmap.org/
 
The current version of ospd-nmap is a very simple one, only retrieving
open tcp ports via a scan with nmap's default settings.
"""


class OSPDnmap(OSPDaemon):

    """ Class for ospd-nmap daemon. """

    def __init__(self, certfile, keyfile, cafile):
        """ Initializes the ospd-nmap daemon's internal data. """
        super(OSPDnmap, self).__init__(certfile=certfile, keyfile=keyfile,
                                       cafile=cafile)
        self.server_version = __version__
        self.scanner_info['name'] = 'nmap'
        self.scanner_info['version'] = '' # achieved during self.check()
        self.scanner_info['description'] = OSPD_DESC

    def check(self):
        """ Checks that nmap command line tool is found and is executable. """

        try:
            result = subprocess.check_output(['nmap', '-oX', '-'], stderr=subprocess.STDOUT)
        except OSError:
            # the command is not available
            return False

        if result is None:
            return False

        tree = ET.fromstring(result);

        if tree.tag != 'nmaprun':
            return False

        version = tree.attrib.get('version')
        if version == '':
            return False

        self.scanner_info['version'] = version

        return True

    def exec_scan(self, scan_id, target):
        """ Starts the nmap scanner for scan_id scan. """

        ports = self.get_scan_ports(scan_id)

        result = subprocess.check_output(['nmap', '-oX', '-', '-p %s' % ports, target])

        if result is None:
            self.add_scan_error(scan_id, host=target,
              value="A problem occurred trying to execute 'nmap'.")
            self.add_scan_error(scan_id, host=target,
                                value="The result of 'nmap' was empty.")
            return 2

        # initialize the port list
        tcp_ports = []

        # parse the output of the nmap command
        tree = ET.fromstring(result);
        for port in tree.find("host/ports"):
            if port.tag == 'port':
                if port.attrib.get('protocol') == 'tcp':
                    tcp_ports.append(port.attrib.get('portid'))


        # Create a general log entry about executing nmap
        # It is important to send at least one result, else
        # the host details won't be stored.
        self.add_scan_log(scan_id, host=target, name='Nmap summary',
                          value='Via Nmap %d open tcp ports were found.' % len(tcp_ports))

        # Create a log entry for each found port
        for port in tcp_ports:
            self.add_scan_log(scan_id, host=target, name='Nmap port detection',
                              port='{0}/tcp'.format(port))

        # store the found ports as host details
        if len(tcp_ports) > 0:
            self.add_scan_host_detail(scan_id, host=target, name="ports",
                                      value=", ".join(tcp_ports))
            self.add_scan_host_detail(scan_id, host=target, name="tcp_ports",
                                      value=", ".join(tcp_ports))
        return 1

def main():
    """ OSP nmap main function. """
    daemon_main('OSPD - nmap wrapper', OSPDnmap)
