-- mimetypes.lua
-- Version 1.0.0

--[[
Copyright (c) 2011 Matthew "LeafStorm" Frazier

Permission is hereby granted, free of charge, to any person
obtaining a copy of this software and associated documentation
files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following
conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

======

In addition, the MIME types contained in the Software were
originally obtained from the Apache HTTP Server available under the 
Apache Software License, Version 2.0 license
(http://directory.fsf.org/wiki/License:Apache2.0)
]]

-- This table is the one that actually contains the exported functions.

local mimetypes = {}

mimetypes.version = '1.0.0'


-- Extracts the extension from a filename and returns it.
-- The extension must be at the end of the string, and preceded by a dot and
-- at least one other character. Only the last part will be returned (so
-- "package-1.2.tar.gz" will return "gz").
-- If there is no extension, this function will return nil.

local function extension (filename)
    return filename:match(".+%.([%a%d]+)$")
end


-- Creates a deep copy of the given table.

local function copy (tbl)
    local ntbl = {}
    for key, value in pairs(tbl) do
        if type(value) == 'table' then
            ntbl[key] = copy(value)
        else
            ntbl[key] = value
        end
    end
    return ntbl
end


-- This is the default MIME type database.
-- It is a table with two members - "extensions" and "filenames".
-- The filenames table maps complete file names (like README) to MIME types.
-- The extensions just maps the files' extensions (like jpg) to types.

local defaultdb = {
     -- The MIME types. Remember to not include the dot on the extension.
     extensions = {
          ['mods']        = 'application/mods+xml',
          ['hps']         = 'application/vnd.hp-hps',
          ['sfv']         = 'text/x-sfv',
          ['pcl']         = 'application/vnd.hp-pcl',
          ['oth']         = 'application/vnd.oasis.opendocument.text-web',
          ['arc']         = 'application/x-freearc',
          ['txd']         = 'application/vnd.genomatix.tuxedo',
          ['tcl']         = 'application/x-tcl',
          ['apr']         = 'application/vnd.lotus-approach',
          ['viv']         = 'video/vnd.vivo',
          ['wmlsc']       = 'application/vnd.wap.wmlscriptc',
          ['inkml']       = 'application/inkml+xml',
          ['sdc']         = 'application/vnd.stardivision.calc',
          ['dll']         = 'application/x-msdownload',
          ['sdkd']        = 'application/vnd.solent.sdkm+xml',
          ['kpr']         = 'application/vnd.kde.kpresenter',
          ['uvd']         = 'application/vnd.dece.data',
          ['gtw']         = 'model/vnd.gtw',
          ['svd']         = 'application/vnd.svd',
          ['m14']         = 'application/x-msmediaview',
          ['spp']         = 'application/scvp-vp-response',
          ['qt']          = 'video/quicktime',
          ['c4g']         = 'application/vnd.clonk.c4group',
          ['dp']          = 'application/vnd.osgi.dp',
          ['oda']         = 'application/oda',
          ['log']         = 'text/plain',
          ['gre']         = 'application/vnd.geometry-explorer',
          ['sda']         = 'application/vnd.stardivision.draw',
          ['rms']         = 'application/vnd.jcp.javame.midlet-rms',
          ['ico']         = 'image/x-icon',
          ['cab']         = 'application/vnd.ms-cab-compressed',
          ['p7c']         = 'application/pkcs7-mime',
          ['cb7']         = 'application/x-cbr',
          ['src']         = 'application/x-wais-source',
          ['uvf']         = 'application/vnd.dece.data',
          ['dms']         = 'application/octet-stream',
          ['ccxml']       = 'application/ccxml+xml',
          ['uvvd']        = 'application/vnd.dece.data',
          ['abw']         = 'application/x-abiword',
          ['gex']         = 'application/vnd.geometry-explorer',
          ['es3']         = 'application/vnd.eszigno3+xml',
          ['mmr']         = 'image/vnd.fujixerox.edmics-mmr',
          ['wgt']         = 'application/widget',
          ['mp4a']        = 'audio/mp4',
          ['gram']        = 'application/srgs',
          ['p7m']         = 'application/pkcs7-mime',
          ['org']         = 'application/vnd.lotus-organizer',
          ['silo']        = 'model/mesh',
          ['scq']         = 'application/scvp-cv-request',
          ['cxx']         = 'text/x-c',
          ['ots']         = 'application/vnd.oasis.opendocument.spreadsheet-template',
          ['fhc']         = 'image/x-freehand',
          ['uvvf']        = 'application/vnd.dece.data',
          ['jpeg']        = 'image/jpeg',
          ['ma']          = 'application/mathematica',
          ['odm']         = 'application/vnd.oasis.opendocument.text-master',
          ['uvvh']        = 'video/vnd.dece.hd',
          ['bed']         = 'application/vnd.realvnc.bed',
          ['doc']         = 'application/msword',
          ['ice']         = 'x-conference/x-cooltalk',
          ['fg5']         = 'application/vnd.fujitsu.oasysgp',
          ['ustar']       = 'application/x-ustar',
          ['mesh']        = 'model/mesh',
          ['smv']         = 'video/x-smv',
          ['imp']         = 'application/vnd.accpac.simply.imp',
          ['movie']       = 'video/x-sgi-movie',
          ['cmp']         = 'application/vnd.yellowriver-custom-menu',
          ['avi']         = 'video/x-msvideo',
          ['cpt']         = 'application/mac-compactpro',
          ['lha']         = 'application/x-lzh-compressed',
          ['cmdf']        = 'chemical/x-cmdf',
          ['wvx']         = 'video/x-ms-wvx',
          ['prf']         = 'application/pics-rules',
          ['wmx']         = 'video/x-ms-wmx',
          ['wmv']         = 'video/x-ms-wmv',
          ['ggt']         = 'application/vnd.geogebra.tool',
          ['kpt']         = 'application/vnd.kde.kpresenter',
          ['kmz']         = 'application/vnd.google-earth.kmz',
          ['mpt']         = 'application/vnd.ms-project',
          ['caf']         = 'audio/x-caf',
          ['mp4']         = 'video/mp4',
          ['dcr']         = 'application/x-director',
          ['wm']          = 'video/x-ms-wm',
          ['ai']          = 'application/postscript',
          ['lrf']         = 'application/octet-stream',
          ['qbo']         = 'application/vnd.intu.qbo',
          ['vob']         = 'video/x-ms-vob',
          ['nml']         = 'application/vnd.enliven',
          ['xlf']         = 'application/x-xliff+xml',
          ['dcurl']       = 'text/vnd.curl.dcurl',
          ['hpid']        = 'application/vnd.hp-hpid',
          ['kwd']         = 'application/vnd.kde.kword',
          ['asx']         = 'video/x-ms-asf',
          ['asf']         = 'video/x-ms-asf',
          ['acu']         = 'application/vnd.acucobol',
          ['fm']          = 'application/vnd.framemaker',
          ['plf']         = 'application/vnd.pocketlearn',
          ['mng']         = 'video/x-mng',
          ['gramps']      = 'application/x-gramps-xml',
          ['mks']         = 'video/x-matroska',
          ['vcard']       = 'text/vcard',
          ['sgml']        = 'text/sgml',
          ['mvb']         = 'application/x-msmediaview',
          ['rp9']         = 'application/vnd.cloanto.rp9',
          ['mkv']         = 'video/x-matroska',
          ['xhtml']       = 'application/xhtml+xml',
          ['dssc']        = 'application/dssc+der',
          ['m4v']         = 'video/x-m4v',
          ['xml']         = 'application/xml',
          ['fli']         = 'video/x-fli',
          ['mpm']         = 'application/vnd.blueice.multipass',
          ['pskcxml']     = 'application/pskc+xml',
          ['webm']        = 'video/webm',
          ['pml']         = 'application/vnd.ctc-posml',
          ['cu']          = 'application/cu-seeme',
          ['rm']          = 'application/vnd.rn-realmedia',
          ['spx']         = 'audio/ogg',
          ['dic']         = 'text/x-c',
          ['esa']         = 'application/vnd.osgi.subsystem',
          ['odi']         = 'application/vnd.oasis.opendocument.image',
          ['irm']         = 'application/vnd.ibm.rights-management',
          ['m4u']         = 'video/vnd.mpegurl',
          ['cdmid']       = 'application/cdmi-domain',
          ['elc']         = 'application/octet-stream',
          ['mxu']         = 'video/vnd.mpegurl',
          ['mdi']         = 'image/vnd.ms-modi',
          ['rld']         = 'application/resource-lists-diff+xml',
          ['sdw']         = 'application/vnd.stardivision.writer',
          ['fvt']         = 'video/vnd.fvt',
          ['pptx']        = 'application/vnd.openxmlformats-officedocument.presentationml.presentation',
          ['dvb']         = 'video/vnd.dvb.file',
          ['book']        = 'application/vnd.framemaker',
          ['pfx']         = 'application/x-pkcs12',
          ['qwd']         = 'application/vnd.quark.quarkxpress',
          ['nbp']         = 'application/vnd.wolfram.player',
          ['vcx']         = 'application/vnd.vcx',
          ['gpx']         = 'application/gpx+xml',
          ['uvvs']        = 'video/vnd.dece.sd',
          ['shf']         = 'application/shf+xml',
          ['n-gage']      = 'application/vnd.nokia.n-gage.symbian.install',
          ['hvd']         = 'application/vnd.yamaha.hv-dic',
          ['lostxml']     = 'application/lost+xml',
          ['knp']         = 'application/vnd.kinar',
          ['dvi']         = 'application/x-dvi',
          ['uvvp']        = 'video/vnd.dece.pd',
          ['psb']         = 'application/vnd.3gpp.pic-bw-small',
          ['asc']         = 'application/pgp-signature',
          ['wsdl']        = 'application/wsdl+xml',
          ['stk']         = 'application/hyperstudio',
          ['x3dbz']       = 'model/x3d+binary',
          ['uvvm']        = 'video/vnd.dece.mobile',
          ['3gp']         = 'video/3gpp',
          ['uvm']         = 'video/vnd.dece.mobile',
          ['sfd-hdstx']   = 'application/vnd.hydrostatix.sof-data',
          ['mpeg']        = 'video/mpeg',
          ['uvh']         = 'video/vnd.dece.hd',
          ['mov']         = 'video/quicktime',
          ['cla']         = 'application/vnd.claymore',
          ['ogv']         = 'video/ogg',
          ['m2v']         = 'video/mpeg',
          ['sse']         = 'application/vnd.kodak-descriptor',
          ['atomsvc']     = 'application/atomsvc+xml',
          ['djv']         = 'image/vnd.djvu',
          ['fe_launch']   = 'application/vnd.denovo.fcselayout-link',
          ['mpe']         = 'video/mpeg',
          ['x3dz']        = 'model/x3d+xml',
          ['et3']         = 'application/vnd.eszigno3+xml',
          ['u32']         = 'application/x-authorware-bin',
          ['mpg']         = 'video/mpeg',
          ['mpg4']        = 'video/mp4',
          ['mp4v']        = 'video/mp4',
          ['nsf']         = 'application/vnd.lotus-notes',
          ['dwg']         = 'image/vnd.dwg',
          ['teicorpus']   = 'application/tei+xml',
          ['sus']         = 'application/vnd.sus-calendar',
          ['uvvz']        = 'application/vnd.dece.zip',
          ['uvvx']        = 'application/vnd.dece.unspecified',
          ['psd']         = 'image/vnd.adobe.photoshop',
          ['ami']         = 'application/vnd.amiga.ami',
          ['nb']          = 'application/mathematica',
          ['nzb']         = 'application/x-nzb',
          ['tsd']         = 'application/timestamped-data',
          ['f']           = 'text/x-fortran',
          ['c11amc']      = 'application/vnd.cluetrust.cartomobile-config',
          ['torrent']     = 'application/x-bittorrent',
          ['cdkey']       = 'application/vnd.mediastation.cdkey',
          ['mj2']         = 'video/mj2',
          ['nns']         = 'application/vnd.noblenet-sealer',
          ['jpgm']        = 'video/jpm',
          ['jpm']         = 'video/jpm',
          ['jpgv']        = 'video/jpeg',
          ['h264']        = 'video/h264',
          ['mus']         = 'application/vnd.musician',
          ['ppsm']        = 'application/vnd.ms-powerpoint.slideshow.macroenabled.12',
          ['appcache']    = 'text/cache-manifest',
          ['rpss']        = 'application/vnd.nokia.radio-presets',
          ['smi']         = 'application/smil+xml',
          ['h263']        = 'video/h263',
          ['h261']        = 'video/h261',
          ['3g2']         = 'video/3gpp2',
          ['rtx']         = 'text/richtext',
          ['vcf']         = 'text/x-vcard',
          ['texi']        = 'application/x-texinfo',
          ['cdbcmsg']     = 'application/vnd.contact.cmsg',
          ['uu']          = 'text/x-uuencode',
          ['nsc']         = 'application/x-conference',
          ['pic']         = 'image/x-pict',
          ['cil']         = 'application/vnd.ms-artgalry',
          ['t']           = 'text/troff',
          ['psf']         = 'application/x-font-linux-psf',
          ['p']           = 'text/x-pascal',
          ['vsf']         = 'application/vnd.vsf',
          ['odc']         = 'application/vnd.oasis.opendocument.chart',
          ['lbe']         = 'application/vnd.llamagraphics.life-balance.exchange+xml',
          ['java']        = 'text/x-java-source',
          ['tfm']         = 'application/x-tex-tfm',
          ['cpio']        = 'application/x-cpio',
          ['gnumeric']    = 'application/x-gnumeric',
          ['for']         = 'text/x-fortran',
          ['uvu']         = 'video/vnd.uvvu.mp4',
          ['qxt']         = 'application/vnd.quark.quarkxpress',
          ['fxpl']        = 'application/vnd.adobe.fxp',
          ['hh']          = 'text/x-c',
          ['uvi']         = 'image/vnd.dece.graphic',
          ['cpp']         = 'text/x-c',
          ['x3db']        = 'model/x3d+binary',
          ['obd']         = 'application/x-msbinder',
          ['cc']          = 'text/x-c',
          ['rtf']         = 'application/rtf',
          ['ddd']         = 'application/vnd.fujixerox.ddd',
          ['ttf']         = 'application/x-font-ttf',
          ['iif']         = 'application/vnd.shana.informed.interchange',
          ['tao']         = 'application/vnd.tao.intent-module-archive',
          ['potm']        = 'application/vnd.ms-powerpoint.template.macroenabled.12',
          ['oxt']         = 'application/vnd.openofficeorg.extension',
          ['mif']         = 'application/vnd.mif',
          ['mk3d']        = 'video/x-matroska',
          ['mrc']         = 'application/marc',
          ['cxt']         = 'application/x-director',
          ['thmx']        = 'application/vnd.ms-officetheme',
          ['ext']         = 'application/vnd.novadigm.ext',
          ['tr']          = 'text/troff',
          ['gxt']         = 'application/vnd.geonext',
          ['rcprofile']   = 'application/vnd.ipunplugged.rcprofile',
          ['xlsx']        = 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
          ['wmls']        = 'text/vnd.wap.wmlscript',
          ['wml']         = 'text/vnd.wap.wml',
          ['dpg']         = 'application/vnd.dpgraph',
          ['wri']         = 'application/x-mswrite',
          ['cif']         = 'chemical/x-cif',
          ['wks']         = 'application/vnd.ms-works',
          ['ods']         = 'application/vnd.oasis.opendocument.spreadsheet',
          ['lasxml']      = 'application/vnd.las.las+xml',
          ['jpg']         = 'image/jpeg',
          ['ltf']         = 'application/vnd.frogans.ltf',
          ['spot']        = 'text/vnd.in3d.spot',
          ['3dml']        = 'text/vnd.in3d.3dml',
          ['curl']        = 'text/vnd.curl',
          ['xdp']         = 'application/vnd.adobe.xdp+xml',
          ['flx']         = 'text/vnd.fmi.flexstor',
          ['fly']         = 'text/vnd.fly',
          ['uvva']        = 'audio/vnd.dece.audio',
          ['p7r']         = 'application/x-pkcs7-certreqresp',
          ['cryptonote']  = 'application/vnd.rig.cryptonote',
          ['cdmio']       = 'application/cdmi-object',
          ['igx']         = 'application/vnd.micrografx.igx',
          ['smzip']       = 'application/vnd.stepmania.package',
          ['ogx']         = 'application/ogg',
          ['scs']         = 'application/scvp-cv-response',
          ['gv']          = 'text/vnd.graphviz',
          ['urls']        = 'text/uri-list',
          ['otc']         = 'application/vnd.oasis.opendocument.chart-template',
          ['png']         = 'image/png',
          ['uri']         = 'text/uri-list',
          ['ttl']         = 'text/turtle',
          ['metalink']    = 'application/metalink+xml',
          ['cmc']         = 'application/vnd.cosmocaller',
          ['ahead']       = 'application/vnd.ahead.space',
          ['taglet']      = 'application/vnd.mynfc',
          ['list']        = 'text/plain',
          ['dae']         = 'model/vnd.collada+xml',
          ['me']          = 'text/troff',
          ['srt']         = 'application/x-subrip',
          ['fgd']         = 'application/x-director',
          ['man']         = 'text/troff',
          ['jpe']         = 'image/jpeg',
          ['clkw']        = 'application/vnd.crick.clicker.wordbank',
          ['mfm']         = 'application/vnd.mfmp',
          ['pfb']         = 'application/x-font-type1',
          ['roff']        = 'text/troff',
          ['dtd']         = 'application/xml-dtd',
          ['ifm']         = 'application/vnd.shana.informed.formdata',
          ['pas']         = 'text/x-pascal',
          ['bz']          = 'application/x-bzip',
          ['ras']         = 'image/x-cmu-raster',
          ['car']         = 'application/vnd.curl.car',
          ['list3820']    = 'application/vnd.ibm.modcap',
          ['mgz']         = 'application/vnd.proteus.magazine',
          ['crt']         = 'application/x-x509-ca-cert',
          ['sbml']        = 'application/sbml+xml',
          ['pcurl']       = 'application/vnd.curl.pcurl',
          ['dbk']         = 'application/docbook+xml',
          ['sgm']         = 'text/sgml',
          ['ac']          = 'application/pkix-attr-cert',
          ['opf']         = 'application/oebps-package+xml',
          ['mar']         = 'application/octet-stream',
          ['asm']         = 'text/x-asm',
          ['wav']         = 'audio/x-wav',
          ['x3dvz']       = 'model/x3d+vrml',
          ['bpk']         = 'application/octet-stream',
          ['dtb']         = 'application/x-dtbook+xml',
          ['dsc']         = 'text/prs.lines.tag',
          ['wpd']         = 'application/vnd.wordperfect',
          ['xsl']         = 'application/xml',
          ['in']          = 'text/plain',
          ['stc']         = 'application/vnd.sun.xml.calc.template',
          ['musicxml']    = 'application/vnd.recordare.musicxml+xml',
          ['sc']          = 'application/vnd.ibm.secure-container',
          ['def']         = 'text/plain',
          ['uvs']         = 'video/vnd.dece.sd',
          ['wma']         = 'audio/x-ms-wma',
          ['text']        = 'text/plain',
          ['obj']         = 'application/x-tgif',
          ['cap']         = 'application/vnd.tcpdump.pcap',
          ['lvp']         = 'audio/vnd.lucent.voice',
          ['kfo']         = 'application/vnd.kde.kformula',
          ['sitx']        = 'application/x-stuffitx',
          ['jnlp']        = 'application/x-java-jnlp-file',
          ['rmvb']        = 'application/vnd.rn-realmedia-vbr',
          ['blb']         = 'application/x-blorb',
          ['evy']         = 'application/x-envoy',
          ['htm']         = 'text/html',
          ['html']        = 'text/html',
          ['sv4cpio']     = 'application/x-sv4cpio',
          ['mpkg']        = 'application/vnd.apple.installer+xml',
          ['xer']         = 'application/patch-ops-error+xml',
          ['cdmic']       = 'application/cdmi-container',
          ['csv']         = 'text/csv',
          ['aso']         = 'application/vnd.accpac.simply.aso',
          ['css']         = 'text/css',
          ['pki']         = 'application/pkixcmp',
          ['sdkm']        = 'application/vnd.solent.sdkm+xml',
          ['ifb']         = 'text/calendar',
          ['mcd']         = 'application/vnd.mcd',
          ['nnd']         = 'application/vnd.noblenet-directory',
          ['pbm']         = 'image/x-portable-bitmap',
          ['srx']         = 'application/sparql-results+xml',
          ['cdmiq']       = 'application/cdmi-queue',
          ['qxl']         = 'application/vnd.quark.quarkxpress',
          ['ics']         = 'text/calendar',
          ['icm']         = 'application/vnd.iccprofile',
          ['x3dv']        = 'model/x3d+vrml',
          ['vrml']        = 'model/vrml',
          ['xbm']         = 'image/x-xbitmap',
          ['spl']         = 'application/x-futuresplash',
          ['gtm']         = 'application/vnd.groove-tool-message',
          ['paw']         = 'application/vnd.pawaafile',
          ['csp']         = 'application/vnd.commonspace',
          ['jlt']         = 'application/vnd.hp-jlyt',
          ['rip']         = 'audio/vnd.rip',
          ['mxl']         = 'application/vnd.recordare.musicxml',
          ['rep']         = 'application/vnd.businessobjects',
          ['mts']         = 'model/vnd.mts',
          ['mads']        = 'application/mads+xml',
          ['gdl']         = 'model/vnd.gdl',
          ['ksp']         = 'application/vnd.kde.kspread',
          ['dwf']         = 'model/vnd.dwf',
          ['ms']          = 'text/troff',
          ['udeb']        = 'application/x-debian-package',
          ['iges']        = 'model/iges',
          ['fcdt']        = 'application/vnd.adobe.formscentral.fcdt',
          ['xo']          = 'application/vnd.olpc-sugar',
          ['xlt']         = 'application/vnd.ms-excel',
          ['gtar']        = 'application/x-gtar',
          ['p12']         = 'application/x-pkcs12',
          ['gqf']         = 'application/vnd.grafeq',
          ['wbmp']        = 'image/vnd.wap.wbmp',
          ['mime']        = 'message/rfc822',
          ['sv4crc']      = 'application/x-sv4crc',
          ['xwd']         = 'image/x-xwindowdump',
          ['qxb']         = 'application/vnd.quark.quarkxpress',
          ['xpm']         = 'image/x-xpixmap',
          ['tga']         = 'image/x-tga',
          ['seed']        = 'application/vnd.fdsn.seed',
          ['rss']         = 'application/rss+xml',
          ['ppm']         = 'image/x-portable-pixmap',
          ['fsc']         = 'application/vnd.fsc.weblaunch',
          ['cat']         = 'application/vnd.ms-pki.seccat',
          ['mwf']         = 'application/vnd.mfer',
          ['pnm']         = 'image/x-portable-anymap',
          ['pct']         = 'image/x-pict',
          ['pcx']         = 'image/x-pcx',
          ['gph']         = 'application/vnd.flographit',
          ['sid']         = 'image/x-mrsid-image',
          ['class']       = 'application/java-vm',
          ['c4p']         = 'application/vnd.clonk.c4group',
          ['mp3']         = 'audio/mpeg',
          ['uvvi']        = 'image/vnd.dece.graphic',
          ['fh4']         = 'image/x-freehand',
          ['rgb']         = 'image/x-rgb',
          ['vcg']         = 'application/vnd.groove-vcard',
          ['cmx']         = 'image/x-cmx',
          ['3ds']         = 'image/x-3ds',
          ['cdy']         = 'application/vnd.cinderella',
          ['webp']        = 'image/webp',
          ['xif']         = 'image/vnd.xiff',
          ['p10']         = 'application/pkcs10',
          ['wax']         = 'audio/x-ms-wax',
          ['npx']         = 'image/vnd.net-fpx',
          ['ufdl']        = 'application/vnd.ufdl',
          ['rlc']         = 'image/vnd.fujixerox.edmics-rlc',
          ['fst']         = 'image/vnd.fst',
          ['setreg']      = 'application/set-registration-initiation',
          ['mbox']        = 'application/mbox',
          ['kpxx']        = 'application/vnd.ds-keypoint',
          ['fbs']         = 'image/vnd.fastbidsheet',
          ['z1']          = 'application/x-zmachine',
          ['weba']        = 'audio/webm',
          ['clkt']        = 'application/vnd.crick.clicker.template',
          ['m13']         = 'application/x-msmediaview',
          ['cww']         = 'application/prs.cww',
          ['sxd']         = 'application/vnd.sun.xml.draw',
          ['xpw']         = 'application/vnd.intercon.formnet',
          ['dd2']         = 'application/vnd.oma.dd2+xml',
          ['odf']         = 'application/vnd.oasis.opendocument.formula',
          ['fzs']         = 'application/vnd.fuzzysheet',
          ['portpkg']     = 'application/vnd.macports.portpkg',
          ['oti']         = 'application/vnd.oasis.opendocument.image-template',
          ['hlp']         = 'application/winhlp',
          ['cst']         = 'application/x-director',
          ['mpn']         = 'application/vnd.mophun.application',
          ['xvm']         = 'application/xv+xml',
          ['cgm']         = 'image/cgm',
          ['fh5']         = 'image/x-freehand',
          ['h']           = 'text/x-c',
          ['xar']         = 'application/vnd.xara',
          ['vis']         = 'application/vnd.visionary',
          ['wqd']         = 'application/vnd.wqd',
          ['cbr']         = 'application/x-cbr',
          ['svgz']        = 'image/svg+xml',
          ['wpl']         = 'application/vnd.ms-wpl',
          ['lwp']         = 'application/vnd.lotus-wordpro',
          ['igm']         = 'application/vnd.insors.igm',
          ['flo']         = 'application/vnd.micrografx.flo',
          ['zirz']        = 'application/vnd.zul',
          ['svg']         = 'image/svg+xml',
          ['m3u8']        = 'application/vnd.apple.mpegurl',
          ['dra']         = 'audio/vnd.dra',
          ['oprc']        = 'application/vnd.palm',
          ['vxml']        = 'application/voicexml+xml',
          ['dis']         = 'application/vnd.mobius.dis',
          ['qps']         = 'application/vnd.publishare-delta-tree',
          ['vsw']         = 'application/vnd.visio',
          ['ntf']         = 'application/vnd.nitf',
          ['btif']        = 'image/prs.btif',
          ['uris']        = 'text/uri-list',
          ['mxf']         = 'application/mxf',
          ['wps']         = 'application/vnd.ms-works',
          ['aas']         = 'application/x-authorware-seg',
          ['c4d']         = 'application/vnd.clonk.c4group',
          ['zip']         = 'application/zip',
          ['gif']         = 'image/gif',
          ['susp']        = 'application/vnd.sus-calendar',
          ['air']         = 'application/vnd.adobe.air-application-installer-package+zip',
          ['ris']         = 'application/x-research-info-systems',
          ['pgn']         = 'application/x-chess-pgn',
          ['gxf']         = 'application/gxf',
          ['g3']          = 'image/g3fax',
          ['sdd']         = 'application/vnd.stardivision.impress',
          ['bmp']         = 'image/bmp',
          ['otg']         = 'application/vnd.oasis.opendocument.graphics-template',
          ['eps']         = 'application/postscript',
          ['xyz']         = 'chemical/x-xyz',
          ['sema']        = 'application/vnd.sema',
          ['csml']        = 'chemical/x-csml',
          ['swf']         = 'application/x-shockwave-flash',
          ['qxd']         = 'application/vnd.quark.quarkxpress',
          ['cdx']         = 'chemical/x-cdx',
          ['123']         = 'application/vnd.lotus-1-2-3',
          ['dotx']        = 'application/vnd.openxmlformats-officedocument.wordprocessingml.template',
          ['xm']          = 'audio/xm',
          ['spq']         = 'application/scvp-vp-request',
          ['jisp']        = 'application/vnd.jisp',
          ['rmp']         = 'audio/x-pn-realaudio-plugin',
          ['tex']         = 'application/x-tex',
          ['xlm']         = 'application/vnd.ms-excel',
          ['ppam']        = 'application/vnd.ms-powerpoint.addin.macroenabled.12',
          ['xht']         = 'application/xhtml+xml',
          ['odb']         = 'application/vnd.oasis.opendocument.database',
          ['ram']         = 'audio/x-pn-realaudio',
          ['xfdl']        = 'application/vnd.xfdl',
          ['mka']         = 'audio/x-matroska',
          ['flac']        = 'audio/x-flac',
          ['aifc']        = 'audio/x-aiff',
          ['sfs']         = 'application/vnd.spotfire.sfs',
          ['rl']          = 'application/resource-lists+xml',
          ['wdb']         = 'application/vnd.ms-works',
          ['tra']         = 'application/vnd.trueapp',
          ['dfac']        = 'application/vnd.dreamfactory',
          ['aif']         = 'audio/x-aiff',
          ['mp2']         = 'audio/mpeg',
          ['vtu']         = 'model/vnd.vtu',
          ['mny']         = 'application/x-msmoney',
          ['ecelp7470']   = 'audio/vnd.nuera.ecelp7470',
          ['ppsx']        = 'application/vnd.openxmlformats-officedocument.presentationml.slideshow',
          ['kon']         = 'application/vnd.kde.kontour',
          ['odp']         = 'application/vnd.oasis.opendocument.presentation',
          ['twds']        = 'application/vnd.simtech-mindmapper',
          ['dtshd']       = 'audio/vnd.dts.hd',
          ['dts']         = 'audio/vnd.dts',
          ['sit']         = 'application/x-stuffit',
          ['ipfix']       = 'application/ipfix',
          ['sgi']         = 'image/sgi',
          ['application'] = 'application/x-ms-application',
          ['roa']         = 'application/rpki-roa',
          ['xaml']        = 'application/xaml+xml',
          ['sil']         = 'audio/silk',
          ['smil']        = 'application/smil+xml',
          ['s3m']         = 'audio/s3m',
          ['ogg']         = 'audio/ogg',
          ['chm']         = 'application/vnd.ms-htmlhelp',
          ['mscml']       = 'application/mediaservercontrol+xml',
          ['atom']        = 'application/atom+xml',
          ['eva']         = 'application/x-eva',
          ['oga']         = 'audio/ogg',
          ['prc']         = 'application/x-mobipocket-ebook',
          ['sldm']        = 'application/vnd.ms-powerpoint.slide.macroenabled.12',
          ['m2a']         = 'audio/mpeg',
          ['frame']       = 'application/vnd.framemaker',
          ['fh7']         = 'image/x-freehand',
          ['mp2a']        = 'audio/mpeg',
          ['aac']         = 'audio/x-aac',
          ['mpga']        = 'audio/mpeg',
          ['rmi']         = 'audio/midi',
          ['m21']         = 'application/mp21',
          ['kar']         = 'audio/midi',
          ['p7b']         = 'application/x-pkcs7-certificates',
          ['gac']         = 'application/vnd.groove-account',
          ['midi']        = 'audio/midi',
          ['mid']         = 'audio/midi',
          ['ott']         = 'application/vnd.oasis.opendocument.text-template',
          ['rif']         = 'application/reginfo+xml',
          ['cfs']         = 'application/x-cfs-compressed',
          ['au']          = 'audio/basic',
          ['adp']         = 'audio/adpcm',
          ['pfr']         = 'application/font-tdpfr',
          ['ief']         = 'image/ief',
          ['xul']         = 'application/vnd.mozilla.xul+xml',
          ['dart']        = 'application/vnd.dart',
          ['yin']         = 'application/yin+xml',
          ['pya']         = 'audio/vnd.ms-playready.media.pya',
          ['uvg']         = 'image/vnd.dece.graphic',
          ['str']         = 'application/vnd.pg.format',
          ['tiff']        = 'image/tiff',
          ['crd']         = 'application/x-mscardfile',
          ['c4f']         = 'application/vnd.clonk.c4group',
          ['xhvml']       = 'application/xv+xml',
          ['ssdl']        = 'application/ssdl+xml',
          ['mxml']        = 'application/xv+xml',
          ['g2w']         = 'application/vnd.geoplan',
          ['xslt']        = 'application/xslt+xml',
          ['nnw']         = 'application/vnd.noblenet-web',
          ['xop']         = 'application/xop+xml',
          ['flv']         = 'video/x-flv',
          ['ra']          = 'audio/x-pn-realaudio',
          ['tpt']         = 'application/vnd.trid.tpt',
          ['bdm']         = 'application/vnd.syncml.dm+wbxml',
          ['davmount']    = 'application/davmount+xml',
          ['z6']          = 'application/x-zmachine',
          ['xenc']        = 'application/xenc+xml',
          ['yang']        = 'application/yang',
          ['uva']         = 'audio/vnd.dece.audio',
          ['ez2']         = 'application/vnd.ezpix-album',
          ['dmp']         = 'application/vnd.tcpdump.pcap',
          ['mets']        = 'application/mets+xml',
          ['z8']          = 'application/x-zmachine',
          ['xfdf']        = 'application/vnd.adobe.xfdf',
          ['fcs']         = 'application/vnd.isac.fcs',
          ['z7']          = 'application/x-zmachine',
          ['xdssc']       = 'application/dssc+xml',
          ['z5']          = 'application/x-zmachine',
          ['z4']          = 'application/x-zmachine',
          ['daf']         = 'application/vnd.mobius.daf',
          ['sql']         = 'application/x-sql',
          ['z3']          = 'application/x-zmachine',
          ['z2']          = 'application/x-zmachine',
          ['uvvt']        = 'application/vnd.dece.ttml+xml',
          ['xz']          = 'application/x-xz',
          ['xdm']         = 'application/vnd.syncml.dm+xml',
          ['xpi']         = 'application/x-xpinstall',
          ['fig']         = 'application/x-xfig',
          ['der']         = 'application/x-x509-ca-cert',
          ['chrt']        = 'application/vnd.kde.kchart',
          ['vcs']         = 'text/x-vcalendar',
          ['pgp']         = 'application/pgp-encrypted',
          ['onetmp']      = 'application/onenote',
          ['mjp2']        = 'video/mj2',
          ['pvb']         = 'application/vnd.3gpp.pic-bw-var',
          ['f90']         = 'text/x-fortran',
          ['tar']         = 'application/x-tar',
          ['hvp']         = 'application/vnd.yamaha.hv-voice',
          ['afm']         = 'application/x-font-type1',
          ['semd']        = 'application/vnd.semd',
          ['mp4s']        = 'application/mp4',
          ['eml']         = 'message/rfc822',
          ['otp']         = 'application/vnd.oasis.opendocument.presentation-template',
          ['gml']         = 'application/gml+xml',
          ['itp']         = 'application/vnd.shana.informed.formtemplate',
          ['st']          = 'application/vnd.sailingtracker.track',
          ['svc']         = 'application/vnd.dvb.service',
          ['cml']         = 'chemical/x-cml',
          ['pkipath']     = 'application/pkix-pkipath',
          ['sh']          = 'application/x-sh',
          ['spf']         = 'application/vnd.yamaha.smaf-phrase',
          ['mcurl']       = 'text/vnd.curl.mcurl',
          ['spc']         = 'application/x-pkcs7-certificates',
          ['rq']          = 'application/sparql-query',
          ['uvvv']        = 'video/vnd.dece.video',
          ['cdf']         = 'application/x-netcdf',
          ['atomcat']     = 'application/atomcat+xml',
          ['jad']         = 'text/vnd.sun.j2me.app-descriptor',
          ['gqs']         = 'application/vnd.grafeq',
          ['bin']         = 'application/octet-stream',
          ['m3u']         = 'audio/x-mpegurl',
          ['scd']         = 'application/x-msschedule',
          ['pub']         = 'application/x-mspublisher',
          ['sm']          = 'application/vnd.stepmania.stepchart',
          ['ecelp9600']   = 'audio/vnd.nuera.ecelp9600',
          ['mpy']         = 'application/vnd.ibm.minipay',
          ['xlam']        = 'application/vnd.ms-excel.addin.macroenabled.12',
          ['vox']         = 'application/x-authorware-bin',
          ['emf']         = 'application/x-msmetafile',
          ['mc1']         = 'application/vnd.medcalcdata',
          ['zmm']         = 'application/vnd.handheld-entertainment+xml',
          ['mxs']         = 'application/vnd.triscape.mxs',
          ['msi']         = 'application/x-msdownload',
          ['mseq']        = 'application/vnd.mseq',
          ['emma']        = 'application/emma+xml',
          ['bat']         = 'application/x-msdownload',
          ['com']         = 'application/x-msdownload',
          ['exe']         = 'application/x-msdownload',
          ['xpx']         = 'application/vnd.intercon.formnet',
          ['twd']         = 'application/vnd.simtech-mindmapper',
          ['clp']         = 'application/x-msclip',
          ['mdb']         = 'application/x-msaccess',
          ['xbap']        = 'application/x-ms-xbap',
          ['xlc']         = 'application/vnd.ms-excel',
          ['fpx']         = 'image/vnd.fpx',
          ['wmd']         = 'application/x-ms-wmd',
          ['lnk']         = 'application/x-ms-shortcut',
          ['lrm']         = 'application/vnd.ms-lrm',
          ['latex']       = 'application/x-latex',
          ['xdw']         = 'application/vnd.fujixerox.docuworks',
          ['dump']        = 'application/octet-stream',
          ['plc']         = 'application/vnd.mobius.plc',
          ['cdxml']       = 'application/vnd.chemdraw+xml',
          ['umj']         = 'application/vnd.umajin',
          ['osfpvg']      = 'application/vnd.yamaha.openscoreformat.osfpvg+xml',
          ['mobi']        = 'application/x-mobipocket-ebook',
          ['dxr']         = 'application/x-director',
          ['mie']         = 'application/x-mie',
          ['ei6']         = 'application/vnd.pg.osasli',
          ['lzh']         = 'application/x-lzh-compressed',
          ['dxp']         = 'application/vnd.spotfire.dxp',
          ['iso']         = 'application/x-iso9660-image',
          ['install']     = 'application/x-install-instructions',
          ['hdf']         = 'application/x-hdf',
          ['igs']         = 'model/iges',
          ['ims']         = 'application/vnd.ms-ims',
          ['dist']        = 'application/octet-stream',
          ['efif']        = 'application/vnd.picsel',
          ['wrl']         = 'model/vrml',
          ['esf']         = 'application/vnd.epson.esf',
          ['link66']      = 'application/vnd.route66.link66+xml',
          ['woff']        = 'application/font-woff',
          ['gim']         = 'application/vnd.groove-identity-message',
          ['t3']          = 'application/x-t3vm-image',
          ['mp21']        = 'application/mp21',
          ['xla']         = 'application/vnd.ms-excel',
          ['odft']        = 'application/vnd.oasis.opendocument.formula-template',
          ['msf']         = 'application/vnd.epson.msf',
          ['ktr']         = 'application/vnd.kahootz',
          ['box']         = 'application/vnd.previewsystems.box',
          ['crl']         = 'application/pkix-crl',
          ['skp']         = 'application/vnd.koan',
          ['ttc']         = 'application/x-font-ttf',
          ['osf']         = 'application/vnd.yamaha.openscoreformat',
          ['oa2']         = 'application/vnd.fujitsu.oasys2',
          ['pcf']         = 'application/x-font-pcf',
          ['fxp']         = 'application/vnd.adobe.fxp',
          ['trm']         = 'application/x-msterminal',
          ['oxps']        = 'application/oxps',
          ['gsf']         = 'application/x-font-ghostscript',
          ['swa']         = 'application/x-director',
          ['mb']          = 'application/mathematica',
          ['bdf']         = 'application/x-font-bdf',
          ['mmf']         = 'application/vnd.smaf',
          ['ufd']         = 'application/vnd.ufdl',
          ['n3']          = 'text/n3',
          ['rdz']         = 'application/vnd.data-vision.rdz',
          ['pot']         = 'application/vnd.ms-powerpoint',
          ['ncx']         = 'application/x-dtbncx+xml',
          ['wad']         = 'application/x-doom',
          ['w3d']         = 'application/x-director',
          ['mpc']         = 'application/vnd.mophun.certificate',
          ['ktz']         = 'application/vnd.kahootz',
          ['cct']         = 'application/x-director',
          ['dir']         = 'application/x-director',
          ['distz']       = 'application/octet-stream',
          ['rpst']        = 'application/vnd.nokia.radio-preset',
          ['csh']         = 'application/x-csh',
          ['skd']         = 'application/vnd.koan',
          ['jsonml']      = 'application/jsonml+json',
          ['sisx']        = 'application/vnd.symbian.install',
          ['boz']         = 'application/x-bzip2',
          ['deploy']      = 'application/octet-stream',
          ['msh']         = 'model/mesh',
          ['epub']        = 'application/epub+zip',
          ['smf']         = 'application/vnd.stardivision.math',
          ['dmg']         = 'application/x-apple-diskimage',
          ['deb']         = 'application/x-debian-package',
          ['f77']         = 'text/x-fortran',
          ['ssf']         = 'application/vnd.epson.ssf',
          ['etx']         = 'text/x-setext',
          ['chat']        = 'application/x-chat',
          ['shar']        = 'application/x-shar',
          ['wbxml']       = 'application/vnd.wap.wbxml',
          ['atx']         = 'application/vnd.antix.game-component',
          ['unityweb']    = 'application/vnd.unity',
          ['cbz']         = 'application/x-cbr',
          ['cbt']         = 'application/x-cbr',
          ['ait']         = 'application/vnd.dvb.ait',
          ['cba']         = 'application/x-cbr',
          ['bz2']         = 'application/x-bzip2',
          ['blorb']       = 'application/x-blorb',
          ['txt']         = 'text/plain',
          ['bcpio']       = 'application/x-bcpio',
          ['ktx']         = 'image/ktx',
          ['scm']         = 'application/vnd.lotus-screencam',
          ['lbd']         = 'application/vnd.llamagraphics.life-balance.desktop',
          ['mmd']         = 'application/vnd.chipnuts.karaoke-mmd',
          ['x32']         = 'application/x-authorware-bin',
          ['ecma']        = 'application/ecmascript',
          ['pbd']         = 'application/vnd.powerbuilder6',
          ['docx']        = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
          ['stf']         = 'application/vnd.wt.stf',
          ['hal']         = 'application/vnd.hal+xml',
          ['aab']         = 'application/x-authorware-bin',
          ['json']        = 'application/json',
          ['uvt']         = 'application/vnd.dece.ttml+xml',
          ['pfa']         = 'application/x-font-type1',
          ['xbd']         = 'application/vnd.fujixerox.docuworks.binder',
          ['7z']          = 'application/x-7z-compressed',
          ['wspolicy']    = 'application/wspolicy+xml',
          ['zaz']         = 'application/vnd.zzazz.deck+xml',
          ['aam']         = 'application/x-authorware-map',
          ['rar']         = 'application/x-rar-compressed',
          ['saf']         = 'application/vnd.yamaha.smaf-audio',
          ['exi']         = 'application/exi',
          ['azf']         = 'application/vnd.airzip.filesecure.azf',
          ['msty']        = 'application/vnd.muvee.style',
          ['gam']         = 'application/x-tads',
          ['grxml']       = 'application/srgs+xml',
          ['ghf']         = 'application/vnd.groove-help',
          ['conf']        = 'text/plain',
          ['otf']         = 'application/x-font-otf',
          ['hqx']         = 'application/mac-binhex40',
          ['ssml']        = 'application/ssml+xml',
          ['tif']         = 'image/tiff',
          ['xvml']        = 'application/xv+xml',
          ['pkg']         = 'application/octet-stream',
          ['qam']         = 'application/vnd.epson.quickanime',
          ['bmi']         = 'application/vnd.bmi',
          ['pcap']        = 'application/vnd.tcpdump.pcap',
          ['ftc']         = 'application/vnd.fluxtime.clip',
          ['wmlc']        = 'application/vnd.wap.wmlc',
          ['vss']         = 'application/vnd.visio',
          ['std']         = 'application/vnd.sun.xml.draw.template',
          ['vst']         = 'application/vnd.visio',
          ['vsd']         = 'application/vnd.visio',
          ['msl']         = 'application/vnd.mobius.msl',
          ['uvv']         = 'video/vnd.dece.video',
          ['uoml']        = 'application/vnd.uoml+xml',
          ['sig']         = 'application/pgp-signature',
          ['utz']         = 'application/vnd.uiq.theme',
          ['wdp']         = 'image/vnd.ms-photo',
          ['tmo']         = 'application/vnd.tmobile-livetv',
          ['mqy']         = 'application/vnd.mobius.mqy',
          ['skt']         = 'application/vnd.koan',
          ['sdp']         = 'application/sdp',
          ['vor']         = 'application/vnd.stardivision.writer',
          ['geo']         = 'application/vnd.dynageo',
          ['sru']         = 'application/sru+xml',
          ['tei']         = 'application/tei+xml',
          ['xap']         = 'application/x-silverlight-app',
          ['c']           = 'text/x-c',
          ['ez']          = 'application/andrew-inset',
          ['ink']         = 'application/inkml+xml',
          ['edm']         = 'application/vnd.novadigm.edm',
          ['gca']         = 'application/x-gca-compressed',
          ['cii']         = 'application/vnd.anser-web-certificate-issue-initiation',
          ['oas']         = 'application/vnd.fujitsu.oasys',
          ['onetoc']      = 'application/onenote',
          ['oa3']         = 'application/vnd.fujitsu.oasys3',
          ['pptm']        = 'application/vnd.ms-powerpoint.presentation.macroenabled.12',
          ['wtb']         = 'application/vnd.webturbo',
          ['xlw']         = 'application/vnd.ms-excel',
          ['xsm']         = 'application/vnd.syncml+xml',
          ['dataless']    = 'application/vnd.fdsn.seed',
          ['iota']        = 'application/vnd.astraea-software.iota',
          ['dotm']        = 'application/vnd.ms-word.template.macroenabled.12',
          ['odt']         = 'application/vnd.oasis.opendocument.text',
          ['sxi']         = 'application/vnd.sun.xml.impress',
          ['uvp']         = 'video/vnd.dece.pd',
          ['pfm']         = 'application/x-font-type1',
          ['gbr']         = 'application/rpki-ghostbusters',
          ['pyv']         = 'video/vnd.ms-playready.media.pyv',
          ['f4v']         = 'video/x-f4v',
          ['uvvg']        = 'image/vnd.dece.graphic',
          ['pdf']         = 'application/pdf',
          ['apk']         = 'application/vnd.android.package-archive',
          ['swi']         = 'application/vnd.aristanetworks.swi',
          ['nc']          = 'application/x-netcdf',
          ['mrcx']        = 'application/marcxml+xml',
          ['meta4']       = 'application/metalink4+xml',
          ['ulx']         = 'application/x-glulx',
          ['rs']          = 'application/rls-services+xml',
          ['tcap']        = 'application/vnd.3gpp2.tcap',
          ['ipk']         = 'application/vnd.shana.informed.package',
          ['acc']         = 'application/vnd.americandynamics.acc',
          ['potx']        = 'application/vnd.openxmlformats-officedocument.presentationml.template',
          ['bh2']         = 'application/vnd.fujitsu.oasysprs',
          ['nfo']         = 'text/x-nfo',
          ['ppd']         = 'application/vnd.cups-ppd',
          ['mathml']      = 'application/mathml+xml',
          ['onepkg']      = 'application/onenote',
          ['azs']         = 'application/vnd.airzip.filesecure.azs',
          ['icc']         = 'application/vnd.iccprofile',
          ['eol']         = 'audio/vnd.digital-winds',
          ['fti']         = 'application/vnd.anser-web-funds-transfer-initiation',
          ['joda']        = 'application/vnd.joost.joda-archive',
          ['ivp']         = 'application/vnd.immervision-ivp',
          ['acutc']       = 'application/vnd.acucorp',
          ['uvvu']        = 'video/vnd.uvvu.mp4',
          ['ser']         = 'application/java-serialized-object',
          ['fnc']         = 'application/vnd.frogans.fnc',
          ['wg']          = 'application/vnd.pmi.widget',
          ['irp']         = 'application/vnd.irepository.package+xml',
          ['karbon']      = 'application/vnd.kde.karbon',
          ['p7s']         = 'application/pkcs7-signature',
          ['mseed']       = 'application/vnd.fdsn.mseed',
          ['pre']         = 'application/vnd.lotus-freelance',
          ['pwn']         = 'application/vnd.3m.post-it-notes',
          ['semf']        = 'application/vnd.semf',
          ['nitf']        = 'application/vnd.nitf',
          ['m1v']         = 'video/mpeg',
          ['aw']          = 'application/applixware',
          ['jar']         = 'application/java-archive',
          ['hvs']         = 'application/vnd.yamaha.hv-script',
          ['kia']         = 'application/vnd.kidspiration',
          ['flw']         = 'application/vnd.kde.kivio',
          ['ace']         = 'application/x-ace-compressed',
          ['vcd']         = 'application/x-cdlink',
          ['s']           = 'text/x-asm',
          ['plb']         = 'application/vnd.3gpp.pic-bw-large',
          ['i2g']         = 'application/vnd.intergeo',
          ['pclxl']       = 'application/vnd.hp-pclxl',
          ['qfx']         = 'application/vnd.intu.qfx',
          ['qwt']         = 'application/vnd.quark.quarkxpress',
          ['snd']         = 'audio/basic',
          ['xls']         = 'application/vnd.ms-excel',
          ['listafp']     = 'application/vnd.ibm.modcap',
          ['mag']         = 'application/vnd.ecowin.chart',
          ['slt']         = 'application/vnd.epson.salt',
          ['tfi']         = 'application/thraud+xml',
          ['x3d']         = 'model/x3d+xml',
          ['sxm']         = 'application/vnd.sun.xml.math',
          ['teacher']     = 'application/vnd.smart.teacher',
          ['pls']         = 'application/pls+xml',
          ['rsd']         = 'application/rsd+xml',
          ['ps']          = 'application/postscript',
          ['so']          = 'application/octet-stream',
          ['wmz']         = 'application/x-msmetafile',
          ['aiff']        = 'audio/x-aiff',
          ['nlu']         = 'application/vnd.neurolanguage.nlu',
          ['les']         = 'application/vnd.hhe.lesson-player',
          ['grv']         = 'application/vnd.groove-injector',
          ['wcm']         = 'application/vnd.ms-works',
          ['omdoc']       = 'application/omdoc+xml',
          ['hpgl']        = 'application/vnd.hp-hpgl',
          ['kwt']         = 'application/vnd.kde.kword',
          ['res']         = 'application/x-dtbresource+xml',
          ['cer']         = 'application/pkix-cert',
          ['ecelp4800']   = 'audio/vnd.nuera.ecelp4800',
          ['ggb']         = 'application/vnd.geogebra.file',
          ['edx']         = 'application/vnd.novadigm.edx',
          ['xltm']        = 'application/vnd.ms-excel.template.macroenabled.12',
          ['ivu']         = 'application/vnd.immervision-ivu',
          ['djvu']        = 'image/vnd.djvu',
          ['ngdat']       = 'application/vnd.nokia.n-gage.data',
          ['dna']         = 'application/vnd.dna',
          ['snf']         = 'application/x-font-snf',
          ['js']          = 'application/javascript',
          ['xpr']         = 'application/vnd.is-xpr',
          ['jam']         = 'application/vnd.jam',
          ['stl']         = 'application/vnd.ms-pki.stl',
          ['afp']         = 'application/vnd.ibm.modcap',
          ['p8']          = 'application/pkcs8',
          ['kne']         = 'application/vnd.kinar',
          ['xltx']        = 'application/vnd.openxmlformats-officedocument.spreadsheetml.template',
          ['ez3']         = 'application/vnd.ezpix-package',
          ['dgc']         = 'application/x-dgc-compressed',
          ['sxc']         = 'application/vnd.sun.xml.calc',
          ['clkk']        = 'application/vnd.crick.clicker.keyboard',
          ['c11amz']      = 'application/vnd.cluetrust.cartomobile-config-pkg',
          ['hbci']        = 'application/vnd.hbci',
          ['clkp']        = 'application/vnd.crick.clicker.palette',
          ['skm']         = 'application/vnd.koan',
          ['zir']         = 'application/vnd.zul',
          ['atc']         = 'application/vnd.acucorp',
          ['sxg']         = 'application/vnd.sun.xml.writer.global',
          ['wmf']         = 'application/x-msmetafile',
          ['texinfo']     = 'application/x-texinfo',
          ['mbk']         = 'application/vnd.mobius.mbk',
          ['wbs']         = 'application/vnd.criticaltools.wbs+xml',
          ['g3w']         = 'application/vnd.geospace',
          ['aep']         = 'application/vnd.audiograph',
          ['sxw']         = 'application/vnd.sun.xml.writer',
          ['emz']         = 'application/x-msmetafile',
          ['mlp']         = 'application/vnd.dolby.mlp',
          ['odg']         = 'application/vnd.oasis.opendocument.graphics',
          ['xlsb']        = 'application/vnd.ms-excel.sheet.binary.macroenabled.12',
          ['ptid']        = 'application/vnd.pvi.ptid1',
          ['xlsm']        = 'application/vnd.ms-excel.sheet.macroenabled.12',
          ['setpay']      = 'application/set-payment-initiation',
          ['maker']       = 'application/vnd.framemaker',
          ['igl']         = 'application/vnd.igloader',
          ['dot']         = 'application/msword',
          ['eot']         = 'application/vnd.ms-fontobject',
          ['fdf']         = 'application/vnd.fdf',
          ['ppt']         = 'application/vnd.ms-powerpoint',
          ['uvx']         = 'application/vnd.dece.unspecified',
          ['sgl']         = 'application/vnd.stardivision.writer-global',
          ['rdf']         = 'application/rdf+xml',
          ['m3a']         = 'audio/mpeg',
          ['mft']         = 'application/rpki-manifest',
          ['mpp']         = 'application/vnd.ms-project',
          ['docm']        = 'application/vnd.ms-word.document.macroenabled.12',
          ['tsv']         = 'text/tab-separated-values',
          ['tpl']         = 'application/vnd.groove-tool-template',
          ['rnc']         = 'application/relax-ng-compact-syntax',
          ['onetoc2']     = 'application/onenote',
          ['clkx']        = 'application/vnd.crick.clicker',
          ['xpl']         = 'application/xproc+xml',
          ['c4u']         = 'application/vnd.clonk.c4group',
          ['opml']        = 'text/x-opml',
          ['cdmia']       = 'application/cdmi-capability',
          ['xdf']         = 'application/xcap-diff+xml',
          ['gmx']         = 'application/vnd.gmx',
          ['pgm']         = 'image/x-portable-graymap',
          ['sub']         = 'text/vnd.dvb.subtitle',
          ['sldx']        = 'application/vnd.openxmlformats-officedocument.presentationml.slide',
          ['pps']         = 'application/vnd.ms-powerpoint',
          ['txf']         = 'application/vnd.mobius.txf',
          ['mgp']         = 'application/vnd.osgeo.mapguide.package',
          ['pdb']         = 'application/vnd.palm',
          ['pqa']         = 'application/vnd.palm',
          ['xspf']        = 'application/xspf+xml',
          ['cod']         = 'application/vnd.rim.cod',
          ['htke']        = 'application/vnd.kenameaapp',
          ['xps']         = 'application/vnd.ms-xpsdocument',
          ['kml']         = 'application/vnd.google-earth.kml+xml',
          ['scurl']       = 'text/vnd.curl.scurl',
          ['uvz']         = 'application/vnd.dece.zip',
          ['fh']          = 'image/x-freehand',
          ['sis']         = 'application/vnd.symbian.install',
          ['azw']         = 'application/vnd.amazon.ebook',
          ['see']         = 'application/vnd.seemail',
          ['stw']         = 'application/vnd.sun.xml.writer.template',
          ['dxf']         = 'image/vnd.dxf',
          ['sti']         = 'application/vnd.sun.xml.impress.template',

          -- aditional extensions

          ['vtt']         = 'text/vtt',
          ['crx']         = 'application/x-chrome-extension',
          ['htc']         = 'text/x-component',
          ['manifest']    = 'text/cache-manifest',
          ['buffer']      = 'application/octet-stream',
          ['m4p']         = 'application/mp4',
          ['m4a']         = 'audio/mp4',
          ['ts']          = 'video/MP2T',
          ['webapp']      = 'application/x-web-app-manifest+json',
          ['lua']         = 'text/x-lua',
          ['luac']        = 'application/x-lua-bytecode',
          ['markdown']    = 'text/x-markdown',
          ['md']          = 'text/x-markdown',
          ['mkd']         = 'text/x-markdown',
          ['ini']         = 'text/plain',
          ['mdp']         = 'application/dash+xml',
          ['map']         = 'application/json',
          ['xsd']         = 'application/xml',
          ['opus']        = 'audio/ogg',
          ['gz']          = 'application/x-gzip'
     },

     -- This contains filename overrides for certain files, like README files.
     -- Sort them in the same order as extensions.

     filenames = {
          ['COPYING']  = 'text/plain',
          ['LICENSE']  = 'text/plain',
          ['Makefile'] = 'text/x-makefile',
          ['README']   = 'text/plain'
     }
}


-- Creates a copy of the MIME types database for customization.

function mimetypes.copy (db)
    db = db or defaultdb
    return copy(db)
end


-- Guesses the MIME type of the file with the given name.
-- It is returned as a string. If the type cannot be guessed, then nil is
-- returned.

function mimetypes.guess (filename, db)
    db = db or defaultdb
    if db.filenames[filename] then
        return db.filenames[filename]
    end
    local ext = extension(filename)
    if ext then
        return db.extensions[ext]
    end
    return nil
end

return mimetypes
