-- SPDX-License-Identifier: GPL-3.0-or-later

local debug = require('debug')
local ffi = require('ffi')

-- Units
kB = 1024
MB = 1024*kB
GB = 1024*MB
-- Time
sec = 1000
second = sec
minute = 60 * sec
min = minute
hour = 60 * minute
day = 24 * hour

-- Logging
function panic(fmt, ...)
        print(debug.traceback('error occured here (config filename:lineno is '
                .. 'at the bottom, if config is involved):', 2))
        error(string.format('ERROR: '.. fmt, ...), 0)
end
function warn(fmt, ...)
        io.stderr:write(string.format(fmt..'\n', ...))
end
function log(fmt, ...)
        print(string.format(fmt, ...))
end

-- Resolver bindings
kres = require('kres')
if rawget(kres, 'str2dname') ~= nil then
	todname = kres.str2dname
end

worker.resolve_pkt = function (pkt, options, finish, init)
	options = kres.mk_qflags(options)
	local task = ffi.C.worker_resolve_start(pkt, options)

	-- Deal with finish and init callbacks
	if finish ~= nil then
		local finish_cb
		finish_cb = ffi.cast('trace_callback_f',
			function (req)
				jit.off(true, true) -- JIT for (C -> lua)^2 nesting isn't allowed
				finish(req.answer, req)
				finish_cb:free()
			end)
		task.ctx.req.trace_finish = finish_cb
	end
	if init ~= nil then
		init(task.ctx.req)
	end

	return ffi.C.worker_resolve_exec(task, pkt) == 0
end

worker.resolve = function (qname, qtype, qclass, options, finish, init)
	-- Alternatively use named arguments
	if type(qname) == 'table' then
		local t = qname
		qname = t.name
		qtype = t.type
		qclass = t.class
		options = t.options
		finish = t.finish
		init = t.init
	end
	qtype = qtype or kres.type.A
	qclass = qclass or kres.class.IN
	options = kres.mk_qflags(options)
	-- LATER: nicer errors for rubbish in qname, qtype, qclass?
	local pkt = ffi.C.worker_resolve_mk_pkt(qname, qtype, qclass, options)
	if pkt == nil then
		panic('failure in worker.resolve(); probably invalid qname "%s"', qname)
	end
	local ret = worker.resolve_pkt(pkt, options, finish, init)
	ffi.C.knot_pkt_free(pkt);
	return ret
end
resolve = worker.resolve

-- Shorthand for aggregated per-worker information
worker.info = function ()
	local t = worker.stats()
	t.pid = worker.pid
	return t
end

-- Resolver mode of operation
local current_mode = 'normal'
local mode_table = { normal=0, strict=1, permissive=2 }
function mode(m)
	if not m then return current_mode end
	if not mode_table[m] then error('unsupported mode: '..m) end
	-- Update current operation mode
	current_mode = m
	option('STRICT', current_mode == 'strict')
	option('PERMISSIVE', current_mode == 'permissive')
	return true
end

-- Trivial option alias
function reorder_RR(val)
	return option('REORDER_RR', val)
end

-- Get/set resolver options via name (string)
function option(name, val)
	local flags = kres.context().options;
	-- Note: no way to test existence of flags[name] but we want error anyway.
	name = string.upper(name) -- convenience
	if val ~= nil then
		if (val ~= true) and (val ~= false) then
			panic('invalid option value: ' .. tostring(val))
		end
		flags[name] = val;
	end
	return flags[name];
end

-- Function aliases
-- `env.VAR returns os.getenv(VAR)`
env = {}
setmetatable(env, {
	__index = function (_, k) return os.getenv(k) end
})

-- Quick access to interfaces
-- `net.<iface>` => `net.interfaces()[iface]`
-- `net = {addr1, ..}` => `net.listen(name, addr1)`
-- `net.ipv{4,6} = {true, false}` => enable/disable IPv{4,6}
setmetatable(net, {
	__index = function (t, k)
		local v = rawget(t, k)
		if v then return v
		elseif k == 'ipv6' then return not option('NO_IPV6')
		elseif k == 'ipv4' then return not option('NO_IPV4')
		else return net.interfaces()[k]
		end
	end,
	__newindex = function (t,k,v)
		if     k == 'ipv6' then return option('NO_IPV6', not v)
		elseif k == 'ipv4' then return option('NO_IPV4', not v)
		else
			local iname = rawget(net.interfaces(), v)
			if iname then t.listen(iname)
			else t.listen(v)
			end
		end
	end
})

-- Syntactic sugar for module loading
-- `modules.<name> = <config>`
setmetatable(modules, {
	__newindex = function (_, k, v)
		if type(k) == 'number' then
			k, v = v, nil
		end
		if not rawget(_G, k) then
			modules.load(k)
			k = string.match(k, '[%w_]+')
			local mod = _G[k]
			local config = mod and rawget(mod, 'config')
			if mod ~= nil and config ~= nil then
				if k ~= v then config(v)
				else           config()
				end
			end
		end
	end
})

-- Set up lua table for a C module. (Internal function.)
function modules_create_table_for_c(kr_module_ud)
	local kr_module = ffi.cast('struct kr_module **', kr_module_ud)[0]
	--- Set up the global table named according to the module.
	if kr_module.config == nil and kr_module.props == nil then
		return
	end
	local module = {}
	local module_name = ffi.string(kr_module.name)
	_G[module_name] = module

	--- Construct lua functions for properties.
	if kr_module.props ~= nil then
		local i = 0
		while true do
			local prop = kr_module.props[i]
			local cb = prop.cb
			if cb == nil then break; end
			module[ffi.string(prop.name)] =
				function (arg) -- lua wrapper around kr_prop_cb function typedef
					local arg_conv
					if type(arg) == 'table' or type(arg) == 'boolean' then
						arg_conv = tojson(arg)
					elseif arg ~= nil then
						arg_conv = tostring(arg)
					end
					local ret_cstr = cb(ffi.C.the_worker.engine, kr_module, arg_conv)
					if ret_cstr == nil then
						return nil
					end
					-- LATER(optim.): superfluous copying
					local ret_str = ffi.string(ret_cstr)
					-- This is a bit ugly, but the API is that invalid JSON
					-- should be just returned as string :-(
					local status, ret = pcall(fromjson, ret_str)
					if not status then ret = ret_str end
					ffi.C.free(ret_cstr)
					return ret
				end
			i = i + 1
		end
	end

	--- Construct lua function for config().
	if kr_module.config ~= nil then
		module.config =
			function (arg)
				local arg_conv
				if type(arg) == 'table' or type(arg) == 'boolean' then
					arg_conv = tojson(arg)
				elseif arg ~= nil then
					arg_conv = tostring(arg)
				end
				return kr_module.config(kr_module, arg_conv)
			end
	end

	--- Add syntactic sugar for get() and set() properties.
	--- That also "catches" any commands like `moduleName.foo = bar`.
	local m_index, m_newindex
	local get_f = rawget(module, 'get')
	if get_f ~= nil then
		m_index = function (_, key)
			return get_f(key)
		end
	else
		m_index = function ()
			error('module ' .. module_name .. ' does not support indexing syntax sugar')
		end
	end
	local set_f = rawget(module, 'set')
	if set_f ~= nil then
		m_newindex = function (_, key, value)
			-- This will produce a nasty error on some non-string parameters.
			-- Still, we already use it with integer values, e.g. in predict module :-/
			return set_f(key .. ' ' .. value)
		end
	else
		m_newindex = function ()
			error('module ' .. module_name .. ' does not support assignment syntax sugar')
		end
	end
	setmetatable(module, {
		-- note: the two functions only get called for *missing* indices
		__index = m_index,
		__newindex = m_newindex,
	})
end

local layer_ctx = ffi.C.kr_layer_t_static
-- Utilities internal for lua layer glue; see ../ffimodule.c
modules_ffi_layer_wrap1 = function (layer_cb)
	return layer_cb(layer_ctx.state, layer_ctx.req)
end
modules_ffi_layer_wrap2 = function (layer_cb)
	return layer_cb(layer_ctx.state, layer_ctx.req, layer_ctx.pkt)
end
modules_ffi_layer_wrap_checkout = function (layer_cb)
	return layer_cb(layer_ctx.state, layer_ctx.req, layer_ctx.pkt,
					layer_ctx.dst, layer_ctx.is_stream)
end
modules_ffi_wrap_modcb = function (cb, kr_module_ud) -- this one isn't for layer
	local kr_module = ffi.cast('struct kr_module **', kr_module_ud)[0]
	return cb(kr_module)
end

-- Return filesystem size where the cache resides.
cache.fssize = function ()
	local path = cache.current_storage or '.'
	-- As it is now, `path` may or may not include the lmdb:// prefix.
	if string.sub(path, 1, 7) == 'lmdb://' then
		path = string.sub(path, 8)
	end
	if #path == 0 then
		path = '.'
	end
	local size = tonumber(ffi.C.kr_fssize(path))
	if size < 0 then
		panic('cache.fssize(): %s', ffi.string(ffi.C.knot_strerror(size)))
	else
		return size
	end
end

cache.clear = function (name, exact_name, rr_type, chunk_size, callback, prev_state)
	if name == nil or (name == '.' and not exact_name) then
		-- keep same output format as for 'standard' clear
		local total_count = cache.count()
		if not cache.clear_everything() then
			error('unable to clear everything')
		end
		return {count = total_count}
	end
	-- Check parameters, in order, and set defaults if missing.
	local dname = kres.str2dname(name)
	if not dname then error('cache.clear(): incorrect name passed') end
	if exact_name == nil then exact_name = false end
	if type(exact_name) ~= 'boolean'
		then error('cache.clear(): incorrect exact_name passed') end

	local cach = kres.context().cache;
	local rettable = {}
	-- Apex warning.  If the caller passes a custom callback,
	-- we assume they are advanced enough not to need the check.
	-- The point is to avoid repeating the check in each callback iteration.
	if callback == nil then
		local apex_array = ffi.new('knot_dname_t *[1]')  -- C: dname **apex_array
		local ret = ffi.C.kr_cache_closest_apex(cach, dname, false, apex_array)
		if ret < 0 then
			error(ffi.string(ffi.C.knot_strerror(ret))) end
		if not ffi.C.knot_dname_is_equal(apex_array[0], dname) then
			local apex_str = kres.dname2str(apex_array[0])
			rettable.not_apex = 'to clear proofs of non-existence call '
				.. 'cache.clear(\'' .. tostring(apex_str) ..'\')'
			rettable.subtree = apex_str
		end
		ffi.C.free(apex_array[0])
	end

	if rr_type ~= nil then
		-- Special case, without any subtree searching.
		if not exact_name
			then error('cache.clear(): specifying rr_type only supported with exact_name') end
		if chunk_size or callback
			then error('cache.clear(): chunk_size and callback parameters not supported with rr_type') end
		local ret = ffi.C.kr_cache_remove(cach, dname, rr_type)
		if ret < 0 then error(ffi.string(ffi.C.knot_strerror(ret))) end
		return {count = 1}
	end

	if chunk_size == nil then chunk_size = 100 end
	if type(chunk_size) ~= 'number' or chunk_size <= 0
		then error('cache.clear(): chunk_size has to be a positive integer') end

	-- Do the C call, and add chunk_size warning.
	rettable.count = ffi.C.kr_cache_remove_subtree(cach, dname, exact_name, chunk_size)
	if rettable.count == chunk_size then
		local msg_extra = ''
		if callback == nil then
			msg_extra = '; the default callback will continue asynchronously'
		end
		rettable.chunk_limit = 'chunk size limit reached' .. msg_extra
	end

	-- Default callback function: repeat after 1ms
	if callback == nil then callback =
		function (cbname, cbexact_name, cbrr_type, cbchunk_size, cbself, cbprev_state, cbrettable)
			if cbrettable.count < 0 then error(ffi.string(ffi.C.knot_strerror(cbrettable.count))) end
			if cbprev_state == nil then cbprev_state = { round = 0 } end
			if type(cbprev_state) ~= 'table'
				then error('cache.clear() callback: incorrect prev_state passed') end
			cbrettable.round = cbprev_state.round + 1
			if (cbrettable.count == cbchunk_size) then
				event.after(1, function ()
						cache.clear(cbname, cbexact_name, cbrr_type, cbchunk_size, cbself, cbrettable)
					end)
			elseif cbrettable.round > 1 then
				log('[cache] asynchonous cache.clear(\'' .. cbname .. '\', '
				    .. tostring(cbexact_name) .. ') finished')
			end
			return cbrettable
		end
	end
	return callback(name, exact_name, rr_type, chunk_size, callback, prev_state, rettable)
end
-- Syntactic sugar for cache
-- `cache[x] -> cache.get(x)`
-- `cache.{size|storage} = value`
setmetatable(cache, {
	__index = function (t, k)
		local res = rawget(t, k)
		if not res and not rawget(t, 'current_size') then return res end
		-- Beware: t.get returns empty table on failure to find.
		-- That would be confusing here (breaking kresc), so return nil instead.
		res = t.get(k)
		if res and next(res) ~= nil then return res else return nil end
	end,
	__newindex = function (t,k,v)
		-- Defaults
		local storage = rawget(t, 'current_storage')
		if not storage then storage = 'lmdb://' end
		local size = rawget(t, 'current_size')
		if not size then size = 10*MB end
		-- Declarative interface for cache
		if     k == 'size'    then t.open(v, storage)
		elseif k == 'storage' then t.open(size, v) end
	end
})

-- Make sandboxed environment
local function make_sandbox(defined)
	local __protected = { worker = true, env = true, modules = true, cache = true, net = true, trust_anchors = true }

	-- Compute and export the list of top-level names (hidden otherwise)
	local nl = ""
	for n in pairs(defined) do
		nl = nl .. n .. "\n"
	end

	return setmetatable({ __orig_name_list = nl }, {
		__index = defined,
		__newindex = function (_, k, v)
			if __protected[k] then
				for k2,v2 in pairs(v) do
					defined[k][k2] = v2
				end
			else
				defined[k] = v
			end
		end
	})
end

-- Compatibility sandbox
_G = make_sandbox(getfenv(0))
setfenv(0, _G)

-- Load default modules
trust_anchors = require('trust_anchors')
modules.load('ta_update')
modules.load('ta_signal_query')
modules.load('policy')
modules.load('priming')
modules.load('detect_time_skew')
modules.load('detect_time_jump')
modules.load('ta_sentinel')
modules.load('edns_keepalive')
modules.load('refuse_nord')
modules.load('watchdog')

-- Load keyfile_default
trust_anchors.add_file('/usr/share/dnssec-root/trusted-key.key', true)

-- Interactive command evaluation
function eval_cmd(line, raw)
	-- Compatibility sandbox code loading
	local function load_code(code)
	    if getfenv then -- Lua 5.1
	        return loadstring(code)
	    else            -- Lua 5.2+
	        return load(code, nil, 't', _ENV)
	    end
	end
	local err, chunk
	chunk, err = load_code(raw and 'return '..line or 'return table_print('..line..')')
	if err then
		chunk, err = load_code(line)
	end
	if not err then
		return chunk()
	else
		error(err)
	end
end

-- Pretty printing

local function funcsign(f)
-- thanks to AnandA777 from StackOverflow! Function funcsign is adapted version of
-- https://stackoverflow.com/questions/51095022/inspect-function-signature-in-lua-5-1
	assert(type(f) == 'function', "bad argument #1 to 'funcsign' (function expected)")
	local debuginfo = debug.getinfo(f)
	if debuginfo.what == 'C' then  -- names N/A
		return '(?)'
	end

	local func_args = {}
	pcall(function()
		local oldhook
		local delay = 2
		local function hook()
			delay = delay - 1
			if delay == 0 then  -- call this only for the introspected function
				-- stack depth 2 is the introspected function
				for i = 1, debuginfo.nparams do
					local k = debug.getlocal(2, i)
					table.insert(func_args, k)
				end
				if debuginfo.isvararg then
					table.insert(func_args, "...")
				end
				debug.sethook(oldhook)
				error('aborting the call to introspected function')
			end
		end
		oldhook = debug.sethook(hook, "c")  -- invoke hook() on function call
		f(unpack({}))  -- huh?
	end)
	return "(" .. table.concat(func_args, ", ") .. ")"
end

function table_print (tt, indent, done)
	done = done or {}
	indent = indent or 0
	local result = ""
	-- Ordered for-iterator for tables with tostring-able keys.
	local function ordered_iter(unordered_tt)
		local keys = {}
		for k in pairs(unordered_tt) do
			table.insert(keys, k)
		end
		table.sort(keys, function (a, b) return tostring(a) < tostring(b) end)
		local i = 0
		return function()
			i = i + 1
			if keys[i] then
				return keys[i], unordered_tt[keys[i]]
			end
		end
	end
	-- Convert to printable string (escape unprintable)
	local function printable(value)
		value = tostring(value)
		local bytes = {}
		for i = 1, #value do
			local c = string.byte(value, i)
			if c >= 0x20 and c < 0x7f then table.insert(bytes, string.char(c))
			else                           table.insert(bytes, '\\'..tostring(c))
			end
			if i > 80 then table.insert(bytes, '...') break end
		end
		return table.concat(bytes)
	end
	if type(tt) == "table" then
		for key, value in ordered_iter(tt) do
			result = result .. string.rep (" ", indent)
			if type (value) == "table" and not done [value] then
				done [value] = true
				result = result .. string.format("[%s] => {\n", printable (key))
				result = result .. table_print (value, indent + 4, done)
				result = result .. string.rep (" ", indent)
				result = result .. "}\n"
			elseif type (value) == "function" then
				result = result .. string.format("[%s] => function %s%s: %s\n",
				         tostring(key), tostring(key), funcsign(value),
					 string.sub(tostring(value), 11))
			else
				result = result .. string.format("[%s] => %s\n",
				         tostring (key), printable(value))
			end
		end
	else  -- not a table
		local tt_str
		if type(tt) == "function" then
			tt_str = string.format("function%s: %s\n", funcsign(tt),
						string.sub(tostring(tt), 11))
		else
			tt_str = tostring(tt)
		end
		result = result .. tt_str .. "\n"
	end
	return result
end

-- This extends the worker module to allow asynchronous execution of functions and nonblocking I/O.
-- The current implementation combines cqueues for Lua interface, and event.socket() in order to not
-- block resolver engine while waiting for I/O or timers.
--
local has_cqueues, cqueues = pcall(require, 'cqueues')
if has_cqueues then

	-- Export the asynchronous sleep function
	worker.sleep = cqueues.sleep

	-- Create metatable for workers to define the API
	-- It can schedule multiple cqueues and yield execution when there's a wait for blocking I/O or timer
	local asynchronous_worker_mt = {
		work = function (self)
			local ok, err, _, co = self.cq:step(0)
			if not ok then
				warn('[%s] error: %s %s', self.name or 'worker', err, debug.traceback(co))
			end
			-- Reschedule timeout or create new one
			local timeout = self.cq:timeout()
			if timeout then
				-- Throttle timeouts to avoid too frequent wakeups
				if timeout == 0 then timeout = 0.00001 end
				-- Convert from seconds to duration
				timeout = timeout * sec
				if not self.next_timeout then
					self.next_timeout = event.after(timeout, self.on_step)
				else
					event.reschedule(self.next_timeout, timeout)
				end
			else -- Cancel running timeout when there is no next deadline
				if self.next_timeout then
					event.cancel(self.next_timeout)
					self.next_timeout = nil
				end
			end
		end,
		wrap = function (self, f)
			self.cq:wrap(f)
		end,
		loop = function (self)
			self.on_step = function () self:work() end
			self.event_fd = event.socket(self.cq:pollfd(), self.on_step)
		end,
		close = function (self)
			if self.event_fd then
				event.cancel(self.event_fd)
				self.event_fd = nil
			end
		end,
	}

	-- Implement the coroutine worker with cqueues
	local function worker_new (name)
		return setmetatable({name = name, cq = cqueues.new()}, { __index = asynchronous_worker_mt })
	end

	-- Create a default background worker
	worker.bg_worker = worker_new('worker.background')
	worker.bg_worker:loop()

	-- Wrap a function for asynchronous execution
	function worker.coroutine (f)
		worker.bg_worker:wrap(f)
	end
else
	-- Disable asynchronous execution
	local function disabled ()
		error('Lua library cqueues is required for asynchronous execution (luaJIT requires library for Lua 5.1)')
	end
	worker.sleep = disabled
	worker.map = disabled
	worker.coroutine = disabled
	worker.bg_worker = setmetatable({}, { __index = disabled })
end
