package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.30.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.30.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`/@`!````;B4```````!``````````$BQ````
M`````````$``.``*`$``&``7``8````$````0`````````!``````````$``
M````````,`(````````P`@````````@``````````P````0```!P`@``````
M`'`"````````<`(````````9`````````!D``````````0`````````!````
M!``````````````````````````````````````8`````````!@`````````
M$`````````$````%`````"``````````(``````````@````````[S0`````
M``#O-``````````0`````````0````0`````8`````````!@`````````&``
M```````(0P````````A#`````````!`````````!````!@```""L````````
M(+P````````@O````````$`$````````Z!0`````````$`````````(````&
M````2*P```````!(O````````$B\````````H`$```````"@`0````````@`
M````````4.5T9`0```!4G0```````%2=````````5)T```````#D````````
M`.0`````````!`````````!1Y71D!@``````````````````````````````
M```````````````````````````0`````````%+E=&0$````(*P````````@
MO````````""\````````X`,```````#@`P````````$`````````+VQI8B]L
M9"UM=7-L+7@X-E\V-"YS;RXQ```````````1````0`````0````(````(!P0
M``5`!((H`%`$D(8`A((A`"!)`@0)N((90;H$`@I`````00```$4```!)````
M2@```%````!1````4P````````!7````6````%D```!;````70```%\`````
M`````````*/I&01J?YI\ZM/O#L[>1RUOQ=\?+L6;MKB-\0[49-1\(?_D#MEQ
M6!SZH6Q@PN4_++(!BV]"^M>'XD?&`;OCDGQ5PSS2Q+F<0&>OS&8X\HL<Q(D%
MD(@'<--#1=7L,V+;[16,)<Q6I)N6I]VC:WZIG)O19<YMX*<""FL:GE`*BA!7
MH^#[20````````````````````````````````P````2````````````````
M`````````+,!```2`````````````````````````%4!```2````````````
M`````````````!L````2`````````````````````````)(!```2````````
M`````````````````",````2`````````````````````````"H````2````
M`````````````````````.8!```2`````````````````````````)X!```2
M`````````````````````````#,````@`````````````````````````$L`
M```2`````````````````````````%$````2````````````````````````
M`%@````2`````````````````````````.8"```2````````````````````
M`````-<"```2`````````````````````````&L"```2````````````````
M``````````P"```2`````````````````````````%\````2````````````
M`````````````*`"```2`````````````````````````&8````2````````
M`````````````````&X````2`````````````````````````(D"```2````
M`````````````````````'8````2`````````````````````````'P````2
M`````````````````````````(4````@`````````````````````````-`"
M```2`````````````````````````*$````2````````````````````````
M`*8````2`````````````````````````*<"```2````````````````````
M`````#D"```2`````````````````````````*T````2````````````````
M`````````"L"```2`````````````````````````#T#```2````````````
M`````````````%T"```2``````````````````````````4#```2````````
M`````````````````+0````2`````````````````````````/T"```2````
M`````````````````````&,!```2`````````````````````````,P````@
M`````````````````````````.(````2`````````````````````````,@"
M```2`````````````````````````$D"```2````````````````````````
M`.<````2`````````````````````````.T````@````````````````````
M`````)0"```2``````````````````````````<!```2````````````````
M`````````(,!```2``````````````````````````P!```2````````````
M`````````````'D"```2`````````````````````````'(!```2````````
M`````````````````!(!```2`````````````````````````",!```2````
M``````````````````````<"```2`````````````````````````'<````2
M`````````````````````````!@"```2`````````````````````````%("
M```2`````````````````````````/8"```2````````````````````````
M`"D!```2`````````````````````````.X"```2````````````````````
M``````("```2`````````````````````````,$!```2````````````````
M`````````#T!```2`````````````````````````$X!```2````````````
M`````````````!<#```A`!4`:,`````````(`````````!(#```2``H`N",`
M``````"V`0```````-,!```2``L`YU0````````!`````````'\#```2``H`
MMC$```````#-!0```````"L$```2``H`P3H```````!/`````````%P#```2
M``H`LRP````````%`````````.D#```2``<``"`````````!`````````*(#
M```2``H`@S<````````^`P````````\$```2``H`N"P```````"R`0``````
M`-H#```0`!4`8,```````````````````&T#```2``H`'3$```````"9````
M`````(P#```2``H`62L````````K`````````)0#```2``H`PC````````!$
M`````````+@#```2``H`=5(```````#W`````````,T#```2``H`:BX`````
M```6`````````"8$```0`!4`"-$``````````````````"(#```2``H`A"4`
M```````D`````````!@#```1`!4`:,`````````(`````````.\#```2``H`
M@"X```````"C`````````!0````1`!$`0+P````````(`````````!D#```A
M`!4`:,`````````(`````````,,#```1`!0`6,`````````(``````````@$
M```0`!0`8,```````````````````-\#```0``H`;B4`````````````````
M`.8#```2``H`)U(```````!.`````````/<!```1`!4`<,`````````$````
M`````!H#```A`!4`:,`````````(`````````"L#```2``H`!C$````````7
M`````````+T````B`````````````````````````$8#```2``H`R"L`````
M``#K`````````!P$```2``H`;%,```````!9`0```````-D!```1`!4`8,``
M```````!`````````/L#```2``H`(R\```````"?`0````````!L:6)P97)L
M+G-O`&]P96YD:7(`<W1D97)R`')E86QL;V,`86-C97-S`'-N<')I;G1F`%]?
M9&5R96=I<W1E<E]F<F%M95]I;F9O`&QS965K`&=E='!I9`!S=')C;7``=6YL
M:6YK`&UE;6UO=F4`<W1R;F-M<`!L<W1A=`!C;&]S961I<@!?251-7V1E<F5G
M:7-T97)434-L;VYE5&%B;&4`;W!E;@!S=')S='(`<W1R8VAR`')E861L:6YK
M`%]?8WAA7V9I;F%L:7IE`%]?<F5G:7-T97)?9G)A;65?:6YF;P!E>&ET`&-L
M;W-E`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`<F5A9`!R;61I<@!?7V5R
M<FYO7VQO8V%T:6]N`&UK9&ER`'!T:')E861?9V5T<W!E8VEF:6,`7U]S=&%C
M:U]C:&M?9F%I;`!G971U:60`4&5R;%]S>7-?=&5R;0!097)L7W-Y<U]I;FET
M,P!097)L7V%T9F]R:U]L;V-K`'!E<FQ?8V]N<W1R=6-T`%!E<FQ?9V5T7W-V
M`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E<FQ?879?<VAI9G0`4&5R;%]S=E\R
M<'9?9FQA9W,`7V9I;FD`4$Q?9&]?=6YD=6UP`%!E<FQ?;F5W6%-?9FQA9W,`
M4$Q?=&AR7VME>0!P97)L7V9R964`4&5R;%]M9U]S970`4&5R;%]A=&9O<FM?
M=6YL;V-K`%!E<FQ?<W9?<V5T<'8`4&5R;%]G=E]F971C:'!V`'!E<FQ?<G5N
M`%!E<FQ?;F5W6%,`<&5R;%]D97-T<G5C=`!097)L7V%V7V9E=&-H`&)O;W1?
M1'EN84QO861E<@!P97)L7W!A<G-E`%!E<FQ?<V%F97-Y<VUA;&QO8P!P97)L
M7V%L;&]C`&QI8F,N;75S;"UX.#9?-C0N<V\N,0!F<')I;G1F`'-T<F1U<`!P
M=&AR96%D7V%T9F]R:P!R96%D9&ER`'-P<FEN=&8`<W1R=&]K`'-T<FYC<'D`
M7U]L:6)C7W-T87)T7VUA:6X`7U]?96YV:7)O;@!?<W1A<G1?8P!G971?=7-E
M<FYA;65?9G)O;5]G971P=W5I9`!P87)?8W5R<F5N=%]E>&5C7W!R;V,`<&%R
M7V-U<G)E;G1?97AE8P!P87)?<V5T=7!?;&EB<&%T:`!P87)?;6MT;7!D:7(`
M>'-?:6YI=`!P87)?96YV7V-L96%N`%A37TEN=&5R;F%L<U]005)?0D]/5`!S
M:&%?=7!D871E`&UY7W!A<E]P;`!P87)?8F%S96YA;64`7U]B<W-?<W1A<G0`
M<VAA7VEN:70`<&%R7V1I<FYA;64`<&%R7VEN:71?96YV`%]E9&%T80!P87)?
M9FEN9'!R;V<`<VAA7V9I;F%L`%]E;F0`<&%R7V-L96%N=7``+W5S<B]L:6(O
M<&5R;#4O8V]R95]P97)L+T-/4D4``````,`````````(``````````#`````
M````(,`````````(``````````!@````````*,`````````(`````````/)@
M````````,,`````````(``````````M@````````.,`````````(````````
M``1A````````0,`````````(`````````!-@````````6,`````````(````
M`````+MB````````P+\````````&````"@``````````````R+\````````&
M````&0``````````````T+\````````&````7```````````````V+\`````
M```&````)P``````````````X+\````````&````+```````````````Z+\`
M```````&````,0``````````````\+\````````&````,@``````````````
M^+\````````&````-P``````````````0+P````````%````4P``````````
M````8,`````````%````7P``````````````:,`````````%````40``````
M````````<,`````````%````60```````````````+X````````'`````0``
M````````````"+X````````'`````@``````````````$+X````````'````
M`P``````````````&+X````````'````!```````````````(+X````````'
M````!0``````````````*+X````````'````!@``````````````,+X`````
M```'````!P``````````````.+X````````'````"```````````````0+X`
M```````'````"0``````````````2+X````````'````"P``````````````
M4+X````````'````#```````````````6+X````````'````#0``````````
M````8+X````````'````#@``````````````:+X````````'````#P``````
M````````<+X````````'````$```````````````>+X````````'````$0``
M````````````@+X````````'````$@``````````````B+X````````'````
M$P``````````````D+X````````'````%```````````````F+X````````'
M````%0``````````````H+X````````'````%@``````````````J+X`````
M```'````%P``````````````L+X````````'````&```````````````N+X`
M```````'````&@``````````````P+X````````'````&P``````````````
MR+X````````'````'```````````````T+X````````'````'0``````````
M````V+X````````'````'@``````````````X+X````````'````'P``````
M````````Z+X````````'````(```````````````\+X````````'````(0``
M````````````^+X````````'````(@```````````````+\````````'````
M(P``````````````"+\````````'````)```````````````$+\````````'
M````)0``````````````&+\````````'````)@``````````````(+\`````
M```'````*```````````````*+\````````'````*0``````````````,+\`
M```````'````*@``````````````.+\````````'````*P``````````````
M0+\````````'````+0``````````````2+\````````'````+@``````````
M````4+\````````'````+P``````````````6+\````````'````,```````
M````````8+\````````'````,P``````````````:+\````````'````-```
M````````````<+\````````'````-0``````````````>+\````````'````
M-@``````````````@+\````````'````.```````````````B+\````````'
M````.0``````````````D+\````````'````.@``````````````F+\`````
M```'````.P``````````````H+\````````'````/```````````````J+\`
M```````'````/0``````````````L+\````````'````/@``````````````
MN+\````````'````/P``````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``!0Z((&``#HNC0``%C#````_S7:G0``_R7<G0``#Q]``/\EVIT``&@`````
MZ>#_____)=*=``!H`0```.G0_____R7*G0``:`(```#IP/____\EPIT``&@#
M````Z;#_____);J=``!H!````.F@_____R6RG0``:`4```#ID/____\EJIT`
M`&@&````Z8#_____):*=``!H!P```.EP_____R6:G0``:`@```#I8/____\E
MDIT``&@)````Z5#_____)8J=``!H"@```.E`_____R6"G0``:`L```#I,/__
M__\E>IT``&@,````Z2#_____)7*=``!H#0```.D0_____R5JG0``:`X```#I
M`/____\E8IT``&@/````Z?#^____)5J=``!H$````.G@_O___R52G0``:!$`
M``#IT/[___\E2IT``&@2````Z<#^____)4*=``!H$P```.FP_O___R4ZG0``
M:!0```#IH/[___\E,IT``&@5````Z9#^____)2J=``!H%@```.F`_O___R4B
MG0``:!<```#I</[___\E&IT``&@8````Z6#^____)1*=``!H&0```.E0_O__
M_R4*G0``:!H```#I0/[___\E`IT``&@;````Z3#^____)?J<``!H'````.D@
M_O___R7RG```:!T```#I$/[___\EZIP``&@>````Z0#^____)>*<``!H'P``
M`.GP_?___R7:G```:"````#IX/W___\ETIP``&@A````Z=#]____)<J<``!H
M(@```.G`_?___R7"G```:",```#IL/W___\ENIP``&@D````Z:#]____);*<
M``!H)0```.F0_?___R6JG```:"8```#I@/W___\EHIP``&@G````Z7#]____
M)9J<``!H*````.E@_?___R62G```:"D```#I4/W___\EBIP``&@J````Z4#]
M____)8*<``!H*P```.DP_?___R5ZG```:"P```#I(/W___\E<IP``&@M````
MZ1#]____)6J<``!H+@```.D`_?___R5BG```:"\```#I\/S___\E6IP``&@P
M````Z>#\____)5*<``!H,0```.G0_/___R5*G```:#(```#IP/S___\E0IP`
M`&@S````Z;#\____)3J<``!H-````.F@_/___R4RG```:#4```#ID/S___\E
M*IP``&@V````Z8#\____)2*<``!H-P```.EP_/___R4:G```9I#_)2*<``!F
MD/\E(IP``&:0051(@^P@B7PD'$B-?"0<2(ET)!!(C70D$$B)5"0(2(U4)`CH
M</[__TB+-1&<``!(BST"G```2(GRZ/K\__^`/6.<````=37HO/W__TB)!?6L
M``!(B<=(A<!U"K\!````Z$/^___HGO[__XL]2)P``.@3____QH#1"0```(L]
M-IP``.@!____BSTKG```@(@Z!````NCO_O__@(@Z!````8M$)!R->`5(8_](
MP><#Z#7^__](BWPD$(M4)!S'!7*L```$````2(G!2(D%8*P``$B+!TB)`4B-
M!2`^``!(B4$(2(L%N)L``$B)01!(C04-/@``2(E!&#'`2/_`.<)^$4R+!,>-
M<`1(8_9,B43Q^.OHC4($13'`@\(#2(L])ZP``$B82(TUH@8``$C'1,'X````
M`.AP_/__08G$A<!U#TB+/0*L``#H??W__T&)Q$B+/?.K``#H_OS__S'`2(T]
MD3L``.@8`@``2(7`=2Y(@SW+JP```'432(T]*SL``.C]`0``2(D%MJL``$B+
M/:^K``!(A?]T!>AN%0``2(L]IJL``.@!_O__Z-SZ__](@\0@1(G@05S#2#'M
M2(GG2(TUS98``$B#Y/#H`````$B-5PA(BS=%,<E,C052+P``2(T-9/K__TB-
M/17^___I>/S__TB-/;&:``!(C06JF@``2#GX=`Y(BP4&F@``2(7`=`+_X,-(
MC3V/F@``2(TUB)H``+D"````2"G^2,'^`TB)\$B92/?Y2(G&2(7`=`Y(BP7E
MF0``2(7`=`+_X,.`/7::````=7M52(,]NYD```!(B>5!5%-T#$B+/=R9``#H
M?_W__TB-!0"6``!(C1T!E@``2"G#28G$2,'[`TC_RTB+!3V:``!(.=AS$$C_
MP$B)!2Z:``!!_Q3$Z^3H0____TB#/5.9````=`Q(C3T:>```Z"7]__];05S&
M!?N9```!7</#2(,]2)D```!T&%5(C34&F@``2(T][W<``$B)Y>@'_?__7>D;
M____05=!5D%5051%,>154U%(A?]T9TR+-9^9``!-A?9T6TF)_4F)]TB)^(H0
MA-)T"H#Z/70%2/_`Z_!$*>A,B?5(8]A,BV4`387D=#!(B=I,B>Y,B>?H2?K_
M_X7`=1A!@#P</7413"GU38UD'`%(P?T#08DOZP9(@\4(Z\=:3(G@6UU!7$%=
M05Y!7\-(@^P89$B+!"4H````2(E$)`@QP$B-="0$Z%G___](BU0D"&1(,Q0E
M*````'0%Z!7\__](@\08PU532(G[2(/L&&1(BP0E*````$B)1"0(,<!(C6PD
M!#'`2(GN2(G?Z!3___](A<!T(DAC5"0$2(L%O)@``$B-!-!(BU`(2(D02(72
M=-)(@\`(Z^Y(BT0D"&1(,P0E*````'0%Z*G[__](@\086UW#059!54%454B)
M_5-(B?-(@^P09$B+!"4H````2(E$)`@QP(`^/74#2/_#28/._T4Q[4B-="0$
M2(G?1(GH3(GQ\JY(B>](]]%,C6'_Z('^__](B<)(A<!T,TB)QTR)\42)Z/*N
M28G-2??52?_-33GE#X*E````,<"*#`.(#`)(_\"$R77S,<#I#0$``$R+-?B7
M```QP$&)Q4C_P$F#?,;X`'7R08UU`DAC]DC!Y@.#/4ZH````=!Q,B??HM/?_
M_TB)!<67``!(A<!U.(/(_^G&````QP4FJ````0```$B)]^AN^/__2(7`=.%)
M8]5,B?9(B<=(P>(#Z&?X__](B06(EP``2(L%@9<``$ECU4C'1-`(`````$2)
M;"0$2(GHBA"$TG0*@/H]=`5(_\#K\$@IZ$AC3"0$2(L53I<``$B828U\!`),
MC2S*Z`;X__])B44`2(7`#X1Q____2&-4)`1(BP4EEP``2(L$T(I5`$C_Q4B)
MP4C_P(A0_X#Z/70$A-)UZ,8!/3'2B@P3B`P02/_"A,EU\^GL_O__2(M<)`AD
M2#,<)2@```!T!>CU^?__2(/$$%M=05Q!74%>PT%6055!5%532('LH````&1(
MBP0E*````$B)A"28````,<!(A?\/A.P```"`/P!(B?T/A.````#H3/;__TB#
MR_])B<5(A<`/A,L```!,B>_H]/;__TB%P`^$J@```$R-8!-(C348-@``3(GG
MZ,CV__^%P'382(TU!#8``$R)Y^BU]O__A<!TQ3'`2(GO2(G9\JY,B>=(B<I(
MB=GRKDCWTDB)R$CWT(T\$$AC_^CH]O__3(GA2(GJ2(TUQ34``$F)QDB)QS'`
MZ.[X__],B?=(C70D".BA^/__3(GW_\!T%XM$)"`E`/```#T`0```=0?H#?__
M_^L%Z([V__],B??H9OC__^E%____3(GOZ-GV__](B>_H(?C__TB+A"28````
M9$@S!"4H````=`7HN?C__TB!Q*````!;74%<05U!7L-(@>RX````9$B+!"4H
M````2(F$)*@````QP$B-="082(E\)`CH</;__X7`=3.+5"0P2(M\)`B!X@#P
M``"!^@!```!T"('Z`*```'46O@(```#H0_7__X7`#Y3`#[;`ZP(QP$B+C"2H
M````9$@S#"4H````=`7H+_C__TB!Q+@```##4(L]$)4``.C;]___2(L5?)0`
M`$%82(T-TW$``$B)QTB--:$T``#IG/?__U"+/>64``#HL/?__TB)Q^C!____
MBSW2E```Z)WW__],C04*-P``13')05I(B<=,B<%(C17'"P``2(TU>30``.G(
M]/__05154TB![!`@``!D2(L$)2@```!(B80D""```#'`3(UD)`;HT/3__TR-
M!5HT``"^`1```$R)YXG!2(T5330``#'`Z''T__^#^/]U!#'`ZW=(8]!(@?H`
M$```=@(/"X7`>.I(C:PD!Q```+H`$```3(GG2(GNZ.[U__](B<.%P'C+C7@!
M2&/_Z-ST__](A<!TNTACTT@YZ',)2(T,$$@YS7*Z2#GH=@I(C4P5`$@YR'*K
M@?L!$```?Z.)T4B)QTB)[O.DQ@00`$B+M"0((```9$@S-"4H````=`7HV?;_
M_TB!Q!`@``!;74%<P^D0____05=!5D%528G]2(T]F3,``$%454B)]5-(@>RH
M$```9$B+!"4H````2(F$))@0```QP.A*^O__OB\```!,B>](B</HXO3__TB%
MP'063(GN2(T]73,``#'`Z,KZ___I)`$``#'`3(GO2(TU4C,``$B#R?_RKDB)
M[TR-/>,R``!(]]%$C7'_Z._U__])B<1-A>1TND&`/"0`=0--B?Q(A=MT$TR)
MYDB)W^AM\___A<`/A+P````QP$B#R?],B>?RKDCWT8U!_TACT$&`?!3_+W4,
M_\A(8]!!Q@04`.OI0HU$,`$]_P\```^/8/___TB-K"28````38GI3(GA,<!,
MC06_,@``2(T5G#,``+X`$```2(GOZ++R____P#T`$```=@(/"TB-="0(2(GO
MZ"KU__^%P'4]BT0D("4`\```/0"```!U+;X!````2(GOZ&GR__^%P'4<2(GN
M2(T]53(``.C$^?__2(GOZ&[S__])B<7K%DB--4DR```Q_^C[]/__28G$Z0?_
M__](BX0DF!```&1(,P0E*````'0%Z"OU__](@<2H$```3(GH6UU!7$%=05Y!
M7\-(B?B*%X32=`Q(_\>`^B](#T3'Z^[#2(7_=$^`/P!(B?YT1TB#R?\QP/*N
M2/?12(U4#OY(.=9S(8`Z+W4*2/_*Z_%(.=9S$HH*2(G02/_*@/DO=>Y(B<+K
M(D@YUG4=@#XO2(T]IS$``'4'2(T]/3$``.FP\O__@#HO=0A(_\I(.=9R\T@I
M\DB-0@)(/0`0``!W)$C_PDB!^@`0``!V`@\+4$B-/<Z1``#H*?/__TB-!<*1
M``!:PS'`PTB#[#A(C37R;0``N2(```!D2(L$)2@```!(B40D*#'`2(U\)`;S
MI$B-/2HQ``#H&OC__S'`2(T])3$``.@,^/__,<!(C3T?,0``Z/[W__\QP$B-
M/1HQ``#H\/?__S'`2(T]$S$``.CB]___,<!(C3T5,0``Z-3W__\QP$B-/;PP
M``#HQO?__S'`2(T]!3$``.BX]___,<!(C3T!,0``Z*KW__\QP$B-/?TP``#H
MG/?__S'`2(T]C3```.B.]___,<!(C3WK,```Z$CW__](A<!T$4B)QDB-/<,P
M```QP.C8]___2(T]VC```#'`Z"3W__](B<9(A<!T#DB-/=TO```QP.BT]___
M2(T]R#```#'`Z`#W__](C3TB,```2(G&2(7`=1U(C3V[,```,<#HX_;__TB)
MQDB%P'0+2(T]5#```#'`ZR9(C7PD%[H*````2(TUH3```.C2\/__A<!U$4B-
M="0A2(T]*C```.A+]___,<!(C36),```2(T]^2\``.@V]___2(M$)"AD2#,$
M)2@```!T!>C#\O__2(/$.,.#/8>/``#_=3112(T]Y2\``#'`Z%[V__\QTDB%
MP'00B@`\,`^5P3'2A,`/E<(ARHD568\``(L%4X\``%K#BP5+CP``PU'HA/+_
M_XG'Z.WP__](A<!T`TB+`%K#054QP$%428G\2(T]^B\``%7H!?;__TR)YDB%
MP'1B@#@`2(G%=%I(B<?H9?#__TB%P'5@2(/._TR)YTB)\?*N2(GO2(G*2(GQ
M\JY(]])(B<A(]]!(C3P"Z+?O__](C37=+P``28GH3(GB28G%2(G'2(T-Y"X`
M`#'`Z+;Q__],B>Y=2(T]@B\``$%<,<!!7>DN]O__74%<05W#05=(C35AC@``
MN0P```!!5D%505152(G]4TB![%B!``!D2(L$)2@```!(B80D2($``#'`2(T%
M3B\``$B-?"1(2,=$)"@`````\Z5(B40D&$B-!0PN``!(C3U.+@``2(E$)"!(
MC04D+P``2(E$)#!(C04=+P``2(E$)#@QP$C'1"1``````.CX]/__2(7`=""`
M.`!)B<1T&$B)Q^C(_O__3(GGZ#/O__])B<3I\00``#'`3(UD)##HE?[__TB)
MPTB%P'4Y28L\)$B%_W0I,<#HK_3__TB)QTB%P'4&28/$".OB@#@`=/7H[^[_
M_TB)PTB%P'3HZP=(C1V%+@``,<!(B=],C2V3+@``2(/)__*N2/?12(U\"?_H
M7^[__TF)QDF)Q`^V$X32=!9,B>=,B>XQP$C_P^AB\/__28/$`NOC2(U<)$A,
MBR--A>1T+DR)YS'`Z"WT__])B<1(A<!T(X`X`'0>2(G'Z+?W__^%P'023(GG
MZ&3N__])B<1(C5PD&.LP2(/#".N]3(LK387M="=!@'T``'0@3(GOZ(3W__^%
MP'0+3(GOZ#'N__])B<1(@\,(387D=-%(@\[_,<!,B>=(B?'RKDR)]TB)RDB)
M\?*N2/?22(G(2/?01(V\$`($``!-8_],B?_HD.W__TV)\4R-!:PM``!(C0W(
M+```2(G'28G%3(GB,<!(C368+0``Z(CO__^^P`$``$R)[^@;[____\!U+.@"
M[___1(L`08/X$70>2(M5`$B+/4Z(``!,B>DQP$B--68M``#H;>[__^LP2(UT
M)'A,B>_H7NW__T&)QH7`="1(BU4`2(L]'(@``$R)Z3'`2(TU;RT``.@[[O__
M13'DZ1D#``"+A"20````)0#P```]`$```'7)BYPDE````.@T[___.<-UN8N$
M))`````E_P$``#W``0``=:9,B?_HM>S__TB-/;`L``!)B<0QP.BL\O__2(G'
M2(7`=09,BWT`ZQGH\>S__TB+?0!(B<;H#?C__TF)QTB%P'3A2(/)_S'`3(G_
M\JY(]]%(C5G_@_L#?F-(8\-(C34>+0``28U\!_SH\NO__X7`=4N#^P1T#H/K
M!$ACVT&`?!__+W4X2(M="$B%VW0O,<!(@\G_2(G?\JY(]]%(C4'_2(/X`W87
M2(U\"_M(C375+```Z*GK__^%P$P/1/LQP.B.^___A<`/A6D!```Q]DR)_^A:
M[/__B<6%P`^$50$``+H"````2,?&[O___XG'Z$WK__](C9PD2`$``+H&````
MB>](B=[H-NW__X"\)$@!````=6.`O"1)`0``0W59@+PD2@$``$%U3X"\)$L!
M``!#=46`O"1,`0``2'4[@+PD30$``$5U,;H"````2,?&QO___XGOZ.?J__^Z
M*0```$B)WHGOZ-CL__],C0W5+```28G8Z9X````QTC'VB>_HONK__S'`Z"X<
M``!)B<:Z`(```$B)WHGOZ*7L__](B<*%P'X-2(G>3(GWZ%@<``#KW8GO3(V\
M)`L!``!(C:PD'P$``#';Z%CL__],B?9,B?],C34,*P``Z"(=``!)B>A!#[84
M'TB)[S'`3(GV2/_#3(E$)`A(@\4"Z.7L__](@_L43(M$)`AUU\:$)$<!````
M3(T-+RP``$B-#>TI``!,B>I,B><QP$B--6$K``#HL.S__^F0````2(TU>2H`
M`$B-/04J```QP.@D\?__2(T=\RL``.CZZ?__3(T-YRL``$R)ZDB-#:(I``!!
MB<!(C34J*P``3(GG,<#H9>S__[[``0``3(GGZ/CK____P'4YZ-_K__^#.!%U
M+^BUZ?__0?_&3(GJ3(GG08G`4#'`2(T-6"D``%-%B?%(C37M*@``Z!_L__]:
M6>NV3(GOZ,/K__](C3T=*0``3(GF,<#HD/#__TR)Y^C'^?__2(N$)$B!``!D
M2#,$)2@```!T!>@2[/__2('$6($``$R)X%M=05Q!74%>05_#055!5%534HL]
MX(@``.BKZ___BSW5B```2(G#Z)WK__^*D[@```!(C3WL*```B)"Y````,<#H
M>^___TB%P'0+@#@Q=0:`>`$`=`<QP.A/]___BSV6B```Z&'K__^Y"P```+H!
M````2(TU.BH``$B)Q^C8Z?__2(7`=36+/6V(``#H..O__[D#````N@$```!(
MC346*@``2(G'Z*_I__](B<-(A<`/A<\```#I$0$``$B+0!"[`0```$B+:"@Y
M':N8``"+/26(``!$C6/_?AKHZNK__S')26/42(GN2(G'Z)KH__](A<!U%(L%
M?Y@``/_(1#G@=86[`0```.M73(LH08M%#"4`!"``/0`$``!U!DF+=1#K((L]
MTH<``.B=ZO__3(GNN0(````QTDB)Q^B[ZO__2(G&2(L5*9@``$ACPTB+/,+H
M!>C__X7`=9S_P^EI____.1T4F```BSV.AP``#XX5____Z%/J__](B>[_PTB)
MQ^@VY___Z]I(BT`0BSUJAP``2(LHZ#+J__](C14<*0``2(G'2(GNZ-#H__](
MBT,02(LH]D4.0'06BST]AP``Z`CJ__](B>Y(B<?HS>?__XL])X<``.CRZ?__
MN0,```"Z`0```$B--=<H``!(B<?H:>C__TB)PTB%P'1\2(T]]"8``#'`Z+OM
M__])B<1(A<!T#TB+0Q"+/>&&``!(BRCK1.@O\O__BSW1A@``2(7`2(G%2(M#
M$'0@3(L@Z(WI__](B>I(B<=,B>;H+^C__TB)[^@WZ?__ZR!(BQ46EP``2(LH
M3(LBZ&/I__],B>)(B>Y(B<?H!>C__XL]?X8``.A*Z?__N@$```!(C34V*```
M2(G'Z%;F__^+/6"&``!(B<7H*.G__TB-%2XH``!(B>Y(B<?HQN?__XL]0(8`
M`.@+Z?__2(T])28``,:`N0`````QP.CN[/__2(D%IY8``$B%P'4D2(L]BY8`
M`.A4]___2(D%CY8``$B%P'4,2(T]WB<``.@FYO__2(L]=Y8``+[``0``Z&7H
M__^)P8U``8/X`78D@_D1=!](BP5'E@``2(L54)8``$B-/<4G``!(BS`QP.CG
MY?__6%M=05Q!7<-52(G]4U'HM//__TB)Q^B6\___2(G#,<#HY/7__X7`="I(
MA>UT)8!]``!T'TB--6(F``!(B=_HM.;__T@YPW4+6DB)[UM=Z8SN__]86UW#
M05<QTD%6055!5%532('LF`(``&1(BP0E*````$B)A"2(`@``,<"+1!<X2(MT
M%S@/R(G`2(GQ2(E$5`A(B?!(P>D@2,'H*$C![C@E`/\``$@)\$B)SDC!X0A(
MP>88@>$``/\`B?9("?!("<A(B414$$B#P@A(@_I`=:Q(C50D"$B-M"0(`@``
M2(M":$@S0D!(@\((2#-""$@S0OA(C0P`2,'H'PG(2(E">$@Y\G793(L?2(M?
M$$B+;QA,B=A,B=I)B=E-B=A(P>H;2,'@!4@)T$B+5R`#1"0(2<'@'HV4$)EY
M@EI(B=A(,>A((T<(2#'H1(TD`DB+5PA(BT<(3(GF2,'H`DC!XAY("<)(BT0D
M$$C![AN)T4DQR4B-A`69>8):32'923'93`'(38GA2<'A!4D)\4D!P4R)V$C!
MZ`)%B<Y)"<!(BT0D&$PQPD6)PDPAXDB-M`.9>8):N/____](,<I(P>`%2`'6
M3(G*2<'I&TC!X@5!@^$?2"'"3`G*3(M,)"!(`?),B>9(P>8>2<'L`DJ-C`F9
M>8):08G53`GF23'P08GT32'P33'020'(2(G12,'J&TC!X06#XA]((<%("<I,
MB?%)P>X"3`'"2,'A'DR+1"0H3`GQ08G12#'.3XV$`IEY@EI)B=)(P>H;3"'N
M2<'B!8/B'T&)SDDAPDPQYDP)TDP!QDR-!!9,B>Y)P>T"2(M4)#!(P>8>18G"
M3`GN38VD%)EY@EI,B<)(,?%!B?5,(<E,,?%)`<Q(P>(%2<'H&T@APD&#X!])
M"=!,B<I)P>D"2XT,!$C!XAY,BT0D.$P)RD&)S$@QUD^-M`:9>8):28G(2,'I
M&TPAUDG!X`6#X1]!B=%,,>Y)(<!,"<%)`?9-B=!)P>H"28TT#DG!X!Y(BTPD
M0$T)T$&)]DPQPDV-K`V9>8):2(GQ2,'N&TPAXDC!X06#YA]%B<),,<I((<%(
M"<Y)`=5,B>%)P>P"28U4-0!(P>$>2(MT)$A,">%!B=5),<A-C8PQF7F"6DB)
MUDC!ZAM-(?!(P>8%@^(?08G,33'02"'&2`GR30'!3(GV2<'N`DV-!!%(P>8>
M2(M4)%!,"?9%B<%(,?%-C902F7F"6DR)PDG!Z!M,(>E(P>(%08/@'T&)]DPQ
MX4@APDD)T$D!RDR)ZDG![0)+C0P"2,'B'DR+1"183`GJ08G*2#'63XVD!)EY
M@EI)B<A(P>D;3"'.2<'@!8/A'T&)U4PQ]DDAP$D!]$P)P4V)R$G!Z0))C30,
M2<'@'DB+3"1@30G(08GT3#'"38VT#IEY@EI(B?%(P>X;3"'22,'A!8/F'T6)
MP4PQZD@AP4@)SDD!UDR)T4G!Z@))C10V2(MT)&A(P>$>08G63`G138VL-9EY
M@EI(B=9(P>H;23'(2,'F!8/B'TTAX$&)RD@AQDTQR$@)\DR)YDG![`)-`<5(
MP>8>38U$%0!(BU0D<$P)YD6)Q4@Q\4V-C!&9>8):3(G"08GT3"'Q3#'120')
M2,'B!4G!Z!M((<)!@^`?20G03(GR2<'N`DN-#`%(P>(>3(M$)'A,"?)!B<E(
M,=9/C90"F7F"6DF)R$C!Z1M,(>Y)P>`%@^$?08G63#'F22'`3`G!20'R38GH
M2<'M`DF--`I)P>`>2(N,)(````!-">A!B?),,<)-C:0,F7F"6DB)\4C![AM,
M(<I(P>$%@^8?18G%3#'R2"'!2`G.20'43(G)2<'I`DF-%#1(P>$>2(NT)(@`
M``!,"<E!B=1),<A-C;0VF7F"6DB)UDC!ZAM-(=!(P>8%@^(?08G)33'H2"'&
M2`GR30'&3(G62<'J`DV-!!9(P>8>2(N4))````!,"=9%B<=(,?%-C:P5F7F"
M6DR)PDG!Z!M,(>%(P>(%08/@'T&)\DPQR4@APDD)T$D!S4R)XDG![`)+C4P%
M`$C!XAY,BX0DF````$P)XD&)SD@QUD^-A`&9>8):28G)2,'I&TPA_DG!X06#
MX1]!B=5,,=9)(<%,`<9,"<E-B?E)P>\"2<'A'D&)R$B+C"2@````30GY20'P
M3#'*18G$28V,"IEY@EI,(?),B>9,,>I(P>X;2`'13(GB2,'B!4@)\DR)SD6)
MR4P#C"2P````B=)(`<I,B?%)P>X"2,'A'D&)TDP)\4@QSD$Q\$B+M"2H````
M38VL-:'KV6Y,B=9-`<5-B=!(P>8%2<'H&TP)QHGV38U$-0!,B>9)P>P"2,'F
M'D6)Q4P)YDF)S(G)23'T1#'B38VD$:'KV6Y,B>I-B>E)P>D;2,'B!4P)RHG2
M38T,%$R)TDC!XAY)P>H"18G,2`.,)+@```!,"=))B?*)]D@#M"3`````23'2
M13'038GB3HV$`:'KV6Y,B>%)P>H;2,'A!4P)T8G)3`'!38GH2<'M`DG!X!Y!
MB<I-">A)B=6)TD@#E"3(````33'%13'I38G53HV,#J'KV6Y,B=9)P>T;2,'F
M!4P)[HGV3`'.38GA2<'L`DG!X1Y!B?5-">%-B<1%B<!-,<Q$,>%-B>Q(C8P*
MH>O9;DR)ZDG![!M(P>(%3`GBB=)(`<I,B=%)P>H"2,'A'D&)U$P)T4V)RDP#
MA"30````18G)23'*3`.,)-@```!$,=9-C90PH>O9;DR)YDV)X$G!Z!M(P>8%
M3`G&B?9-C00R3(GN2<'M`DC!YAY%B<),">Y)B<V)R4@#C"3@````23'U1#'J
M38VL$:'KV6Y,B=)-B=%)P>D;2,'B!4P)RHG238U,%0!,B>))P>P"2,'B'D6)
MS4P)XDF)](GV2`.T).@```!),=1%,>!-B>Q.C80!H>O9;DR)Z4G![!M(P>$%
M3`GAB<E,`<%-B=!)P>H"2<'@'D&)S$T)T$F)THG233'"13'138GB3HV,#J'K
MV6Y,B>9(P>8%2<'J&T@#E"3P````3`G6B?9,`<Y-B>E)P>T"2<'A'D&)\DT)
MZ4V)Q46)P$P#A"3X````33'-1#'I38G52(V,"J'KV6Y,B=))P>T;2,'B!4P)
MZHG22`'*3(GA2<'L`DC!X1Y!B=5,">%-B<Q%B<E,`XPD``$``$DQS$0QYDV-
MI#"AZ]EN3(GN38GH2<'H&TC!Y@5,"<:)]DV-!#1,B=9)P>H"2,'F'D6)Q$P)
MUDF)RHG)23'R1#'238V4$:'KV6Y,B>)-B>%)P>D;2,'B!4P)RHG238T,$DR)
MZDC!XAY)P>T"18G*2`.,)`@!``!,">I)B?6)]D@#M"00`0``23'513'H38G5
M3HV$`:'KV6Y,B=%)P>T;2,'A!4P)Z8G)3`'!38G@2<'L`DG!X!Y!B<U-">!)
MB=2)TD@#E"08`0``33'$13'A38GL3HV,#J'KV6Y,B>Y)P>P;2,'F!4P)YHGV
M3`'.38G12<'J`DG!X1Y!B?1-"=%-B<)%B<!-,<I$,=%-B>)(C8P*H>O9;DR)
MXDG!ZAM(P>(%3`G2B=)(`<I,B>E)P>T"2,'A'D&)TDP)Z4V)S4P#A"0@`0``
M18G)23'-3`.,)"@!``!$,>Y-C:PPH>O9;DR)UDV)T$G!Z!M(P>8%3`G&B?9-
MC40U`$R)YDG![`)(P>8>18G%3`GF28G,38GNB<E),?1)P>X;2`.,)#`!``!$
M,>)-C8P1H>O9;DR)ZDC!X@5,"?*)TDV--!%,B=))P>H"2,'B'D6)\4P)TDF)
M\DV)S(GV23'22<'L&T@#M"0X`0``13'03HV$`:'KV6Y,B<E(P>$%3`GAB<E-
MC20(38GH2<'M`DB)T4G!X!Y%B>>)TDT)Z$V)^DPQP4G!X@5!,<Y,B?E*C;0V
MH>O9;DC!Z1M-B<Y)"<I)P>D"2`.4)$`!``!)P>8>20'R3(G&18G`30G.1(G1
M3#'V18GU03'T3HV,(J'KV6Y,B=))P>H;38G\2,'B!4&#XA])P>\"2"'"2<'D
M'DD)TDT)_+K<O!N/2XTT$4V)\D6)YTT)XDTA_D&)\4DARDT)\DR+M"1(`0``
M20'630'P28GV2,'N&TG!Y@6#YA]-`=!)(<9,"?9)B<Y(P>D"2<'F'DV-%#!)
M"<Y$B=9-"?1,B?%%B?!-(<Q,(?E)"<Q(BXPD4`$``$@!T4P!Z4V)U4G!ZAM)
MP>4%08/B'TD!S$DAQ4T)ZDN-#!1-B<I)P>D"2<'B'D&)S4T)RDV)\46)U$T)
MT4DA\4TAYDT)SDR+C"18`0``20'130'Y28G/2,'I&TG!YP6#X1]-`<Y)(<=,
M"?E)B?=(P>X"2<'G'DV-#`Y)"?=%B<Y-"?I,B?Y$B?E,(>9-(>I)"?)(B[0D
M8`$``$@!UDP!QDV)R$G!Z1M)P>`%08/A'TD!\DDAP$T)P4V)^$N--`I-B>E)
MP>T"2<'A'D&)\DT)Z46)S4T)R$TA\$TA[TT)QTR+A"1H`0``20'030'@28GT
M2,'N&TG!Y`6#YA]-`<=)(<1,">9-C00W38GW2<'N`DG!YQY%B<1-"?=-"?E,
MB?Y%B?Y,(>Y-(=%)"?%(B[0D<`$``$@!UD@!\4R)QDG!Z!M(P>8%08/@'TD!
MR4@AQDD)\$R)UDG!Z@)(P>8>2XT,`4P)UDV)^D&)R4&)\$D)\DTAXDTAQTT)
MUTR+E"1X`0``20'230'J28G-2,'I&TG!Y06#X1]-`==)(<5,">E-B>5)P>P"
M2<'E'DV-%`]-">5%B==,">Y,B>E%B>Q,(<%,(<Y("<Y(BXPD@`$``$@!T4P!
M\4V)UD@!\4G!Y@5)P>H;22'&08/B'TT)\DJ--!%,B<E)P>D"2,'A'D&)]DP)
MR4V)Z4&)RDD)R4TA^4TAU4T)S4R+C"2(`0``20'130'!28GP2,'N&TG!X`6#
MYA]-`<U)(<!,"<9-B?A)P>\"2<'@'DV-3#4`30GX18G-3`G!3(G&18G'3"'6
M3"'Q2`GQ2(NT))`!``!(`=9,`>9-B<Q)P>0%2`'.22'$2<'I&T&#X1]-">%*
MC0P.3(GV2<'N`DV)P4C!YAY!B<Q,"?9!B?9)"?%-(>E-(?!-"<A,BXPDF`$`
M`$D!T4T!RDF)R4C!Z1M)P>$%@^$?30'"22'!3`G)38T$"DV)ZDG![0))P>(>
M18G!30GJ3`G63(G118G53"'Q3"'F2`G.2(N,)*`!``!(`=%,`?E-B<=)P>@;
M2<'G!4&#X!](`?%)(<=-"?A*C30!3(GA2<'L`DV)T$C!X1Y!B?=,">%!B<Q)
M"<A-(<A-(>)-"<),BX0DJ`$``$D!T$T!QDF)\$C![AM)P>`%@^8?30'622'`
M3`G&38G(2<'I`DG!X!Y-C10V30G(18G63`G!38G!1(G&32'A3"'Y3`G)3(N,
M)+`!``!)`=%-`<U-B=%)P>H;2<'A!4&#XA])`<U)(<%-"<I-B<%+C4P5`$V)
M^DG!XAY)P>\"08G-30GZ18G730G132'Q32'X30G(3(N,)+@!``!)`=%-`>%)
MB<Q(P>D;2<'D!8/A'TT!R$DAQ$P)X4V-#`A,B?%)P>X"2,'A'D6)S$P)\4D)
MRDF)R$&)SDTA^$TAZDT)PDR+A"3``0``20'020'P3(G.2<'I&TC!Y@5!@^$?
M30'"2"'&20GQ2(G.3XT$"DV)Z4G![0))P>$>18G"30GI3`G.18G-3"'F3"'I
M2`GQ2(NT),@!``!(`=9,`?Y-B<=)P>@;2<'G!4&#X!](`?%)(<=-"?A*C30!
M38G@2<'L`DG!X!Y!B?=-">!-"<%,B<%%B<1,(>E-(=%)"<E(BXPDT`$``$@!
MT4P!\4F)]DC![AM)P>8%@^8?20')22'&3`GV38G&28T,,4V)T4G!Z@))P>$>
MB<Y-"<I%B=%-"=9-(?Y-(<A-"?!,B[0DV`$``$D!UD@#E"3@`0``30'U28G.
M2,'I&TP!XDG!Y@6#X1]-`<5)(<9,"?%!B<A-`>A-B?U)P>\"2<'E'D2)P4T)
M[TT)^DV)_46)_DTAS4DA\DT)ZDV-+!))B<I(B<I(P>H;2<'B!4D)TDB)\DC!
M[@)%B=)(P>(>30'J3(NL).@!``!("=:ZUL%BRDDQ]T6)U$D!U44Q^$T!Z4V)
MY4T!R$V)X4G!Y05)P>D;30G-28G)2,'I`DG!X1Y%B>U,"<E,BXPD\`$``$T!
MZ$F)]4DQS46)QXGV20'113'J30'.38GY30'638GZ2<'A!4G!ZAM-"<I-B>%)
MP>P"2<'A'D6)TDT)S$R+C"3X`0``30'R18G520'138GN3`'.28G)2<'F!8G)
M33'A13'(38GI2<'I&TP!QDR+A"0(`@``30GQ18G)20'Q3(G^2<'O`DC!YAY%
MB<Y)"?=(B[0D``(``$@!UD@!\4R)YDD!T$6)Y$PQ_DT!Q$V)^$6)_T$Q\DR)
M]DP!T4V)\DC!Y@5)P>H;20GR3(GN2<'M`DC!YAY%B=),">Y)`<I),?!$B=%%
M,<%-B?!)P>X"28G-2<'@'DT!S$F)R4G!Y05-"<9,BX0D$`(``$G!Z1M-">E)
M`=!%B<E-`<=)B?!-`>&)]DTQ\$6)S44QPDV)Z$T!UTV)ZDG!X`5)P>H;30G"
M28G(18G230'Z2<'@'DC!Z0)%B=1)"<A(BXPD&`(``$@!T4@!SDR)\46)]DPQ
MP4$QR4R)X4P!SDV)X4C!X05)P>D;20G)3(GI2<'M`DC!X1Y%B<E)"<U(BXPD
M(`(``$D!\4B+M"0H`@``18G/2`'12`'620'.3(G!18G`3#'I20'P3(GN18GM
M03'*3(GY30'638GZ2,'A!4G!ZAM)"<I,B>%)P>P"2,'A'D6)TDP)X4T!\D@Q
MSD6)UD$Q\4R)]DT!R$V)\4C!Y@5)P>D;20GQ3(G^2<'O`DC!YAY%B<E)"?=(
MB[0D,`(``$T!P4R+A"0X`@``18G,2`'620'020'U2(G.B<E,,?Y,`<%-B?A%
MB?]!,?),B>9-`=5-B>)(P>8%2<'J&TD)\DR)]DG![@)(P>8>18G23`GV30'J
M23'P18G513'!38GH3`')38GI2<'@!4G!Z1M-"<%-B>A%B<E)`<E,B>%)P>P"
M2,'A'D6)SDD)S$B+C"1``@``2`'120'/2(GQ2<'M`HGV3#'A2<'@'D$QRDR)
M\4T)Z$T!UTV)\DC!X05)P>H;20G*2(N,)$@"``!%B=)(`=%-`?I(`<Y,B>%%
MB==%B>1,,<%!,<E,B?E,`<Y-B?E(P>$%2<'I&TD)R4R)\4G![@)(P>$>18G)
M20G.2(N,)%`"``!)`?%(B[0D6`(``$6)S4@!T4D!S$R)P46)P$PQ\4$QRDR)
MZ4T!U$V)ZDC!X05)P>H;20G*3(GY18G230'B2,'A'D@!UDG![P))`?!,B?9%
MB=1,"?E%B?9(,<Y!,?%,B>9-`<A-B>%(P>8%2<'I&TD)\4R)[DG![0)(P>8>
M18G)20GU2(NT)&`"``!-`<%%B<](`=9)`?9(B<Z)R4PQ[D$Q\DR)_DT!UDV)
M^DC!Y@5)P>H;20GR3(GF2<'L`DC!YAY%B=),">9,BZ0D:`(``$T!\D6)T$F)
M]DD!U$P!X4V)[$6)[4DQ](GV13'A38G$3`')38G!2<'D!4G!Z1M-">%%B<E)
M`<E,B?E)P>\"2,'A'D6)S$D)STB+C"1P`@``33'^2`'113'R20'-3(GA30'5
M38GB2,'A!4G!ZAM)"<I,B<%)P>@"2,'A'D6)TDD)R$B+C"1X`@``30'J18G6
M2`'12`.4)(`"``!(`<Y,B?E%B?],,<%!,<E,B?%,`<Y-B?%(P>$%2<'I&TP)
MR4@!SDR)X4G![`)(P>$>20G,20'32(GQ30'?38G#2,'A!40#1R!-,>-((<A!
M`>Q,B4<@13':3(EG&$T!UTF)\@-W"$G!ZAM(B7<(08/B'TP)T$0!^$B)!TR)
M\$G![@)(P>`>20G&1`'S2(E?$$B+A"2(`@``9$@S!"4H````=`7H:]'__TB!
MQ)@"``!;74%<05U!7D%?PU"_@````.C^SO__NHFKS>^Y_MRZF+[PX=+#2,<`
M`2-%9TB)4`A(B4@02,=`&'94,A!(B7`@2,=`*`````!(QT`P`````,=`>```
M``!:PT%62&/"055!5%5(B<532(M/*$B)^XT4P4@YT78$2/]',$C!Z!U(`4,P
M2&-#>$R-8SA(B5,HA<!T3[I`````*<(YU0].U4P!X$ACRD@Y\',+2(T\"$@Y
M_G,"#PM,C2P.2#GP=@5,.>AR\$B)Q_.DBT-X`=")0WB#^$!U<TB)WRG5Z!;H
M__],B>Y,C6MX@_T_?C),.>9V!4PY[G*^3#GF3(UV0',%33GT<K"Y$````$R)
MYX/M0/.E2(G?Z-WG__],B?;KR4ACS4PYYG8)28T$#$@YQG*%3#GF<PU(C00.
M23G$#X)S____3(GG\Z2):WA;74%<05U!7L-!5D%5051,C68X54B)]5-(BT8H
M2(G[1(MN*$2+=C!(P>@#@^`_00_-00_.C7`!2)A(8];&1`4X@$P!XH/^.'XD
MN4`````QP$B)URGQ\ZI(B>_H3N?__[D.````,<!,B>?SJ^L.N3@````QP$B)
MURGQ\ZI$B75P2(GO1(EM=.@BY___2(M%`$B)[TC!Z!B(`TB+10!(P>@0B$,!
M2(M%`$C!Z`B(0P)(BT4`B$,#2(M%"$C!Z!B(0P1(BT4(2,'H$(A#!4B+10A(
MP>@(B$,&2(M%"(A#!TB+11!(P>@8B$,(2(M%$$C!Z!"(0PE(BT402,'H"(A#
M"DB+11"(0PM(BT482,'H&(A##$B+11A(P>@0B$,-2(M%&$C!Z`B(0PY(BT48
MB$,/2(M%($C!Z!B(0Q!(BT4@2,'H$(A#$4B+12!(P>@(B$,22(M%((A#$UM=
M05Q!74%>Z3O.__]52(GE4TB-'4]G``!22(L#2(/X_W0(_]!(@^L(Z^]86UW#
M4.@6T?__6,,`````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````4$%27U1-4$1)4@!414U01$E2`%1-4``N+@`E<R\E<P!$>6YA3&]A
M9&5R.CIB;V]T7T1Y;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`97AE
M`"]P<F]C+R5I+R5S`%!!4E]414U0`%!!4E]04D]'3D%-10`Z`"\`4$523#5,
M24(`4$523$Q)0@!015),-4]05`!015),24\`4$%27TE.251)04Q)6D5$`%!!
M4E]34$%73D5$`%!!4E]#3$5!3@!005)?1$5"54<`4$%27T-!0TA%`%!!4E]'
M3$]"04Q?1$5"54<`4$%27T=,3T)!3%]435!$25(`4$%27T=,3T)!3%]414U0
M`%!!4E]'3$]"04Q?0TQ%04X`4$%27T-,14%./0`Q`$Q$7TQ)0E)!4EE?4$%4
M2`!365-414T`+W1M<`!54T52`%5315).04U%`"4P,G@`<&%R+0`E<R5S)7,E
M<P`E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I
M;&5D("AE<G)N;STE:2D*`"5S.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I
M<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E
M<F%T:6]N*0H`<&%R;``N<&%R`"5S)7-C86-H92TE<R5S`"5S)7-T96UP+25U
M)7,`)7,E<W1E;7`M)74M)74E<P!!4D=6`!@`<&5R;``P`%!!4CHZ4&%C:V5R
M.CI615)324].`#$N,#0Y`%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T
M;W)Y`"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R
M>2`E<R!F86EL960@*&5R<FYO/25I*0H`+64`+2T`979A;"`G97AE8R`O=7-R
M+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U;FYI
M;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H)%!!
M4E]-04=)0RP@)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE*3L*14Y$
M('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`[
M"G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*
M;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M
M<"D["F]U=',H<7%[4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")]*3L*
M1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@
M=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD
M("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N
M97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@
M)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*<')I;G0@)'1M<"`B(R$O8FEN+W-H
M"G@],3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T
M96UP)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*
M<VQE97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R(@+B`D=&UP+3YF
M:6QE;F%M92`N("(G"B(["F-H;6]D(#`W,#`L)'1M<"T^9FEL96YA;64["FUY
M("1C;60@/2`D=&UP+3YF:6QE;F%M92`N("<@/B]D978O;G5L;"`R/B8Q("8G
M.PIC;&]S92`D=&UP.PIS>7-T96TH)&-M9"D["F]U=',H<7$H4W!A=VYE9"!B
M86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A;G5P.B`I"BX@)'1M
M<"T^9FEL96YA;64I.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"
M3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["B1005)?
M34%'24,@/2`B7&Y005(N<&U<;B(["F5V86P@>PI?<&%R7VEN:71?96YV*"D[
M"FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5#;VYF:6<@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"E]S971?<&%R7W1E;7`H*3L*;7D@*"1S=&%R=%]P;W,L("1D871A7W!O<RD[
M"GL*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["F]P96X@7T9(+"`G
M/"<L("1P<F]G;F%M92!O<B!L87-T.PIB:6YM;V1E*%]&2"D["FUY("1B=68[
M"FUY("1S:7IE(#T@+7,@)'!R;V=N86UE.PIM>2`D8VAU;FM?<VEZ92`](#8T
M("H@,3`R-#L*;7D@)&UA9VEC7W!O<SL*:68@*"1S:7IE(#P]("1C:'5N:U]S
M:7IE*2!["B1M86=I8U]P;W,@/2`P.PI](&5L<VEF("@H;7D@)&T@/2`D<VEZ
M92`E("1C:'5N:U]S:7IE*2`^(#`I('L*)&UA9VEC7W!O<R`]("1S:7IE("T@
M)&T["GT@96QS92!["B1M86=I8U]P;W,@/2`D<VEZ92`M("1C:'5N:U]S:7IE
M.PI]"G=H:6QE("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:RA?1D@L("1M86=I
M8U]P;W,L(#`I.PIR96%D*%]&2"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG
M=&@H)%!!4E]-04=)0RDI.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("10
M05)?34%'24,I*2`^/2`P*2!["B1M86=I8U]P;W,@*ST@)&D["FQA<W0["GT*
M)&UA9VEC7W!O<R`M/2`D8VAU;FM?<VEZ93L*?0IL87-T(&EF("1M86=I8U]P
M;W,@/"`P.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`["G)E860@7T9(
M+"`D8G5F+"`T.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0@+2!U;G!A8VLH
M(DXB+"`D8G5F*2P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY("5R97%U
M:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@
M("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F
M(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B
M=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO
M=71S*'%Q*%5N<&%C:VEN9R!F:6QE("(D9G5L;&YA;64B+BXN*2D["FUY("1C
M<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q
M(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH
M/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@
M7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X
M="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F
M:6QE;F%M92`](%]T96UP9FEL92@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*
M)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE
M.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA
M;64["GT*96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D
M969I;F5D("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?=&5M
M<&9I;&4H(B1B87-E;F%M921E>'0B+"`D8G5F+"`P-S4U*3L*;W5T<R@B4TA,
M24(Z("1F:6QE;F%M95QN(BD["GT*96QS92!["B1R97%U:7)E7VQI<W1[)&9U
M;&QN86UE?2`]"B1005(Z.DAE879Y.CI-;V1U;&5#86-H97LD9G5L;&YA;65]
M(#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN
M86UE+`I].PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H
M<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F
M("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@9&5L971E("1R97%U
M:7)E7VQI<W1[)&UO9'5L97T@;W(@<F5T=7)N.PHD24Y#>R1M;V1U;&5](#T@
M(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@
M86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&
M:6QE+3YN97=?=&UP9FEL92!O<B!D:64@)"$["F)I;FUO9&4H)&9H*3L*<')I
M;G0@)&9H("1I;F9O+3Y[8G5F?3L*<V5E:R@D9F@L(#`L(#`I.PIR971U<FX@
M)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?=&5M<&9I;&4H(B1I;F9O
M+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["F]P96X@;7D@)&9H+"`G/"<L
M("1F:6QE;F%M92!O<B!D:64@(F-A;B=T(')E860@)&9I;&5N86UE.B`D(2([
M"F)I;FUO9&4H)&9H*3L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I
M;F<@9F%I;&5D.B!C86YN;W0@9FEN9"`D;6]D=6QE(5QN(CL*?2P@0$E.0RD[
M"G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ
M2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*
M=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES
M="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE
M(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@
M>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N
M;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID
M96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"FQA<W0@=6YL97-S
M("1B=68@97$@(E!+7#`P,UPP,#0B.PHD<W1A<G1?<&]S(#T@*'1E;&P@7T9(
M*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A
M<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M
M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2
M155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0HD<75I970@/2`P('5N
M;&5S<R`D14Y6>U!!4E]$14)51WT["FEF("@A)'-T87)T7W!O<R!O<B`H)$%2
M1U9;,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H:69T*2D@>PIM>2`E9&ES
M=%]C;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU
M("`@=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*=B`@('9E<FEF>5]P87(*
M*3L*:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G+2UR975S92<I('L*<VAI
M9G0@0$%21U8["B1%3E9[4$%27T%04%]2155317T@/2!S:&EF="!`05)'5CL*
M?0IE;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I
M;F,["G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q
M<&EU<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@97$@)TDG*2!["G!U<V@@
M0&%D9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E<2`G32<I('L*979A;"`B
M=7-E("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A
M<F=S+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2!["B1O=70@/2`D,CL*?0IE
M;'-I9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G<VET92<["GT*96QS:68@
M*"0Q(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q
M("=Q)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E
M;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92`B6%A8.B!#86YN;W0@;W!E;B!L
M;V<Z("0A(CL*?0IE;'-I9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@
M)#(["GT*<VAI9G0H0$%21U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD
M,7TI('L*9&5L971E("1%3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E
M<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S
M<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*
M97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H
M)&]U="D@>PI["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ
M6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`
M05)'5BD["FUY("1Z:7`["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D
M9F@L("<\)RP@)'!A<B!O<B!D:64@(D-A;FYO="!F:6YD("<D<&%R)SH@)"$B
M.PIB:6YM;V1E*"1F:"D["F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@
M/2!!<F-H:79E.CI::7`M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD
M;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D
M:64@(E)E860@)R1P87(G(&5R<F]R.B`D(2(["GT*;7D@)65N=B`](&1O('L*
M:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N
M>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^(',O
M7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<["GT*
M?3L*;&]C86P@)"\@/2!<-#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(%!!
M4BP@)SPG+"`D<&%R(&]R(&1I92`B)"$Z("1P87(B.PIB:6YM;V1E*%!!4BD[
M"F1I92`B)'!A<B!I<R!N;W0@82!005(@9FEL92(@=6YL97-S(#Q005(^(&5Q
M(")02UPP,#-<,#`T(CL*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@
M24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/
M.CI&:6QE.CI/7U=23TY,62@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*,#<W
M-RP**2!O<B!D:64@)"$["F)I;FUO9&4H)&9H*3L*)"\@/2`H9&5F:6YE9"`D
M9&%T85]P;W,I(#\@7"1D871A7W!O<R`Z('5N9&5F.PIS965K(%]&2"P@,"P@
M,#L*;7D@)&QO861E<B`]('-C86QA<B`\7T9(/CL*:68@*"$D14Y6>U!!4E]6
M15)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`M/FYE=RT^87!P;'DH7"1L;V%D97(L("0P*0I]"F9O<F5A8V@@;7D@
M)&ME>2`H<V]R="!K97ES("5E;G8I('L*;7D@)'9A;"`]("1E;G9[)&ME>7T@
M;W(@;F5X=#L*)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>6R<B72\[
M"FUY("1M86=I8R`](")?7T5.5E]005)?(B`N('5C*"1K97DI("X@(E]?(CL*
M;7D@)'-E="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*)&QO861E
M<B`]?B!S>R1M86=I8R@@*RE]>PHD;6%G:6,@+B`D<V5T("X@*"<@)R!X("AL
M96YG=&@H)#$I("T@;&5N9W1H*"1S970I*2D*?65G.PI]"B1F:"T^<')I;G0H
M)&QO861E<BD["B0O(#T@=6YD968["FEF("@D8G5N9&QE*2!["G)E<75I<F4@
M4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII
M;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@
M>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH
M)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D
M0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["FUY("5F:6QE
M<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B
M:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O
M;F9I9WML:6)?97AT?3L*;7D@)7=R:71T96X["F9O<F5A8V@@*'-O<G0@:V5Y
M<R`E9FEL97,I('L*;7D@*"1N86UE+"`D9FEL92D["F9O<F5A8V@@;7D@)&1I
M<B`H0&EN8RD@>PII9B`H)&YA;64@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE
M>R1??2D@>PHD9FEL92`]("1?.PIL87-T.PI]"F5L<VEF("@O7BA<421D:7)<
M15PO*"XJ6UY#8UTI*5Q:+VDI('L**"1F:6QE+"`D;F%M92D@/2`H)#$L("0R
M*3L*;&%S=#L*?0IE;'-I9B`H;2%>+VQO861E<B];7B]=*R\H+BI;7D-C72E<
M6B$I('L*:68@*&UY("1R968@/2`D4$%2.CI(96%V>3HZ36]D=6QE0V%C:&5[
M)#%]*2!["B@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0IE
M;'-I9B`H+68@(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER
M+R0Q(BP@)#$I.PIL87-T.PI]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N
M86UE(&%N9"!N;W0@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D
M9FEL92D@86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O.PIO=71S*"!J
M;VEN("(B+`IQ<2A086-K:6YG("(I+"!R968@)&9I;&4@/R`D9FEL92T^>VYA
M;65](#H@)&9I;&4L"G%Q*"(N+BXI"BD["FUY("1C;VYT96YT.PII9B`H<F5F
M*"1F:6QE*2D@>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*
M;W!E;B!&24Q%+"`G/"<L("1F:6QE(&]R(&1I92`B0V%N)W0@;W!E;B`D9FEL
M93H@)"$B.PIB:6YM;V1E*$9)3$4I.PHD8V]N=&5N="`](#Q&24Q%/CL*8VQO
M<V4@1DE,13L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YN97<M/F%P<&QY*%PD
M8V]N=&5N="P@)&9I;&4I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N
M86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H
M0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D[
M"GT*;W5T<RAQ<2A7<FET=&5N(&%S("(D;F%M92(I*3L*)&9H+3YP<FEN="@B
M1DE,12(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@
M.2DI.PHD9F@M/G!R:6YT*'-P<FEN=&8H"B(E,#AX+R5S(BP@07)C:&EV93HZ
M6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M90HI*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DI.PHD9F@M/G!R:6YT
M*"1C;VYT96YT*3L*?0I]"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@
M:68@)'II<#L*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L
M(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M>2`D;71I;64@/2`H<W1A="@D
M;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P
M96XH;7D@)&9H+"`B/"(L("1O=70I.PIB:6YM;V1E*"1F:"D["B1C='@M/F%D
M9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-A8VAE7VYA;64@/2`D8W1X+3YH
M97AD:6=E<W0["GT*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M(&QE;F=T
M:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE("X](")#04-(12(["B1F:"T^
M<')I;G0H)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D9F@M
M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*)&9H+3YP<FEN="@D4$%27TU!
M1TE#*3L*)&9H+3YC;&]S93L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I[
M"FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M
M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E
M<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI:
M:7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/
M.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE
M("(D(3H@)$`B.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I
M.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B1Z:7`M/G)E861&<F]M
M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!
M6E]/2R@I(&]R(&1I92`B)"$Z("1`(CL*07)C:&EV93HZ6FEP.CIS971#:'5N
M:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*
M)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["B1Q=6EE="`]("$D
M14Y6>U!!4E]$14)51WT["F]U=',H<7$H7"1%3E9[4$%27U1%35!](#T@(B1%
M3E9[4$%27U1%35!](BDI.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@
M>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D
M;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI
M<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA
M;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#
M;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?
M;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL
M92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S
M=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I
M*2!["F]U=',H<7$H4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@
M86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B*2D["GT@96QS92!["F]U
M=',H<7$H17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M
M92(I*3L*)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I
M.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]
M"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID
M:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R
M(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R
M86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%
M3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%2
M1U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X
M="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG
M*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P
M871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R
M;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I
M<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC
M=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A
M<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E
M($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E
M8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@
M24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S
M<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I
M<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*
M<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T
M=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER
M92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A
M;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R
M97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@>R!R97%U
M:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@
M<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F
M:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D
M<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!4
M14U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2![
M"FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H(&%N9"`M
M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U
M:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I
M;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R
M.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R
M;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T52
M3D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0HD=7-E<FYA;64@
M/7X@<R]<5R]?+V<["FUY("1S=&UP9&ER(#T@(B1P871H)$-O;F9I9WM?9&5L
M:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP
M9&ER+"`P-S4U.PII9B`H(21%3E9[4$%27T-,14%.?2!A;F0@;7D@)&UT:6UE
M(#T@*'-T870H)'!R;V=N86UE*2E;.5TI('L*;W!E;B`H;7D@)&9H+"`B/"(N
M("1P<F]G;F%M92D["G-E96L@)&9H+"`M,3@L(#(["G-Y<W)E860@)&9H+"!M
M>2`D8G5F+"`V.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L
M("TU."P@,CL*<WES<F5A9"`D9F@L("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O
M+V<["B1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TB("X@)&)U
M9CL*?0IE;'-E('L*;7D@)&1I9V5S="`](&5V86P*>PIR97%U:7)E($1I9V5S
M=#HZ4TA!.PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E;BAM
M>2`D9F@L("(\(BP@)'!R;V=N86UE*3L*8FEN;6]D92@D9F@I.PHD8W1X+3YA
M9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C='@M/FAE>&1I9V5S=#L*?2`O
M+R`D;71I;64["B1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TD
M9&EG97-T(CL*?0IC;&]S92@D9F@I.PI]"F5L<V4@>PHD14Y6>U!!4E]#3$5!
M3GT@/2`Q.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L:6U]=&5M<"TD)"([
M"GT*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;6MD:7(@)'-T;7!D:7(L
M(#`W-34["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-
M4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+SL*?0IS=6(@7W1E;7!F
M:6QE('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D
M9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L
M;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["F]P96X@
M;7D@)&9H+"`G/B<L("1T96UP;F%M92!O<B!D:64@(F-A;B=T('=R:71E("1T
M96UP;F%M93H@)"$B.PIB:6YM;V1E("1F:#L*<')I;G0@)&9H("1C;VYT96YT
M<SL*8VQO<V4@)&9H.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN
M960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL
M:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E
M=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A
M;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@
M/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@
M86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*
M)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M
M92P@)$-O;F9I9WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/
M1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`^
M/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T
M=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921#;VYF:6=[7V5X
M97TB*2!["B1P<F]G;F%M92`N/2`D0V]N9FEG>U]E>&5].PIR971U<FX["GT*
M?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421#;VYF:6=[<&%T:%]S97!]
M7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%
M35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D
M0V]N9FEG>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF
M:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X97TB*2P@;&%S="D*:68@
M+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG>U]E>&5]
M(CL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M
M92(I+"!L87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M
M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\
M/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L("1#
M;VYF:6=[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M92`]("(N)$-O;F9I9WM?
M9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ
M9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'
M971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D
M*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D0V]N9FEG>U]D96QI;7U<
M12DO)'!W9"1#;VYF:6=[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]
M("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM0
M05)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD
M14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@
M5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@>PID96QE=&4@
M)$5.5GLG4$%27R<N)%]].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.
M($1%0E5'("\I('L*)$5.5GLG4$%27R<N)%]](#T@)$5.5GLG4$%27T=,3T)!
M3%\G+B1??2!I9B!E>&ES=',@)$5.5GLG4$%27T=,3T)!3%\G+B1??3L*?0IM
M>2`D<&%R7V-L96%N(#T@(E]?14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@
M("`@(CL*:68@*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#
M3$5!3GT["GT*96QS:68@*"%E>&ES=',@)$5.5GM005)?1TQ/0D%,7T-,14%.
M?2D@>PIM>2`D=F%L=64@/2!S=6)S='(H)'!A<E]C;&5A;BP@,3(@*R!L96YG
M=&@H(D-,14%.(BDI.PHD14Y6>U!!4E]#3$5!3GT@/2`D,2!I9B`D=F%L=64@
M/7X@+UY005)?0TQ%04X]*%Q3*RDO.PI]"GT*<W5B(&]U=',@>PIR971U<FX@
M:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B([
M"GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E
M<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#
M;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@
M<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI
M8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@
M0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I
M.PID:64@<7$H<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O
M9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&XI"G5N;&5S<R`M
M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@
M*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI]
M.PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@
M*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ
M7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@``````7U]%3E9?
M4$%27T-,14%.7U\@("`@("`@("`@("`@("```````````'!E<FQX<VDN8P``
M``$;`SO@````&P```+R"___\````3(;__R0!``!DAO__K`0``%N)__\\`0``
MY(G__X0!```<BO__G`$``(J*___$`0``1(S__P@"``"#C?__3`(```6.__]H
M`@``,([__X`"``!TCO__F`(``%^/___,`@``9(___^`"```6D?__+`,``"R1
M__]``P``SY'__U@#``!ND___=`,``+*3__^,`P``R9/__Z0#``!BE/__W`,`
M`"^:__\\!```;9W__W@$``"\G?__U`0``-.T__\@!0``(;7__S@%```8MO__
M=`4``!0``````````7I2``%X$`$;#`<(D`$``"0````<````N('__Y`#````
M#A!&#AA*#PMW"(``/QH[*C,D(@`````4````1````""%__\8````````````
M``!$````7````!>(__^)`````$(.$(\"0@X8C@-"#B"-!$(.*(P%1`XPA@9!
M#CB#!T$.0`)M#CA$#C!!#BA"#B!"#AA"#A!"#@@4````I````%B(__\X````
M`$0.(',."``D````O````'B(__]N`````$$.$(8"00X8@P-'#C`"8@X800X0
M00X(0````.0```"^B/__N@$```!"#A".`D(.&(T#0@X@C`1!#BB&!40.,(,&
M1PY``Y\!#C!!#BA!#B!"#AA"#A!"#@@```!`````*`$``#2*__\_`0```$(.
M$(X"0@X8C0-"#B",!$$.*(8%00XP@P9'#M`!`R<!#C!!#BA!#B!"#AA"#A!"
M#@@``!@```!L`0``+XO__X(`````1P[``0)Z#@@````4````B`$``)6+__\K
M`````$$.$%0."``4````H`$``*B+__]$`````$$.$&H."``P````N`$``-2+
M___K`````$(.$(P"00X8A@-!#B"#!$<.L$`"VPX@00X800X00@X(````$```
M`.P!``"+C/__!0````````!(``````(``'R,__^R`0```$(.$(\"0@X8C@-"
M#B"-!$P.*(P%00XPA@9$#CB#!T<.X"$#A@$..$0.,$$.*$(.($(.&$(.$$(.
M"```$````$P"``#BC?__%@`````````4````8`(``.2-__^C``````*+#A!4
M#@@8````>`(``&^.__^?`0```$0.0`.:`0X(````%````)0"``#RC___1```
M``!*#A!R#@@`%````*P"```>D/__%P````!!#A!5#@@`-````,0"```=D/__
MF0````!"#A"-`D0.&(P#2PX@A@0"<`H.&$D.$$0."$4+00X80@X00@X(``!<
M````_`(``'Z0___-!0```$(.$(\"3@X8C@-"#B"-!$(.*(P%00XPA@9$#CB#
M!T<.D(,"`T(%#IB#`DH.H(,"4`Z8@P)!#I"#`@)"#CA$#C!!#BA"#B!"#AA"
M#A!"#@@X````7`,``.N5__\^`P```$(.$(T"0@X8C`-!#B"&!$$.*(,%00XP
M`S`##BA!#B!!#AA"#A!"#@@````P````F`,``.V8__]/`````$$.$(8"1`X8
M@P-!#B!["@X81`X000X(10M!#AA!#A!!#@@`)````,P#``"P@?__M@$```!"
M#A",`D0.,`.J`0X010X(`````````$@```#T`P``X)C__Q<7````0@X0CP)$
M#AB.`T(.((T$0@XHC`5!#C"&!D$..(,'1P[0!0/Y%@XX00XP00XH0@X@0@X8
M0@X00@X(```4````0`0``*NO__].`````$$.$`),#@@X````6`0``.&O___W
M`````$(.$(X"10X8C0-"#B",!$$.*(8%1`XP@P8"X0XH00X@0@X80@X00@X(
M```X````E`0``)RP__]9`0```$(.$(X"0@X8C0-"#B",!$4.*(8%1`XP@P8#
M/@$.*$$.($(.&$(.$$(."```````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````__________\``````````/__________````````````````
M``````$``````````0`````````!`````````+("````````#P`````````W
M!`````````P``````````"`````````-`````````.=4````````]?[_;P``
M``"0`@````````4`````````H`P````````&`````````(@#````````"@``
M``````!5!`````````L`````````&``````````5````````````````````
M`P````````#HO0````````(`````````0`4````````4``````````<`````
M````%P````````#`$@````````<`````````^!`````````(`````````,@!
M````````"0`````````8`````````!@```````````````````#[__]O````
M``$```@`````^?__;P`````'````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````2+P`````````````````````````````)B``````
M```V(````````$8@````````5B````````!F(````````'8@````````AB``
M``````"6(````````*8@````````MB````````#&(````````-8@````````
MYB````````#V(`````````8A````````%B$````````F(0```````#8A````
M````1B$```````!6(0```````&8A````````=B$```````"&(0```````)8A
M````````IB$```````"V(0```````,8A````````UB$```````#F(0``````
M`/8A````````!B(````````6(@```````"8B````````-B(```````!&(@``
M`````%8B````````9B(```````!V(@```````(8B````````EB(```````"F
M(@```````+8B````````QB(```````#6(@```````.8B````````]B(`````
M```&(P```````!8C````````)B,````````V(P```````$8C````````5B,`
M``````!F(P```````'8C````````AB,```````"6(P``````````````````
M````````````````````````````````````````````````````````````
M````````````````P`````````````````````````````````````````!@
M````````\F`````````+8`````````1A````````$V``````````````````
M`/____\`````NV(```````!'0T,Z("A!;'!I;F4@."XS+C`I(#@N,RXP`$=#
M0SH@*$%L<&EN92`Y+C(N,"D@.2XR+C```"YS:'-T<G1A8@`N:6YT97)P`"YG
M;G4N:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E;&$N9'EN`"YR96QA+G!L=``N
M:6YI=``N<&QT+F=O=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F<F%M95]H
M9'(`+F5H7V9R86UE`"YC=&]R<P`N9'1O<G,`+F1A=&$N<F5L+G)O`"YD>6YA
M;6EC`"YD871A`"YB<W,`+F-O;6UE;G0`````````````````````````````
M``````````````````````````````````````````````````````````L`
M```!`````@````````!P`@```````'`"````````&0``````````````````
M``$````````````````````3````]O__;P(`````````D`(```````"0`@``
M`````/@``````````P`````````(````````````````````'0````L````"
M`````````(@#````````B`,````````8"0````````0````!````"```````
M```8`````````"4````#`````@````````"@#````````*`,````````500`
M``````````````````$````````````````````M````!`````(`````````
M^!````````#X$````````,@!`````````P`````````(`````````!@`````
M````-P````0```!"`````````,`2````````P!(```````!`!0````````,`
M```3````"``````````8`````````$$````!````!@``````````(```````
M```@````````#0````````````````````$````````````````````\````
M`0````8`````````$"`````````0(````````)`#```````````````````0
M`````````!``````````1P````$````&`````````*`C````````H",`````
M```8````````````````````"``````````(`````````%`````!````!@``
M``````"X(P```````+@C````````+S$```````````````````$`````````
M``````````!6`````0````8`````````YU0```````#G5`````````@`````
M```````````````!````````````````````7`````$````"``````````!@
M`````````&````````!2/0``````````````````"```````````````````
M`&0````!`````@````````!4G0```````%2=````````Y```````````````
M``````0```````````````````!R`````0````(`````````.)X````````X
MG@```````-`$```````````````````(````````````````````?`````$`
M```#`````````""\````````(*P````````0````````````````````"```
M`````````````````(,````!`````P`````````PO````````#"L````````
M$`````````````````````@```````````````````"*`````0````,`````
M````0+P```````!`K`````````@````````````````````(````````````
M````````EP````8````#`````````$B\````````2*P```````"@`0``````
M``0`````````"``````````0`````````$L````!`````P````````#HO0``
M`````.BM````````&`(```````````````````@`````````"`````````"@
M`````0````,``````````,``````````L````````&``````````````````
M```@````````````````````I@````@````#`````````&#`````````8+``
M``````"H$```````````````````(````````````````````*L````!````
M,````````````````````&"P````````-`````````````````````$`````
M`````0`````````!`````P````````````````````````"4L````````+0`
>```````````````````!````````````````````
