# 2 "atd/src/lexer.mll"
 
  open Import
  open Lexing
  open Parser

  let lexing_error lexbuf msg =
    let loc = (lexeme_start_p lexbuf, lexeme_end_p lexbuf) in
    Ast.error (Ast.string_of_loc loc ^ "\n" ^ msg)

  type accu = { mutable depth : int;
                buf : Buffer.t }

  let newline lexbuf =
    let pos = lexbuf.lex_curr_p in
    lexbuf.lex_curr_p <- { pos with
                             pos_lnum = pos.pos_lnum + 1;
                             pos_bol = pos.pos_cnum }

  let int_of_dec c =
    match c with
        '0'..'9' -> Char.code c - 48
      | _ -> invalid_arg "int_of_dec"

  let int_of_hex c =
    match c with
        '0'..'9' -> Char.code c - 48
      | 'a'..'f' -> Char.code c - 87
      | 'A'..'F' -> Char.code c - 55
      | _ -> invalid_arg "int_of_hex"

  let byte_of_hex a b =
    Char.chr (int_of_hex a lsl 4 + int_of_hex b)

  let byte_of_dec a b c =
    let x = int_of_dec a * 100 + int_of_dec b * 10 + int_of_dec c in
    if x > 255 then
      invalid_arg "byte_of_dec"
    else
      Char.chr x

  let utf8_of_hex4 buf b1 b2 b3 b4 =
    (* covers only U+0000-U+FFFF *)
    let a = int_of_hex b1 lsl 4 + int_of_hex b2 in
    let b = int_of_hex b3 lsl 4 + int_of_hex b4 in
    let u = a lsl 8 + b in
    let add buf i = Buffer.add_char buf (Char.chr (i land 0xff)) in
    if u <= 0x007f then
      add buf u
    else if u <= 0x07ff then (
      add buf (0b11000000 lor (a lsl 2) lor (b lsr 6));
      add buf (0b10000000 lor (b land 0b00111111))
    )
    else if u <= 0xffff then (
      add buf (0b11100000 lor (a lsr 4));
      add buf (0b10000000 lor ((a lsl 2) land 0b00111100) lor (b lsr 6));
      add buf (0b10000000 lor (b land 0b00111111))
    )
    else invalid_arg "utf8_of_hex4"
(*
  let test_utf8_of_hex s =
    assert (String.length s = 4);
    let buf = Buffer.create 10 in
    utf8_of_hex4 buf s.[0] s.[1] s.[2] s.[3];
    let file = Filename.temp_file "debug" "" in
    let oc = open_out file in
    output_string oc (Buffer.contents buf);
    close_out oc;
    assert (Sys.command ("xxd -b " ^ file) = 0);
    Sys.remove file
*)

  type string_kind = Double_quoted | Single_quoted

# 76 "atd/src/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\226\255\229\255\230\255\003\000\232\255\001\000\032\000\
    \116\000\209\000\037\001\121\001\205\001\033\002\239\255\240\255\
    \241\255\242\255\243\255\244\255\245\255\246\255\247\255\248\255\
    \249\255\250\255\251\255\252\255\253\255\254\255\001\000\227\255\
    \117\002\201\002\029\003\113\003\197\003\025\004\109\004\193\004\
    \021\005\105\005\189\005\017\006\130\006\242\255\243\255\246\255\
    \131\006\254\255\255\255\005\000\005\000\247\255\248\255\249\255\
    \250\255\000\000\106\006\253\255\204\006\252\255\141\006\251\255\
    \200\006\250\255\251\255\252\255\006\000\253\255\003\000\006\000\
    \255\255\254\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\255\255\024\000\255\255\029\000\027\000\
    \021\000\029\000\020\000\020\000\020\000\020\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\255\255\
    \020\000\020\000\017\000\018\000\020\000\020\000\020\000\020\000\
    \020\000\019\000\255\255\022\000\255\255\255\255\255\255\255\255\
    \011\000\255\255\255\255\010\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\004\000\255\255\004\000\004\000\
    \255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\000\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\046\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\255\255\255\255\000\000\000\000\000\000\
    \000\000\255\255\255\255\000\000\255\255\000\000\255\255\000\000\
    \066\000\000\000\000\000\000\000\255\255\000\000\255\255\255\255\
    \000\000\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\005\000\005\000\004\000\006\000\051\000\051\000\
    \067\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\000\000\002\000\004\000\000\000\051\000\000\000\007\000\
    \030\000\029\000\019\000\031\000\021\000\073\000\014\000\072\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\020\000\022\000\024\000\017\000\023\000\016\000\
    \000\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\028\000\000\000\027\000\000\000\009\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\011\000\010\000\010\000\010\000\010\000\010\000\012\000\
    \010\000\010\000\010\000\010\000\013\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\026\000\018\000\025\000\015\000\042\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\036\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\035\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\032\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\033\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\034\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\037\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\038\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\039\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\040\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \000\000\000\000\000\000\000\000\010\000\000\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\041\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\000\000\000\000\000\000\000\000\
    \010\000\000\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\043\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \000\000\000\000\000\000\000\000\043\000\000\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\000\000\000\000\000\000\000\000\
    \043\000\000\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\047\000\051\000\000\000\000\000\
    \052\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\050\000\059\000\000\000\000\000\
    \000\000\049\000\059\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\000\000\000\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\000\000\
    \000\000\000\000\000\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\000\000\067\000\000\000\000\000\068\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\048\000\059\000\
    \000\000\000\000\000\000\000\000\000\000\053\000\000\000\000\000\
    \000\000\000\000\069\000\000\000\000\000\000\000\000\000\000\000\
    \070\000\056\000\071\000\000\000\000\000\055\000\000\000\054\000\
    \000\000\000\000\000\000\058\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\045\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \065\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\006\000\004\000\000\000\051\000\052\000\
    \068\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\000\000\004\000\255\255\051\000\255\255\000\000\
    \000\000\000\000\000\000\030\000\000\000\070\000\000\000\071\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\007\000\
    \255\255\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\007\000\007\000\007\000\007\000\007\000\
    \007\000\007\000\007\000\008\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\008\000\255\255\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\008\000\
    \008\000\008\000\008\000\008\000\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \009\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\255\255\255\255\255\255\255\255\
    \009\000\255\255\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\010\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \255\255\255\255\255\255\255\255\010\000\255\255\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \010\000\010\000\010\000\010\000\010\000\010\000\010\000\010\000\
    \011\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\255\255\255\255\255\255\255\255\
    \011\000\255\255\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\012\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \255\255\255\255\255\255\255\255\012\000\255\255\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \013\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\255\255\255\255\255\255\255\255\
    \013\000\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\032\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \255\255\255\255\255\255\255\255\032\000\255\255\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \033\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\255\255\255\255\255\255\255\255\
    \033\000\255\255\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\034\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\255\255\255\255\255\255\034\000\255\255\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \035\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\255\255\255\255\255\255\255\255\
    \035\000\255\255\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\036\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \255\255\255\255\255\255\255\255\036\000\255\255\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \037\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\255\255\255\255\255\255\255\255\
    \037\000\255\255\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\038\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \255\255\255\255\255\255\255\255\038\000\255\255\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \039\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\255\255\255\255\255\255\255\255\
    \039\000\255\255\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\039\000\039\000\039\000\039\000\
    \039\000\039\000\039\000\039\000\040\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \255\255\255\255\255\255\255\255\040\000\255\255\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \040\000\040\000\040\000\040\000\040\000\040\000\040\000\040\000\
    \041\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\255\255\255\255\255\255\255\255\
    \041\000\255\255\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\041\000\041\000\041\000\041\000\
    \041\000\041\000\041\000\041\000\042\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \255\255\255\255\255\255\255\255\042\000\255\255\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \043\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\255\255\255\255\255\255\255\255\
    \043\000\255\255\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\043\000\043\000\043\000\043\000\
    \043\000\043\000\043\000\043\000\044\000\048\000\255\255\255\255\
    \048\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\044\000\048\000\255\255\255\255\
    \255\255\044\000\048\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\255\255\255\255\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\255\255\
    \255\255\255\255\255\255\058\000\058\000\058\000\058\000\058\000\
    \058\000\255\255\064\000\255\255\255\255\064\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\044\000\048\000\
    \255\255\255\255\255\255\255\255\255\255\048\000\255\255\255\255\
    \255\255\255\255\064\000\255\255\255\255\255\255\255\255\255\255\
    \064\000\048\000\064\000\255\255\255\255\048\000\255\255\048\000\
    \255\255\255\255\255\255\048\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\060\000\060\000\060\000\
    \060\000\060\000\060\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\060\000\060\000\060\000\
    \060\000\060\000\060\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\044\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "";
  Lexing.lex_backtrk_code =
   "";
  Lexing.lex_default_code =
   "";
  Lexing.lex_trans_code =
   "";
  Lexing.lex_check_code =
   "";
  Lexing.lex_code =
   "";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 90 "atd/src/lexer.mll"
             ( OP_PAREN )
# 634 "atd/src/lexer.ml"

  | 1 ->
# 91 "atd/src/lexer.mll"
             ( CL_PAREN )
# 639 "atd/src/lexer.ml"

  | 2 ->
# 92 "atd/src/lexer.mll"
             ( OP_BRACK )
# 644 "atd/src/lexer.ml"

  | 3 ->
# 93 "atd/src/lexer.mll"
             ( CL_BRACK )
# 649 "atd/src/lexer.ml"

  | 4 ->
# 94 "atd/src/lexer.mll"
             ( OP_CURL )
# 654 "atd/src/lexer.ml"

  | 5 ->
# 95 "atd/src/lexer.mll"
             ( CL_CURL )
# 659 "atd/src/lexer.ml"

  | 6 ->
# 96 "atd/src/lexer.mll"
             ( LT )
# 664 "atd/src/lexer.ml"

  | 7 ->
# 97 "atd/src/lexer.mll"
             ( GT )
# 669 "atd/src/lexer.ml"

  | 8 ->
# 98 "atd/src/lexer.mll"
             ( SEMICOLON )
# 674 "atd/src/lexer.ml"

  | 9 ->
# 99 "atd/src/lexer.mll"
             ( COMMA )
# 679 "atd/src/lexer.ml"

  | 10 ->
# 100 "atd/src/lexer.mll"
             ( COLON )
# 684 "atd/src/lexer.ml"

  | 11 ->
# 101 "atd/src/lexer.mll"
             ( STAR )
# 689 "atd/src/lexer.ml"

  | 12 ->
# 102 "atd/src/lexer.mll"
             ( BAR )
# 694 "atd/src/lexer.ml"

  | 13 ->
# 103 "atd/src/lexer.mll"
             ( EQ )
# 699 "atd/src/lexer.ml"

  | 14 ->
# 104 "atd/src/lexer.mll"
             ( QUESTION )
# 704 "atd/src/lexer.ml"

  | 15 ->
# 105 "atd/src/lexer.mll"
             ( TILDE )
# 709 "atd/src/lexer.ml"

  | 16 ->
# 106 "atd/src/lexer.mll"
             ( DOT )
# 714 "atd/src/lexer.ml"

  | 17 ->
# 107 "atd/src/lexer.mll"
             ( TYPE )
# 719 "atd/src/lexer.ml"

  | 18 ->
# 108 "atd/src/lexer.mll"
             ( OF )
# 724 "atd/src/lexer.ml"

  | 19 ->
# 109 "atd/src/lexer.mll"
                ( INHERIT )
# 729 "atd/src/lexer.ml"

  | 20 ->
let
# 110 "atd/src/lexer.mll"
              s
# 735 "atd/src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 110 "atd/src/lexer.mll"
                ( LIDENT s )
# 739 "atd/src/lexer.ml"

  | 21 ->
let
# 111 "atd/src/lexer.mll"
              s
# 745 "atd/src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 111 "atd/src/lexer.mll"
                ( UIDENT s )
# 749 "atd/src/lexer.ml"

  | 22 ->
let
# 112 "atd/src/lexer.mll"
                   s
# 755 "atd/src/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 112 "atd/src/lexer.mll"
                      ( TIDENT s )
# 759 "atd/src/lexer.ml"

  | 23 ->
# 113 "atd/src/lexer.mll"
             ( newline lexbuf; token lexbuf )
# 764 "atd/src/lexer.ml"

  | 24 ->
# 114 "atd/src/lexer.mll"
             ( token lexbuf )
# 769 "atd/src/lexer.ml"

  | 25 ->
# 115 "atd/src/lexer.mll"
             ( EOF )
# 774 "atd/src/lexer.ml"

  | 26 ->
# 116 "atd/src/lexer.mll"
             ( STRING (string Double_quoted (Buffer.create 200) lexbuf) )
# 779 "atd/src/lexer.ml"

  | 27 ->
# 117 "atd/src/lexer.mll"
             ( STRING (string Single_quoted (Buffer.create 200) lexbuf) )
# 784 "atd/src/lexer.ml"

  | 28 ->
# 118 "atd/src/lexer.mll"
             ( comment 1 lexbuf; token lexbuf )
# 789 "atd/src/lexer.ml"

  | 29 ->
let
# 119 "atd/src/lexer.mll"
         c
# 795 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 119 "atd/src/lexer.mll"
             ( lexing_error lexbuf
                 (sprintf "Illegal character %S" (String.make 1 c)) )
# 800 "atd/src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and string kind buf lexbuf =
   __ocaml_lex_string_rec kind buf lexbuf 44
and __ocaml_lex_string_rec kind buf lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 123 "atd/src/lexer.mll"
              ( match kind with
                | Double_quoted -> Buffer.contents buf
                | Single_quoted ->
                    Buffer.add_char buf '"';
                    string kind buf lexbuf )
# 816 "atd/src/lexer.ml"

  | 1 ->
# 128 "atd/src/lexer.mll"
              ( match kind with
                | Double_quoted ->
                    Buffer.add_char buf '\'';
                    string kind buf lexbuf
                | Single_quoted -> Buffer.contents buf )
# 825 "atd/src/lexer.ml"

  | 2 ->
let
# 133 "atd/src/lexer.mll"
                             c
# 831 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1) in
# 134 "atd/src/lexer.mll"
              ( Buffer.add_char buf c;
                string kind buf lexbuf )
# 836 "atd/src/lexer.ml"

  | 3 ->
let
# 136 "atd/src/lexer.mll"
                  a
# 842 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2)
and
# 136 "atd/src/lexer.mll"
                             b
# 847 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3) in
# 137 "atd/src/lexer.mll"
              ( Buffer.add_char buf (byte_of_hex a b);
                string kind buf lexbuf )
# 852 "atd/src/lexer.ml"

  | 4 ->
let
# 139 "atd/src/lexer.mll"
                   a
# 858 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 139 "atd/src/lexer.mll"
                                b
# 863 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2)
and
# 139 "atd/src/lexer.mll"
                                             c
# 868 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 3) in
# 140 "atd/src/lexer.mll"
              ( Buffer.add_char buf (byte_of_dec a b c);
                string kind buf lexbuf )
# 873 "atd/src/lexer.ml"

  | 5 ->
# 142 "atd/src/lexer.mll"
              ( Buffer.add_char buf '\n'; string kind buf lexbuf )
# 878 "atd/src/lexer.ml"

  | 6 ->
# 143 "atd/src/lexer.mll"
              ( Buffer.add_char buf '\r'; string kind buf lexbuf )
# 883 "atd/src/lexer.ml"

  | 7 ->
# 144 "atd/src/lexer.mll"
              ( Buffer.add_char buf '\t'; string kind buf lexbuf )
# 888 "atd/src/lexer.ml"

  | 8 ->
# 145 "atd/src/lexer.mll"
              ( Buffer.add_char buf '\b'; string kind buf lexbuf )
# 893 "atd/src/lexer.ml"

  | 9 ->
# 146 "atd/src/lexer.mll"
              ( newline lexbuf;
                Buffer.add_char buf '\n';
                string kind buf lexbuf )
# 900 "atd/src/lexer.ml"

  | 10 ->
# 150 "atd/src/lexer.mll"
              ( newline lexbuf; string kind buf lexbuf )
# 905 "atd/src/lexer.ml"

  | 11 ->
# 151 "atd/src/lexer.mll"
              ( lexing_error lexbuf "Invalid escape sequence" )
# 910 "atd/src/lexer.ml"

  | 12 ->
let
# 152 "atd/src/lexer.mll"
         c
# 916 "atd/src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 152 "atd/src/lexer.mll"
              ( Buffer.add_char buf c; string kind buf lexbuf )
# 920 "atd/src/lexer.ml"

  | 13 ->
# 153 "atd/src/lexer.mll"
              ( lexing_error lexbuf "Unterminated string" )
# 925 "atd/src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec kind buf lexbuf __ocaml_lex_state

and comment depth lexbuf =
   __ocaml_lex_comment_rec depth lexbuf 64
and __ocaml_lex_comment_rec depth lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 156 "atd/src/lexer.mll"
              ( if depth > 1 then
                  comment (depth - 1) lexbuf
              )
# 939 "atd/src/lexer.ml"

  | 1 ->
# 159 "atd/src/lexer.mll"
              ( comment (depth + 1) lexbuf )
# 944 "atd/src/lexer.ml"

  | 2 ->
# 160 "atd/src/lexer.mll"
              ( (* we don't parse single-quoted strings similarly because
                   single-quotes are used all the time as apostrophes. *)
                ignore (string Double_quoted (Buffer.create 200) lexbuf);
                comment depth lexbuf )
# 952 "atd/src/lexer.ml"

  | 3 ->
# 164 "atd/src/lexer.mll"
              ( newline lexbuf; comment depth lexbuf )
# 957 "atd/src/lexer.ml"

  | 4 ->
# 165 "atd/src/lexer.mll"
              ( comment depth lexbuf )
# 962 "atd/src/lexer.ml"

  | 5 ->
# 166 "atd/src/lexer.mll"
              ( lexing_error lexbuf "Unterminated comment" )
# 967 "atd/src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec depth lexbuf __ocaml_lex_state

;;

# 168 "atd/src/lexer.mll"
 
  let init_fname lexbuf fname lnum =
    lexbuf.lex_start_p <- { lexbuf.lex_start_p
                            with
                              pos_fname = fname;
                              pos_lnum = lnum };
    lexbuf.lex_curr_p <- { lexbuf.lex_curr_p
                           with
                             pos_fname = fname;
                             pos_lnum = lnum }

# 986 "atd/src/lexer.ml"
