import sys
from enum import Enum
from gi.repository import GLib, GioUnix
from .key import Key


def gio_lines(stream):
    line = bytearray()

    while not stream.is_closed():
        byte = stream.read_bytes(1).get_data()
        if not byte:
            return
        line += byte
        if line[-1] == ord("\n"):
            yield line.decode()
            line.clear()

def gio_lines_async(stream, callback):
    line = bytearray()

    def _callback(source, result):
        nonlocal line
        byte = source.read_bytes_finish(result).get_data() if result else b""
        line += byte
        if line and line[-1] == ord("\n"):
            more = callback(line.decode())
            if not more:
                return
            line.clear()

        stream.read_bytes_async(1, GLib.PRIORITY_DEFAULT, None, _callback)

    _callback(stream, None)


class PromptMode(Enum):
    disclose = "disclose"
    delete = "delete"
    persist = "persist"

class RememberKind(Enum):
    session = "session"
    timeout = "timeout"
    skip = "skip"
    refuse = "refuse"

class Remember:
    def __init__(self, kind, value=None):
        self.kind = RememberKind(kind)
        self.value = None

        if self.kind == RememberKind("timeout"):
            self.value = value

class Session:
    def __init__(self):
        self.keys = []
        self.queries = []
        self.remembers = []
        self.unlock = False
        self.status = 127
        self.mode = None
        self.stream = GioUnix.InputStream.new(sys.stdin.fileno(), False)
        self.parse()

    def parse(self):
        for line in gio_lines(self.stream):
            parts = line.strip().partition(" ")
            cmd = parts[0]
            args = parts[2]
            if not hasattr(self, "_" + cmd):
                sys.stderr.write(f"Unknown command '{cmd}'\n")
                exit(127)
            more = getattr(self, "_" + cmd)(args)
            if more is False:
                break

    def _version(self, args):
        sys.stdout.write("version 0.0.1\n")
        sys.stdout.flush()

    def _key(self, args):
        self.keys.append(Key(args))

    def _query(self, args):
        self.queries.append(Key(args))

    def _unlock(self, args):
        self.unlock = True
        if len(self.keys) == 0:
            return False

    def _prompt(self, args):
        self.mode = PromptMode(args)
        return False

    def _remember(self, args):
        parts = args.strip().partition(" ")
        if (parts[0] == "session"):
            self.remembers.append(Remember("session"))
        elif (parts[0] == "skip"):
            self.remembers.append(Remember("skip"))
        elif (parts[0] == "refuse"):
            self.remembers.append(Remember("refuse"))
        elif (parts[0] == "timeout"):
            self.remembers.append(Remember("timeout", int(parts[2])))
        else:
            sys.stderr.write(f"Not supported remember '{parts[0]}'\n")
            exit(127)

    def password(self, pw, callback):
        """
        Sends a password to the daemon in response to an unlock command and
        returns immediately. When the daemon answers regarding the password's
        validity, calls the callback function with one argument: True if it was
        correct and False otherwise.
        """
        sys.stdout.write(f"password {pw}\n")
        sys.stdout.flush()

        def _callback(line):
            if line == "password incorrect\n":
                callback(False)
            elif line == "password correct\n":
                self.unlock = False
                self.parse()
                callback(True)
            else:
                sys.stderr.write(f"Unexpected command '{line}'\n")
                exit(127)
            return False

        gio_lines_async(self.stream, _callback)

    def remember(self, remember):
        """
        Sends the choosen remember to the daemon
        """
        if remember.kind == RememberKind("timeout"):
            sys.stdout.write(f"remember timeout {remember.value}\n")
        else:
            sys.stdout.write(f"remember " + remember.kind.value + "\n")
        sys.stdout.flush()
